% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DLD.R
\name{DLD}
\alias{DLD}
\title{The Discrete Lindley family}
\usage{
DLD(mu.link = "log")
}
\arguments{
\item{mu.link}{defines the mu.link, with "log" link as the default for the mu parameter.}
}
\value{
Returns a \code{gamlss.family} object which can be used
to fit a Discrete Lindley distribution
in the \code{gamlss()} function.
}
\description{
The function \code{DLD()} defines the Discrete Lindley distribution, one-parameter
discrete distribution, for a \code{gamlss.family} object to be used in GAMLSS fitting
using the function \code{gamlss()}.
}
\details{
The Discrete Lindley distribution with parameters \eqn{\mu > 0} has a support
0, 1, 2, ... and density given by

\eqn{f(x | \mu) = \frac{e^{-\mu x}}{1 + \mu}  (\mu(1 - 2e^{-\mu}) + (1- e^{-\mu})(1+\mu x))}

The parameter \eqn{\mu} can be interpreted as a strict upper bound on the failure rate function

The conventional discrete distributions (such as geometric, Poisson, etc.) are not
suitable for various scenarios like reliability, failure times, and counts. Consequently,
alternative discrete distributions have been created by adapting well-known continuous
models for reliability and failure times. Among these, the discrete Weibull distribution
stands out as the most widely used. But models like these require two parameters and not many
of the known discrete distributions can provide accurate models for both times and counts,
which the Discrete Lindley distribution does.

Note: in this implementation we changed the original parameters \eqn{\theta} for \eqn{\mu},
we did it to implement this distribution within gamlss framework.
}
\examples{
# Example 1
# Generating some random values with
# known mu
y <- rDLD(n=100, mu=0.3)

# Fitting the model
library(gamlss)
mod1 <- gamlss(y~1, family=DLD,
               control=gamlss.control(n.cyc=500, trace=FALSE))

# Extracting the fitted values for mu
# using the inverse link function
exp(coef(mod1, what='mu'))

# Example 2
# Generating random values under some model

# A function to simulate a data set with Y ~ DLD
gendat <- function(n) {
  x1 <- runif(n)
  mu    <- exp(2 - 4 * x1)
  y <- rDLD(n=n, mu=mu)
  data.frame(y=y, x1=x1)
}

set.seed(1235)
datos <- gendat(n=150)

mod2 <- NULL
mod2 <- gamlss(y~x1, sigma.fo=~x2, family=DLD, data=datos,
                 control=gamlss.control(n.cyc=500, trace=FALSE))

summary(mod2)
}
\references{
\insertRef{bakouch2014new}{DiscreteDists}
}
\seealso{
\link{dDLD}.
}
\author{
Yojan Andrés Alcaraz Pérez, \email{yalcaraz@unal.edu.co}
}
