#' Wrapper function for starting an MCMC simulation
#' 
#' This function provides a wrapper for starting an MCMC simulation, using only
#' the data and some basic options as input.
#' 
#' 
#' @param input Input data. Either a filename pointing to an R data file
#' containing the results of \code{\link{simulateNetwork}}, or a NumTimePoints
#' by NumNodes matrix.
#' @param output.file Where to save the output of the MCMC simulation.
#' @param information.sharing Which information sharing prior to use:
#' \code{'poisson'} for the Poisson prior (no information sharing),
#' \code{'exp_hard'} or \code{'exp_soft'} for the exponential prior with hard
#' or soft coupling among nodes, respectively, and \code{'bino_hard'} or
#' \code{'bino_soft'} for the binomial prior with hard or soft coupling among
#' nodes.
#' @param num.iter Number of iterations of the MCMC simulation.
#' @param prior.params Initial values of the hyperparameters of the information
#' sharing priors.
#' @param options Settings for the MCMC simulation, as generated by
#' \code{\link{defaultOptions}}.
#' @param fixed.edges Matrix of size NumNodes by NumNodes, with 
#' \code{fixed.edges[i,j]==1|0} if the edge between nodes i and j is fixed, and 
#' -1 otherwise. Defaults to \code{NULL} (no edges fixed).
#' @return Returns the results of the MCMC simulation, similar to
#' \code{\link{runDBN}}.
#' @author Sophie Lebre Frank Dondelinger
#' @seealso \code{\link{runDBN}}
#' @references For details on the model and MCMC simulation, see:
#' 
#' Dondelinger et al. (2012), "Non-homogeneous dynamic Bayesian networks with
#' Bayesian regularization for inferring gene regulatory networks with
#' gradually time-varying structure", Machine Learning.
#' @examples
#' 
#' # Generate random gene network and simulate data from it
#' dataset = simulateNetwork(l=25)
#' 
#' # Run MCMC simulation to infer networks and changepoint locations
#' # Uses default settings: Poisson prior and 1500 iterations
#' result.poisson = EDISON.run(dataset$sim_data, num.iter=500)
#' 
#' # Use the binomial information sharing prior with hard node coupling, and
#' # run for 5000 iterations
#' 
#' # NOT EXECUTED
#' #result.bino = EDISON.run(dataset$sim_data, 
#' #                information.sharing='bino_hard', num.iter=5000)
#'                         
#' # Set options to allow saving network and changepoint samples to file
#' options = defaultOptions()
#' options$save.file = TRUE
#' 
#' # NOT EXECUTED
#' # result.bino2 = EDISON.run(dataset$sim_data, 
#' #                  information.sharing='bino_hard',
#' #                  num.iter=5000, output.file='bino2.results',
#' #                  options=options)
#' 
#' @export EDISON.run
EDISON.run <-
function(input, output.file="EDISON.output", 
                       information.sharing='poisson', num.iter=10000, 
                       prior.params=NULL, options=NULL, fixed.edges=NULL) {

  if(!is.matrix(input)) {
    test = NULL  
    load(file=input)
    data=test$sim_data;
  } else {
    data = input
  }
  
  # Time series length
  n = dim(data)[2]

  # Number of variables
  q = dim(data)[1]

  if(is.null(options)) options = defaultOptions()
  
  if(is.null(fixed.edges)) fixed.edges = matrix(-1, q, q)
  
  ##### Important remark ####
  # if you know the changepoint position, and you want to run the procedure only for estimating the model within phases, 
  # you can  set the variable 'CPinit' to the known CP position  set the variable 'cD=0' in the file 'hyperparms.R' 
  # then CP move will never be considered (only the 4th move 'phase.update' will be considered).
  # however there is still some probleme with the 'output' fonctions in this case, I can help updating this when I will be back to work. 

  # Run TVDBN procedure:
  results = runDBN(targetdata=data, n=n, q=q, niter=num.iter, 
                   method=information.sharing, prior.params=prior.params, 
                   options=options, outputFile=output.file, 
                   fixed.edges=fixed.edges)

  return(results)
}

