% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/effect_modification.R
\name{evalues.IC}
\alias{evalues.IC}
\title{Compute an E-value for unmeasured confounding for an additive interaction contrast}
\usage{
evalues.IC(
  stat,
  true = 0,
  unidirBias = FALSE,
  unidirBiasDirection = NA,
  p1_1,
  p1_0,
  n1_1,
  n1_0,
  f1,
  p0_1,
  p0_0,
  n0_1,
  n0_0,
  f0,
  alpha = 0.05
)
}
\arguments{
\item{stat}{The statistic for which to compute the E-value ("est" for the interaction
contrast point estimate or "CI" for its lower confidence interval limit)}

\item{true}{The true (unconfounded) value to which to shift the specified statistic (point estimate or confidence interval limit). Should be smaller than the confounded statistic.}

\item{unidirBias}{Whether the direction of confounding bias is assumed to be the same
in both strata of Z (TRUE or FALSE); see Details}

\item{unidirBiasDirection}{If bias is assumed to be unidirectional, its assumed direction ("positive", "negative", or "unknown"; see Details). If bias is not assumed to be unidirectional, this argument should be NA.}

\item{p1_1}{The probability of the outcome in stratum Z=1 with treatment X=1}

\item{p1_0}{The probability of the outcome in stratum Z=1 with treatment X=0}

\item{n1_1}{The sample size in stratum Z=1 with treatment X=1}

\item{n1_0}{The sample size in stratum Z=1 with treatment X=0}

\item{f1}{The probability in stratum Z=1 of having treatment X=1}

\item{p0_1}{The probability of the outcome in stratum Z=0 with treatment X=1}

\item{p0_0}{The probability of the outcome in stratum Z=0 with treatment X=0}

\item{n0_1}{The sample size in stratum Z=0 with treatment X=1}

\item{n0_0}{The sample size in stratum Z=0 with treatment X=0}

\item{f0}{The probability in stratum Z=0 of treatment X=1}

\item{alpha}{The alpha-level to be used for p-values and confidence intervals}
}
\value{
Returns a list containing two dataframes (\code{evalues} and \code{RDt}). The E-value itself can be accessed as \code{evalues$evalue}.

The dataframe \code{evalues} contains the E-value, the corresponding bias factor, the bound on the interaction contrast if confounding were to attain that bias factor (this bound will be close to \code{true}, by construction), and the direction of bias when the bias factor is attained. If you specify that the bias is potentially multidirectional, is unidirectional and positive, or is unidirectional and negative, the returned direction of bias will simply be what you requested. If you specify unidirectional bias of unknown direction, the bias direction will be either positive or negative depending on which direction produces the maximum bias.

The dataframe \code{RDt} contains, for each stratum and for the interaction contrast, bias-corrected estimates (risk differences for the strata and the interaction contrast for \code{stratum = effectMod}), their standard errors, their confidence intervals, and their p-values. These estimates are bias-corrected for the worst-case bias that could arise for confounder(s) whose strength of association are no more severe than the requested E-value for either the estimate or the confidence interval (i.e., the bias factor indicated by \code{evalues$biasFactor}). The bias-corrected risk differences for the two strata (\code{stratum = "1"} and \code{stratum = "0"}) are corrected in the direction(s) indicated by \code{evalues$biasDir}.

If you specify unidirectional bias of unknown direction, the E-value is calculated by taking the minimum of the E-value under positive unidirectional bias and the E-value under negative unidirectional bias. With this specification, a third dataframe (\code{candidates}) will be returned. This is similar to \code{evalues}, but contains the results for positive unidirectional bias and negative unidirectional bias (the two "candidate" E-values that were considered).
}
\description{
Computes the E-value for an additive interaction contrast, representing the difference between stratum Z=1 and stratum Z=0
in the causal risk differences for a binary treatment X.
}
\details{
\subsection{E-values for additive effect modification}{

The interaction contrast is a measure of additive effect modification that represents the difference between stratum Z=1 versus stratum Z=0 of the causal risk differences relating a treatment X to an outcome Y. The estimated interaction contrast is given by:

\code{(p1_1 - p1_0) - (p0_1 - p0_0)}

To use this function, the strata (Z) should be coded such that the confounded interaction contrast is positive rather than negative.

If, in one or both strata of Z, there are unmeasured confounders of the treatment-outcome association, then the interaction contrast may be biased as well (Mathur et al., 2021). The E-value for the interaction contrast represents the minimum strength of association, on the risk ratio scale, that unmeasured confounder(s) would need to have with both the treatment (X) and the outcome (Y) in both strata of Z to fully explain away the observed interaction contrast, conditional on the measured covariates. This bound is attained when the strata have confounding bias in opposite directions ("potentially multidirectional bias"). Alternatively, if one assumes that the direction of confounding is the same in each stratum of Z ("unidirectional bias"), then the E-value for the interaction contrast is defined as the minimum strength of association, on the risk ratio scale, that unmeasured confounder(s) would need to have with both the treatment (X) and the outcome (Y) in \emph{at least one} stratum of Z to fully explain away the observed interaction contrast, conditional on the measured covariates. This bound under unidirectional confounding arises when one stratum is unbiased. See Mathur et al. (2021) for details.

As for the standard E-value for main effects (Ding & VanderWeele, 2016; VanderWeele & Ding, 2017), the E-value for the interaction contrast can be computed for both the point estimate and the lower confidence interval limit, and it can be also be calculated for shifting the estimate or confidence interval to a non-null value via the argument \code{true}.
}

\subsection{Specifying the bias direction}{

The argument \code{unidirBias} indicates whether you are assuming unidirectional bias (\code{unidirBias = TRUE}) or not (\code{unidirBias = FALSE}). The latter is the default because it is more conservative and requires the fewest assumptions. When setting \code{unidirBias = FALSE}, there is no need to specify the direction of bias via \code{unidirBiasDir}. However, when setting \code{unidirBias = TRUE}, the direction of bias does need to be specified via \code{unidirBiasDir}, whose options are:
\itemize{
\item \code{unidirBiasDir = "positive"}: Assumes that the risk differences in both strata of Z are positively biased.
\item \code{unidirBiasDir = "negative"}: Assumes that the risk differences in both strata of Z are negatively biased.
\item \code{unidirBiasDir = "unknown"}: Assumes that the risk differences in both strata of Z are biased in the same direction, but that the direction could be either positive or negative.
}
}

\subsection{Adjusted interaction contrasts}{

If your estimated interaction contrast has been adjusted for covariates, then you can use covariate-adjusted probabilities for \code{p1_1}, \code{p1_0}, \code{p0_1}, and \code{p0_0}. For example, these could be fitted probabilities from a covariate-adjusted regression model.
}

\subsection{Multiplicative effect modification}{

For multiplicative measures of effect modification (e.g., the ratio of risk ratios between the two strata of Z), you can simply use the function \code{evalue}. To allow the bias to be potentially multidirectional, you would pass the square-root of your multiplicative effect modification estimate on the risk ratio scale to \code{evalue} rather than the estimate itself. To assume unidirectional bias, regardless of direction, you would pass the multiplicative effect modification estimate itself to \code{evalue}. See Mathur et al. (2021) for details.
}
}
\examples{
### Letenneur et al. (2000) example data
# this is the example given in Mathur et al. (2021)
# Z: sex (w = women, m = male; males are the reference category)
# Y: dementia (1 = developed dementia, 0 = did not develop dementia )
# X: low education (1 = up to 7 years, 0 = at least 12 years)
# n: sample size

# data for women
nw_1 = 2988
nw_0 = 364
dw = data.frame(  Y = c(1, 1, 0, 0),
                  X = c(1, 0, 1, 0),
                  n = c( 158, 6, nw_1-158, nw_0-6 ) )

# data for men
nm_1 = 1790
nm_0 = 605
dm = data.frame(  Y = c(1, 1, 0, 0),
                  X = c(1, 0, 1, 0),
                  n = c( 64, 17, nm_1-64, nm_0-17 ) )

# P(Y = 1 | X = 1) and P(Y = 1 | X = 0) for women and for men
( pw_1 = dw$n[ dw$X == 1 & dw$Y == 1 ] / sum(dw$n[ dw$X == 1 ]) )
( pw_0 = dw$n[ dw$X == 0 & dw$Y == 1 ] / sum(dw$n[ dw$X == 0 ]) )
( pm_1 = dm$n[ dm$X == 1 & dm$Y == 1 ] / sum(dm$n[ dm$X == 1 ]) )
( pm_0 = dm$n[ dm$X == 0 & dm$Y == 1 ] / sum(dm$n[ dm$X == 0 ]) )

# prevalence of low education among women and among men
fw = nw_1 / (nw_1 + nw_0)
fm = nm_1 / (nm_1 + nm_0)

# confounded interaction contrast estimate
( pw_1 - pw_0 ) - ( pm_1 - pm_0 )

### E-values without making assumptions on direction of confounding bias
# for interaction contrast point estimate
evalues.IC( stat = "est",
       
            p1_1 = pw_1,
            p1_0 = pw_0,
            n1_1 = nw_1,
            n1_0 = nw_0,
            f1 = fw,
            
            p0_1 = pm_1,
            p0_0 = pm_0,
            n0_1 = nm_1,
            n0_0 = nm_0,
            f0 = fm )

# and for its lower CI limit
evalues.IC( stat = "CI",
            
            p1_1 = pw_1,
            p1_0 = pw_0,
            n1_1 = nw_1,
            n1_0 = nw_0,
            f1 = fw,
            
            p0_1 = pm_1,
            p0_0 = pm_0,
            n0_1 = nm_1,
            n0_0 = nm_0,
            f0 = fm )

### E-values assuming unidirectonal confounding of unknown direction
# for interaction contrast point estimate
evalues.IC( stat = "est",
            unidirBias = TRUE,
            unidirBiasDirection = "unknown",
            
            p1_1 = pw_1,
            p1_0 = pw_0,
            n1_1 = nw_1,
            n1_0 = nw_0,
            f1 = fw,
            
            p0_1 = pm_1,
            p0_0 = pm_0,
            n0_1 = nm_1,
            n0_0 = nm_0,
            f0 = fm )

# and for its lower CI limit
evalues.IC( stat = "CI",
            unidirBias = TRUE,
            unidirBiasDirection = "unknown",
            
            p1_1 = pw_1,
            p1_0 = pw_0,
            n1_1 = nw_1,
            n1_0 = nw_0,
            f1 = fw,
            
            p0_1 = pm_1,
            p0_0 = pm_0,
            n0_1 = nm_1,
            n0_0 = nm_0,
            f0 = fm )
}
\references{
\enumerate{
\item Mathur MB, Smith LH, Yoshida K, Ding P, VanderWeele TJ (2021). E-values for effect modification and approximations for causal interaction. Under review.
\item Ding P & VanderWeele TJ (2016). Sensitivity analysis without assumptions. \emph{Epidemiology.} 27(3), 368.
\item VanderWeele TJ & Ding P (2017). Sensitivity analysis in observational research: Introducing the E-value. \emph{Annals of Internal Medicine.} 27(3), 368.
}
}
