% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GroupSequentialTest.R
\docType{class}
\name{GroupSequentialTest}
\alias{GroupSequentialTest}
\title{Class of GroupSequentialTest}
\description{
Perform group sequential test for a single endpoint based on sequential
one-sided p-values at each stages. Selected alpha spending functions,
including user-defined functions, are supported. Boundaries are calculated
with `rpact`. At the final analysis, adjustment can be applied for
over-running or under-running trial where observed final information is
greater or lower than the planned maximum information. See
Wassmer & Brannath, 2016, p78f. The test is based on p-values not z
statistics because it is easier to not handling direction of alternative
hypothesis in current implementation. In addition, only one-sided test is
supported which should be sufficient for common use in clinical design.
}
\examples{
## Note: examples showed here replicate the results from
## https://www.rpact.org/vignettes/planning/rpact_boundary_update_example/

## Example 1. Generate boundaries for a pre-fix group sequential design
gst <- GroupSequentialTest$new(
  alpha = .025, alpha_spending = 'asOF',
  planned_max_info = 387)

## without giving p-values, boundaries are returned without actual testing
gst$test(observed_info = c(205, 285, 393), is_final = c(FALSE, FALSE, TRUE))
gst

## Example 2. Calculate boundaries with observed information at stages
## No p-values are provided

## get an error without resetting an used object
try( gst$test(observed_info = 500, is_final = FALSE) )

## reset the object for re-use
gst$reset()
gst$test(observed_info = c(205, 285, 393), is_final = c(FALSE, FALSE, TRUE))
gst

## Example 3. Test stagewise p-values sequentially
gst$reset()

gst$test(observed_info = 205, is_final = FALSE, p_values = .09)
gst$test(285, FALSE, .006)

## print testing trajectory by now
gst

gst$test(393, TRUE, .002)

## print all testing trajectory
gst

## you can also test all stages at once
## the result is the same as calling test() for each of the stages
gst$reset()
gst$test(c(205, 285, 393), c(FALSE, FALSE, TRUE), c(.09, .006, .002))
gst

## Example 4. use user-define alpha spending
gst <- GroupSequentialTest$new(
  alpha = .025, alpha_spending = 'asUser',
  planned_max_info = 387)

gst$test(
  observed_info = c(205, 285, 393),
  is_final = c(FALSE, FALSE, TRUE),
  alpha_spent = c(.005, .0125, .025))
gst

}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-GroupSequentialTest-new}{\code{GroupSequentialTest$new()}}
\item \href{#method-GroupSequentialTest-get_name}{\code{GroupSequentialTest$get_name()}}
\item \href{#method-GroupSequentialTest-get_alpha}{\code{GroupSequentialTest$get_alpha()}}
\item \href{#method-GroupSequentialTest-set_alpha_spending}{\code{GroupSequentialTest$set_alpha_spending()}}
\item \href{#method-GroupSequentialTest-get_alpha_spending}{\code{GroupSequentialTest$get_alpha_spending()}}
\item \href{#method-GroupSequentialTest-get_max_info}{\code{GroupSequentialTest$get_max_info()}}
\item \href{#method-GroupSequentialTest-set_max_info}{\code{GroupSequentialTest$set_max_info()}}
\item \href{#method-GroupSequentialTest-get_stage}{\code{GroupSequentialTest$get_stage()}}
\item \href{#method-GroupSequentialTest-reset}{\code{GroupSequentialTest$reset()}}
\item \href{#method-GroupSequentialTest-set_trajectory}{\code{GroupSequentialTest$set_trajectory()}}
\item \href{#method-GroupSequentialTest-get_trajectory}{\code{GroupSequentialTest$get_trajectory()}}
\item \href{#method-GroupSequentialTest-get_stage_level}{\code{GroupSequentialTest$get_stage_level()}}
\item \href{#method-GroupSequentialTest-test_one}{\code{GroupSequentialTest$test_one()}}
\item \href{#method-GroupSequentialTest-test}{\code{GroupSequentialTest$test()}}
\item \href{#method-GroupSequentialTest-print}{\code{GroupSequentialTest$print()}}
\item \href{#method-GroupSequentialTest-clone}{\code{GroupSequentialTest$clone()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-new"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-new}{}}}
\subsection{Method \code{new()}}{
initialize a group sequential test. Now only support one-sided test
based on p-values.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$new(
  alpha = 0.025,
  alpha_spending = c("asP", "asOF", "asUser"),
  planned_max_info,
  name = "H0",
  silent = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{alpha}}{familywise error rate}

\item{\code{alpha_spending}}{alpha spending function. Use \code{"asUser"}
if custom alpha spending schedule is used.}

\item{\code{planned_max_info}}{integer. Planned maximum number of patients for
non-tte endpoints or number of events for tte endpoints}

\item{\code{name}}{character. Name of the hypothesis, e.g. endpoint, subgroup,
etc. Optional.}

\item{\code{silent}}{\code{TRUE} if muting all messages.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_name"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_name}{}}}
\subsection{Method \code{get_name()}}{
get name of hypothesis
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_name()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_alpha"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_alpha}{}}}
\subsection{Method \code{get_alpha()}}{
get overall alpha
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_alpha()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-set_alpha_spending"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-set_alpha_spending}{}}}
\subsection{Method \code{set_alpha_spending()}}{
set alpha spending function. This is useful when set 'asUser' at the
final stage to adjust for an under- or over-running trial.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$set_alpha_spending(asf)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{asf}}{character of alpha spending function.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_alpha_spending"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_alpha_spending}{}}}
\subsection{Method \code{get_alpha_spending()}}{
return character of alpha spending function
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_alpha_spending()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_max_info"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_max_info}{}}}
\subsection{Method \code{get_max_info()}}{
return planned maximum information
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_max_info()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-set_max_info"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-set_max_info}{}}}
\subsection{Method \code{set_max_info()}}{
set planned maximum information. This is used at the final stage
to adjust for an under- or over-running trial.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$set_max_info(obs_max_info)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{obs_max_info}}{integer. Maximum information, which could be
observed number of patients or events at the final stage.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_stage"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_stage}{}}}
\subsection{Method \code{get_stage()}}{
get current stage.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_stage()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-reset"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-reset}{}}}
\subsection{Method \code{reset()}}{
an object of class \code{GroupSequentialTest} is designed to be used
sequentially by calling \code{GroupSequentialTest$test}. When all
planned tests are performed, no further analysis could be done. In that
case keep calling \code{GroupSequentialTest$test} will trigger an error.
To reuse the object for a new set of staged p-values, call this function
to reset the status to stage 1. See examples. This implementation can
prevent the error that more than the planned number of stages are tested.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$reset()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-set_trajectory"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-set_trajectory}{}}}
\subsection{Method \code{set_trajectory()}}{
save testing result at current stage
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$set_trajectory(result, is_final = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{result}}{a data frame storing testing result at a stage.}

\item{\code{is_final}}{logical. \code{TRUE} if final test for the hypothesis,
\code{FALSE} otherwise.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_trajectory"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_trajectory}{}}}
\subsection{Method \code{get_trajectory()}}{
return testing trajectory until current stage. This function can be
called at any stage. See examples.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_trajectory()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-get_stage_level"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-get_stage_level}{}}}
\subsection{Method \code{get_stage_level()}}{
compute boundaries given current (potentially updated) settings. It
returns different values if settings are changed over time.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$get_stage_level()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-test_one"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-test_one}{}}}
\subsection{Method \code{test_one()}}{
test a hypothesis with the given p-value at current stage
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$test_one(
  p_value,
  is_final,
  observed_info,
  alpha_spent = NA_real_
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{p_value}}{numeric. A p-value.}

\item{\code{is_final}}{logical. \code{TRUE} if this test is carried out for
the final analysis.}

\item{\code{observed_info}}{integer. Observed information at current stage. It
can be the number of samples (non-tte) or number of events (tte) at test.
If the current stage is final, observed_info will be used to update
planned_max_info, the alpha spending function (\code{typeOfDesign}
in \code{rpact}) will be updated to \code{'asUser'}, and the argument
\code{userAlphaSpending} will be used when calling
\code{rpact::getDesignGroupSequential}.}

\item{\code{alpha_spent}}{numeric if \code{alpha_spending = "asUser"}. It must
be between 0 and \code{alpha}, the overall alpha of the test.
\code{NA_real_} for other alpha spending functions \code{"asOF"} and
\code{"asP"}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-test"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-test}{}}}
\subsection{Method \code{test()}}{
Carry out test based on group sequential design. If \code{p_values}
is \code{NULL}, dummy values will be use and boundaries are calculated
for users to review.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$test(
  observed_info,
  is_final,
  p_values = NULL,
  alpha_spent = NULL
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{observed_info}}{a vector of integers, observed information at stages.}

\item{\code{is_final}}{logical vector. \code{TRUE} if the test is for the final
analysis.}

\item{\code{p_values}}{a vector of p-values. If specified, its length should
equal to the length of \code{observed_info}.}

\item{\code{alpha_spent}}{accumulative alpha spent at observed information.
It is a numeric vector of values between 0 and 1, and of length that
equals \code{length(observed_info)} if alpha-spending
function is \code{"asUser"}. Otherwise \code{NULL}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-print"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-print}{}}}
\subsection{Method \code{print()}}{
generic function for \code{print}
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$print()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-GroupSequentialTest-clone"></a>}}
\if{latex}{\out{\hypertarget{method-GroupSequentialTest-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{GroupSequentialTest$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
