% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tdv.R
\name{tdv}
\alias{tdv}
\title{The Total Differential Value of a phytosociological table}
\usage{
tdv(m_bin, p, output_type = "normal")
}
\arguments{
\item{m_bin}{A matrix. A phytosociological table of 0s (absences) and 1s
(presences), where rows correspond to taxa and columns correspond to
relevés.}

\item{p}{A vector of integer numbers with the partition of the relevés (i.e.,
a k-partition, consisting in a vector with values from 1 to k, with length
equal to the number of columns of \code{m_bin}, ascribing each relevé to one of
the k groups).}

\item{output_type}{A character determining the amount of information returned
by the function and also the amount of pre-validations. Possible values are
"normal" (the default), "fast" and "full".}
}
\value{
If \code{output_type = "normal"} (the default) pre-validations are done
and a list is returned, with the following components:

\describe{
\item{ifp}{A matrix with the \eqn{\frac{a}{b}}{a/b} values for each taxon
in each group, for short called the 'inner frequency of presences'.}
\item{ofda}{A matrix with the \eqn{\frac{c}{d}}{c/d} values for each
taxon in each group, for short called the 'outer frequency of
differentiating absences'.}
\item{e}{A vector with the \eqn{e} values for each taxon, i.e., the
number of groups containing that taxon.}
\item{diffval}{A matrix with the \eqn{DiffVal} for each taxon.}
\item{tdv}{A numeric with the TDV of matrix \verb{m_bin,} given the partition
\code{p}.}
}

If \code{output_type = "full"}, some extra components are added to the output:
\code{afg}, \code{empty.size}, \code{gct} (= \eqn{e}) and \code{i.mul}. These are intermediate
matrices used in the computation of TDV.

If \code{output_type = "fast"}, only TDV is returned and no pre-validations are
done.
}
\description{
Given a phytosociological table and a partition of its columns, this function
calculates the respective Total Differential Value (TDV).
}
\details{
The function accepts a phytosociological table (\code{m_bin}) and a
k-partition of its columns (\code{p}), returning the corresponding TDV.
TDV was proposed by Monteiro-Henriques and Bellu (2014).
Monteiro-Henriques (2016) proposed TDV1, modifying TDV slightly with the
objective of ensuring a value from 0 to 1. Yet, TDV is always within that
range. In practice, both TDV and TDV1 have 0 as possible minimum value
and 1 as possible maximum value, but TDV1 reduces further the contribution
of differential taxa present in more than one group. TDV is then
implemented here, for parsimony.

TDV is calculated using the \eqn{DiffVal} index for each (and all) of the
taxa present in a tabulated phytosociological table \eqn{M} (also called
sorted table). \eqn{DiffVal} index aims at characterizing how well a taxon
works as a differential taxon in a such tabulated phytosociological table
(for more information on differential taxa see Mueller-Dombois & Ellenberg,
1974).

An archetypal differential taxon of a certain group \eqn{g} of the
partition \eqn{p} (a partition on the columns of \eqn{M}) is the one
present in all relevés of group \eqn{g}, and absent from all the other
groups of that partition. Therefore, \eqn{DiffVal} has two components, an
inner one (\eqn{\frac{a}{b}}{a/b}), which measures the presence of the
taxon inside each of the groups, and an outer one (\eqn{\frac{c}{d}}{c/d}),
which measures the relevant absences of the taxon outside of each of the
groups. Specifically, given a partition \eqn{p} with \eqn{k} groups,
\eqn{DiffVal} is calculated for each taxon \eqn{s} as:

\deqn{DiffVal_{s,p} = \frac{1}{e}\sum_{g=1}^k{\frac{a}{b}\frac{c}{d}}}{
   DiffVal of taxon s, given the partition p = the summation of
   [(a/b)*(c/d)]/e, obtained for each group g, from g = 1 to k
  }
where:
\itemize{
\item \eqn{a}, is the total number of presences of taxon \eqn{s} within group
\eqn{g}.
\item \eqn{b}, is the total number of relevés of group \eqn{g}.
\item \eqn{c}, is the total number of differentiating absences of taxon
\eqn{s}, i.e., absences coming from the groups other than \eqn{g} from
which the taxon \eqn{s} is completely absent.
\item \eqn{d}, is the total number of relevés of all groups but \eqn{g} (i.e.,
the total number of relevés in the table - \eqn{b}).
\item \eqn{e}, is the total number of groups in which the taxon \eqn{s} occurs
at least once.
}

Therefore, for each taxon \eqn{s} and for each group \eqn{g}, the
\eqn{DiffVal} index evaluates:
\itemize{
\item \eqn{\frac{a}{b}}{a/b}, i.e., the frequency of the presences of taxon
\eqn{s}, relative to the size of group \eqn{g}; commonly called 'relative
frequency.' \eqn{\frac{a}{b}}{a/b} is only 1 if and only if taxon \eqn{s}
occurs in all the relevés of group \eqn{g}.
\item \eqn{\frac{c}{d}}{c/d}, i.e., the frequency of the differentiating
absences of taxon \eqn{s} outside group \eqn{g}, relative to the sum of
sizes of all groups but \eqn{g}. \emph{Nota bene}: absences in \eqn{c} are
counted outside the group \eqn{g} but only in the groups from which taxon
\eqn{s} is completely absent (these are the relevant absences, which
produce differentiation among groups); in practice \eqn{c} corresponds to
the sum of the sizes of all groups other than \eqn{g} that are empty.
\eqn{\frac{c}{d}}{c/d} is 1 if and only if the taxon \eqn{s} is absent
from all groups but \eqn{g}.
}

Finally, \eqn{\frac{1}{e}}{1/e} ensures that \eqn{DiffVal} is a value
from 0 to 1.

The Total Differential Value (TDV or \eqn{TotDiffVal}) of a
phytosociological table \eqn{M} tabulated/sorted by the partition \eqn{p}
is:

\deqn{TDV_{M,p} = \frac{1}{n}\sum_{i=1}^n{Diffval_{i,p}}}{TDV of table M,
  given the partition p = the summation of the DiffVal of all taxa in the
  table, divided by n}
where:
\itemize{
\item \eqn{n}, is the number of taxa in table \eqn{M}.
}

The division by the number of taxa present in \eqn{M} ensures that TDV
remains in the [0,1] interval (as \eqn{DiffVal} is also in the same
interval).
}
\examples{
# Getting the Taxus baccata forests data set
data(taxus_bin)

# Creating a group partition, as the one presented in the original article of
# the data set
groups <- rep(c(1, 2, 3), c(3, 11, 19))

# Removing taxa occurring in only one relevé, in order to reproduce exactly
# the example in the original article of the data set
taxus_bin_wmt <- taxus_bin[rowSums(taxus_bin) > 1, ]

# Calculating TDV
result <- tdv(taxus_bin_wmt, groups)

# This is the TDV
result$tdv
# This is TDV1, reproducing exactly the value from the original article
sum(result$diffval / result$e) / nrow(taxus_bin_wmt)

}
\references{
Monteiro-Henriques T. & Bellu A. 2014. \emph{An optimization approach to the}
\emph{production of differentiated tables based on new differentiability}
\emph{measures}. 23rd EVS European Vegetation Survey. Presented orally. Ljubljana,
Slovenia.

Monteiro-Henriques T. 2016. \emph{A bunch of R functions to assist}
\emph{phytosociological tabulation}. 25th Meeting of European Vegetation Survey.
Presented in poster. Rome. Italy.

Mueller-Dombois D. & Ellenberg H. 1974. \emph{Aims and Methods of Vegetation}
\emph{Ecology}. New York: John Wiley & Sons.
}
\author{
Tiago Monteiro-Henriques. E-mail: \email{tmh.dev@icloud.com}.
}
