% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eval_pred_equality.R
\name{eval_pred_equality}
\alias{eval_pred_equality}
\title{Examine Predictive Equality of a Model}
\usage{
eval_pred_equality(
  data,
  outcome,
  group,
  probs,
  cutoff = 0.5,
  confint = TRUE,
  alpha = 0.05,
  bootstraps = 2500,
  digits = 2,
  message = TRUE
)
}
\arguments{
\item{data}{Data frame containing the outcome, predicted outcome, and
binary protected attribute}

\item{outcome}{Name of the outcome variable, it must be binary}

\item{group}{Name of the protected attribute. Must consist of only two groups.}

\item{probs}{Name of the predicted outcome variable}

\item{cutoff}{Threshold for the predicted outcome, default is 0.5}

\item{confint}{Whether to compute 95\% confidence interval, default is TRUE}

\item{alpha}{The 1 - significance level for the confidence interval, default is 0.05}

\item{bootstraps}{Number of bootstrap samples, default is 2500}

\item{digits}{Number of digits to round the results to, default is 2}

\item{message}{Logical; if TRUE (default), prints a textual summary of the
fairness evaluation. Only works if \code{confint} is TRUE.}
}
\value{
A list containing the following elements:
\itemize{
\item FPR_Group1: False Positive Rate for the first group
\item FPR_Group2: False Positive Rate for the second group
\item FPR_Diff: Difference in False Positive Rate
\item FPR_Ratio: Ratio in False Positive Rate
If confidence intervals are computed (\code{confint = TRUE}):
\item FPR_Diff_CI: A vector of length 2 containing the lower and upper bounds
of the 95\% confidence interval for the difference in False Positive Rate
\item FPR_Ratio_CI: A vector of length 2 containing the lower and upper bounds
of the 95\% confidence interval for the ratio in False Positive Rate
}
}
\description{
This function evaluates predictive equality, a fairness metric that compares the
False Positive Rate (FPR) between groups defined by a binary protected attribute. It assesses
whether individuals from different groups are equally likely to be incorrectly flagged as
positive when they are, in fact, negative.
}
\examples{
\donttest{
library(fairmetrics)
library(dplyr)
library(magrittr)
library(randomForest)
data("mimic_preprocessed")
set.seed(123)
train_data <- mimic_preprocessed \%>\%
  dplyr::filter(dplyr::row_number() <= 700)
# Fit a random forest model
rf_model <- randomForest::randomForest(factor(day_28_flg) ~ ., data = train_data, ntree = 1000)
# Test the model on the remaining data
test_data <- mimic_preprocessed \%>\%
  dplyr::mutate(gender = ifelse(gender_num == 1, "Male", "Female")) \%>\%
  dplyr::filter(dplyr::row_number() > 700)

test_data$pred <- predict(rf_model, newdata = test_data, type = "prob")[, 2]

# Fairness evaluation
# We will use sex as the protectedR attribute and day_28_flg as the outcome.
# We choose threshold = 0.41 so that the overall FPR is around 5\%.

# Evaluate Predictive Equality
eval_pred_equality(
  data = test_data,
  outcome = "day_28_flg",
  group = "gender",
  probs = "pred",
  cutoff = 0.41
)
}
}
\seealso{
\code{\link{eval_pos_pred_parity}},  \code{\link{eval_neg_pred_parity}}, \code{\link{eval_stats_parity}}
}
