#' Function for preparing book metadata with organizers for automatic insertion into scientific articles
#'
#' The bookorg_bib function prepares the metadata of books with organizers to be automatically included in a scientific article. The bookorg_bib function internally calls two support functions. The output of the bookorg_bib function is the file with the .bib extension of books with organizers in BibTex format, prepared to be automatically included in the reference list of a scientific article.
#'
#' @param input_date is a dataframe with the scientific production database in the form of a book with organizers
#' @param input_tam is the length of the dataframe with the book with organizers, obtained using package bibliorefer.
#' @return The output of the bookorg_bib function is the file with the .bib extension of books with organizers in BibTex format, prepared to be automatically included in the reference list of a scientific article.
#' @importFrom utils read.csv2
#' @importFrom bibliorefer principal_lister
#' @export
#'
#' @references
#' 1 - Aria, M. & Cuccurullo, C. (2017) bibliometrix: An R-tool for comprehensive science mapping analysis, Journal of Informetrics, 11(4), pp 959-975, Elsevier.
#' 2 - Bibliometric indicators to evaluate scientific activity. C García-Villar, J M García-Santos. Radiologia (Engl Ed). 2021 May-Jun;63(3):228-235.
#'
#'
#' @examples
#'
#' # Example
#'
#' # File of database
#' file_db <- system.file("extdata","bookorg.csv", package = "gerefer")
#' separator <- ";"
#'
#' # Parameters of the function
#' input_date <- example_database(file_db, separator)
#' input_tam <- 9
#'
#' #Calls the function bookorg_bib
#' bookorglister_bibtex <- bookorg_bib(input_date, input_tam)
#' bookorglister_bibtex
#'
#'
#'
bookorg_bib <- function(input_date,input_tam){

  # Data entry
  input_date <- input_date
  input_tam <- input_tam

  # Call the function
  gerad_bookorg <- function(input_date){

    # Replace dataframe with matrix
    input_entrada <- as.matrix(input_date)

    # Delete NA
    na_retir <- function(input_entr){

      # Data entry
      entrada_base <- input_entr
      entrada_base <- as.data.frame(entrada_base)
      colnames(entrada_base) <- "base"

      # Delete NA
      variavel <- entrada_base$base
      testena <- is.na(variavel)
      padrao <- "TRUE"
      linha_na <- grep(padrao,testena)
      variavel[linha_na] <- ""

      return(variavel)

    }

    # Create the loop indexer
    nomes_colunas <- names(input_date)

    # Create the loop along the columns
    for (i in seq_along(nomes_colunas)){

      input_entr <- nomes_colunas[i]
      names(input_entrada) <- colnames(input_date)
      coluna <- na_retir(input_entrada[,input_entr])
      input_entrada[,i] <- coluna

    }

    # Change format from matrix to dataframe
    tabela <- input_entrada
    tabela <- as.data.frame(input_entrada)
    ##

    # Create the extra table
    n <- length(tabela$autor)
    citacoes <- 1:n
    link <- rep("",n)
    abstract <- rep("",n)
    paginas <- rep("",n)
    volume <- rep("",n)
    extra <- cbind(volume,link,abstract,citacoes)
    extra <- as.data.frame(extra)

    # Create the input table and change the names
    nomes <- c("AU","TI", "edicao", "local", "SO", "PY", "PP",
               "VL","DI","AB","TC")
    baseentrada <- cbind(tabela, extra)
    colnames(baseentrada) <- nomes

    # Call the main list function
    input_dateprin <- baseentrada
    input_tam <- n
    listaprincipal <- principal_lister(input_dateprin,input_tam)
    lista_prin <- listaprincipal[[1]]
    posicao_book <- lista_prin$Article
    posicao_info <- which(posicao_book != "")
    ##

    # Calculating the size of tables
    tamanho <- length(posicao_info)

    # Create the input table and change the name
    tabela_book <- tabela
    #nomes_tabela <- c("autor", "titulo", "tradutor", "edicao", "local",
    #"editora", "ano", "pagina", "totalpagina", "volume",
    #"serie", "desconline", "link", "acesso")

    nomes_tabela <- c("autor", "tipo", "titulo", "tradutor",
                      "edicao", "local", "editora", "ano", "pagina",
                      "totalpagina", "volume", "serie", "desconline",
                      "link", "acesso")


    colnames(tabela_book) <- nomes_tabela
    ####

    # Create the local variable publisher
    local_bo <- tabela_book$local
    editora_bo <- tabela_book$editora

    # Create variables
    tabela_bookedicoes <- numeric(tamanho)
    tabela_bookvolumes <- numeric(tamanho)
    tabela_bookpaginas <- numeric(tamanho)

    # Create the test for editing
    teste_edicaona <- is.na(tabela_book$edicao)
    teste_edicaovazio <- tabela_book$edicao == ""

    # Create link for editing
    for(i in 1:tamanho){

      # Standardize the page separator
      if(teste_edicaona[i] == FALSE && teste_edicaovazio[i] == FALSE){

        tabela_bookedicao <- tabela_book$edicao[i]

      }else if(teste_edicaona[i] == FALSE && teste_edicaovazio[i] == TRUE){

        tabela_bookedicao <- ""

      }else{

        tabela_bookedicao <- ""

      }

      tabela_bookedicoes[i] <- tabela_bookedicao
    }

    # Create the test for the volume
    teste_volumena <- is.na(tabela_book$volume)
    teste_volumevazio <- tabela_book$volume == ""

    # Create the loop for the volume
    for(i in 1:tamanho){

      # Standardize the page separator
      if(teste_volumena[i] == FALSE && teste_volumevazio[i] == FALSE){

        tabela_bookvolume <- tabela_book$volume[i]

      }else if(teste_volumena[i] == TRUE){

        tabela_bookvolume <- ""

      }else {

        tabela_bookvolume <- ""

      }

      tabela_bookvolumes[i] <- tabela_bookvolume
    }

    ##
    # Organization of pages
    saida_pagina <- numeric(tamanho)

    pagina_refer <- function(entrada_pagina){

      entrada_pagina <- entrada_pagina

      pagina_refe <- function(entrada_pagina){

        # Check lines with page separation pattern
        hifsesp <- "\u002D"
        hifcesp <- " \u002D "
        linhas_hifsesp <- grepl(hifsesp,entrada_pagina)
        linhas_hifcesp <- grepl(hifcesp,entrada_pagina)

        mriscsesp <- "\u2013"
        mrisccesp <- " \u2013 "
        linhas_mriscsesp <- grepl(mriscsesp,entrada_pagina)
        linhas_mrisccesp <- grepl(mrisccesp,entrada_pagina)

        if(linhas_hifsesp == TRUE && linhas_hifcesp == FALSE){
          hifensesp <- "\u002D"
          pagin_liv <- strsplit(entrada_pagina,
                                split = hifensesp)
          traco1 <- "\u002D"
          traco2 <- "\u002D"
          tracom2 <- paste0(traco1,traco2)
          saida_pagin <- paste0(pagin_liv[[1]][1],tracom2,
                                pagin_liv[[1]][2])

        }else if(linhas_hifsesp == TRUE && linhas_hifcesp == TRUE){
          hifencesp <- " \u002D "
          pagin_liv <- strsplit(entrada_pagina,
                                split = hifencesp)
          traco1 <- "\u002D"
          traco2 <- "\u002D"
          tracom2 <- paste0(traco1,traco2)
          saida_pagin <- paste0(pagin_liv[[1]][1],tracom2,
                                pagin_liv[[1]][2])

        }else if(linhas_mriscsesp == TRUE && linhas_mrisccesp == FALSE){

          mriscsesp <- "\u2013"
          pagin_liv <- strsplit(entrada_pagina,
                                split = mriscsesp)
          traco1 <- "\u002D"
          traco2 <- "\u002D"
          tracom22 <- paste0(traco1,traco2)
          saida_pagin <- paste0(pagin_liv[[1]][1],tracom22,
                                pagin_liv[[1]][2])
        }else{

          mriscsesp <- " \u2013 "
          pagin_liv <- strsplit(entrada_pagina,
                                split = mrisccesp)
          traco1 <- "\u002D"
          traco2 <- "\u002D"
          tracom22 <- paste0(traco1,traco2)
          saida_pagin <- paste0(pagin_liv[[1]][1],tracom22,
                                pagin_liv[[1]][2])
        }

        return(saida_pagin)
      }

      for(i in 1:tamanho){

        entrada_pagina <- tabela_book$pagina[i]

        if(entrada_pagina == ""){

          saida_pagin <- ""

        }else{

          saida_pagin <- pagina_refe(entrada_pagina)

        }

        saida_pagina[i] <- saida_pagin

      }

      return(saida_pagina)

    }
    ##

    # Create the year author identity
    # Separate team authors
    autsep_book <- strsplit(tabela_book$autor, split = "; ")

    # Create storage vectors
    prim_aut <- numeric(tamanho)
    primei_aut <- numeric(tamanho)
    primei_auto <- numeric(tamanho)
    primei_autor <- numeric(tamanho)
    autor_ano <- numeric(tamanho)
    tamanho_aut <- numeric(tamanho)

    # Create the loop to generate the author year identity
    for(i in 1:tamanho){

      prim_aut[i] <- autsep_book[[i]][1]
      primei_aut[i] <- strsplit(prim_aut[[i]][1],split = ",")
      primei_auto[i] <- primei_aut[[i]][1]
      primei_autor[i] <- strsplit(primei_auto[[i]][1],split = " ")
      tamanho_aut[i] <- length(primei_autor[[i]])
      tamanho_auttest <- tamanho_aut[i]
    }

    # Create the loop to generate the author year identity
    for(i in 1:tamanho){

      prim_aut[i] <- autsep_book[[i]][1]
      primei_aut[i] <- strsplit(prim_aut[[i]][1],split = ",")
      primei_auto[i] <- primei_aut[[i]][1]
      primei_autor[i] <- strsplit(primei_auto[[i]][1],split = " ")
      tamanho_aut[i] <- length(primei_autor[[i]])
      tamanho_auttest <- tamanho_aut[i]

      # Standardize the page separator
      if(tamanho_auttest < 2){

        autor_ano[i] <- paste0(primei_autor[[i]][1],tabela_book$ano[i])

      }else{

        autor_ano[i] <- paste0(primei_autor[[i]][1],primei_autor[[i]][2],
                               tabela_book$ano[i])
      }

    }

    # Create variables
    tamanho_autor <- numeric(tamanho)
    equipesnomes <- numeric(tamanho)

    # Create the loop to generate the author year identity
    for(i in 1:tamanho){

      tamanho_autor[i] <- length(autsep_book[[i]])

      if(tamanho_autor[i] <= 1){

        equipesnom <- autsep_book[[i]]

      }else{
        conexao <- " and "
        equipenom <- numeric(tamanho_autor[i])

        # Create team first name
        equipenom[1] <- paste0(autsep_book[[i]][1],conexao)
        equipesnom <- equipenom[1]

        # Create team name sequence
        for(ind in 2:(tamanho_autor[i]-1)) {

          equipenom[ind] <- paste0(equipenom[ind-1],
                                   autsep_book[[i]][ind],conexao)
          equipesnom <- equipenom[ind]
        }

        # Create the last name of the sequence
        equipenom[tamanho_autor[i]] <- paste0(equipenom[tamanho_autor[i]-1],
                                              autsep_book[[i]][tamanho_autor[i]])
        equipesnom <- equipenom[tamanho_autor[i]]

      }
      equipesnomes[i] <- equipesnom
    }
    ##
    tamanho_subti <- numeric(tamanho)
    titul_artic <- numeric(tamanho)
    subtitul_artic <- numeric(tamanho)

    # Call the function from the authors name
    titulo_info <- function(enti_artic){

      entitu_artic <- enti_artic

      for(i in 1:tamanho){

        enti_artic <- entitu_artic[i]

        # Return the author's name
        padrao_pont <- ":"
        padrao_pont1 <- ":"
        test_subtitu <- grepl(padrao_pont,enti_artic)

        padrao_true <- "TRUE"
        linha_sub <- grep(padrao_true,test_subtitu)
        tamanho_titul <- length(linha_sub)

        # Standardize the page separator
        if(test_subtitu == FALSE){

          titu_artic <- enti_artic
          subtitu_artic <- ""

        }else{

          titulo_basesepar <- strsplit(enti_artic,
                                       split = padrao_pont)
          comp_subti <- length(titulo_basesepar[[1]])

          if(comp_subti == 2){

            titu_artic <- paste0(titulo_basesepar[[1]][1],padrao_pont1)
            subtitu_artic <- titulo_basesepar[[1]][2]
            subtitu_artic <- tolower(subtitu_artic)

          }else{

            titu_artic <- paste0(titulo_basesepar[[1]][1],padrao_pont1)
            subtitu_artic <- titulo_basesepar[[1]][2]

            for(k in 3:comp_subti){

              subtitu_artic <- paste0(subtitu_artic, sep = ".",
                                      titulo_basesepar[[1]][k])
              subtitu_artic <- tolower(subtitu_artic)
            }

          }

        }

        titul_artic[i] <- titu_artic
        subtitul_artic[i] <- subtitu_artic

      }

      tabela_titu <- cbind(titul_artic,subtitul_artic)
      tabela_titu <- as.data.frame(tabela_titu)

      return(tabela_titu)

    }

    ##
    # Call the access date function
    acesso_datas <- function(input_dateace){

      # Remove NA
      na_retir <- function(input_entr){

        # Data entry
        entrada_base <- input_entr
        entrada_base <- as.data.frame(entrada_base)
        colnames(entrada_base) <- "base"

        # Remove NA
        variavel <- entrada_base$base
        testena <- is.na(variavel)
        padrao <- "TRUE"
        linha_na <- grep(padrao,testena)
        variavel[linha_na] <- ""

        return(variavel)

      }

      # Call function to remove NA
      input_entr <- input_dateace
      entrada_data <- na_retir(input_entr)

      # Calculate the column size and generate the corresponding vectors
      n <- length(entrada_data)
      data_acessos <- numeric(n)
      dia_acesso <- numeric(n)

      # Create the loop
      for(i in 1:n){

        if(entrada_data[i] == ""){

          dia_acess <- ""

        }else{

          data_sepa <- strsplit(entrada_data, split = "/")
          data_separ <- data_sepa[[i]]

          # Create the day
          ind_dia <- 1
          dia_separ <- data_separ[ind_dia]

          # Create the month
          ind_mes <- 2
          mes_separ <- data_separ[ind_mes]

          # Create the year
          ind_ano <- 3
          ano_sepa <- data_separ[ind_ano]
          inicio_ano <- 20
          ano_separ <- paste0(inicio_ano,ano_sepa)
          ano_separ <- as.numeric(ano_separ)

          # Generate the access date
          hifen <- "\u002D"
          data_acesso <- paste0(ano_separ, hifen, mes_separ, hifen, dia_separ)
          data_acessos[i] <- data_acesso

          dia_acess <- data_acessos[i]

        }

        dia_acesso[i] <- dia_acess

      }

      return(dia_acesso)

    }
    ###
    edic_modi <- function(input_edic,tamanho){

      input_edic <- input_edic
      entrada_edicao <- na_retir(input_edic)

      espaco <- " "
      edic <- "ed"
      ponto <- "."

      edicao_livro <- numeric(tamanho)

      # Create the loop
      for(i in 1:tamanho){

        if(entrada_edicao[i] == ""){

          edicao_liv <- ""

        }else{

          entrada_edic <- entrada_edicao[i]
          edicao_liv <- paste0(entrada_edic,ponto,espaco,edic)
        }

        edicao_livro[i] <- edicao_liv
      }

      edicmod <- edicao_livro

      return(edicmod)

    }
    ###

    book_orgtradu <- function(input_orgtradu){

      input_entr <-  input_orgtradu

      # Delete NA
      na_retir <- function(input_entr){

        # Data entry
        entrada_base <- input_entr
        entrada_base <- as.data.frame(entrada_base)
        colnames(entrada_base) <- "base"

        # Delete NA
        variavel <- entrada_base$base
        testena <- is.na(variavel)
        padrao <- "TRUE"
        linha_na <- grep(padrao,testena)
        variavel[linha_na] <- ""

        return(variavel)

      }

      input_tradu <- na_retir(input_entr)

      entrada <- input_tradu

      n <- length(entrada)
      saida_tradu <- numeric(n)

      for(i in 1:n){

        if(entrada[i] == ""){

          entrada_trad <- ""

        }else{

          entrada_trad <- entrada[i]

        }

        saida_tradu[i] <- entrada_trad

      }

      return(saida_tradu)

    }
    ##
    ebook_it <- function(input_desc){

      entrada <- input_desc
      n <- length(entrada)
      entrada_nova <- numeric(n)

      for(i in 1:n){

        padrao_ebook <- "E-book"
        teste_ebook <- grepl(padrao_ebook,entrada[i])

        if(teste_ebook == TRUE){

          padrao_italico <- "\\\\textit{E-book}"
          entrada_nov <- sub(padrao_ebook,padrao_italico,entrada[i])

        }else{

          entrada_nov <- entrada[i]

        }

        entrada_nova[i] <- entrada_nov

      }

      return(entrada_nova)

    }
    ##
    editor_tipe <- function(input_edtip){

      entrada_tipo <- input_edtip
      n <- length(entrada_tipo)
      editor_tipo <- numeric(n)

      tipo_org <- c("editor", "coordenador", "compilador", "organizador")

      for(i in 1:n){

        if(entrada_tipo[i] == ""){

          editor_tip <- ""

        }else if(entrada_tipo[i] == tipo_org[1]){

          editor_tip <- "ed."

        }else if(entrada_tipo[i] == tipo_org[2]){

          editor_tip <- "coord."

        }else if(entrada_tipo[i] == tipo_org[3]){

          editor_tip <- "comp."

        }else{

          editor_tip <- "org."
        }

        editor_tipo[i] <- editor_tip

      }

      return(editor_tipo)

    }
    ##

    # Create access dates
    entrada_pagina <- tabela_book$pagina
    paginas <- pagina_refer(entrada_pagina)
    input_dataacesso <- tabela_book$acesso
    data_acessos <- acesso_datas(input_dataacesso)

    # Create the titles
    enti_artic <- tabela_book$titulo
    titu_book <- titulo_info(enti_artic)
    titul_artic <- titu_book$titul_artic
    subtitul_artic <- titu_book$subtitul_artic

    # Create edit
    tabela_book$edicao <- as.numeric(tabela_book$edicao)
    input_edic <- tabela_book$edicao
    edicao <- edic_modi(input_edic,tamanho)

    # Create the translator
    input_orgtradu <- tabela_book$tradutor
    saida_orgtradu <- book_orgtradu(input_orgtradu)

    # Make the term ebook italic
    input_desc <- tabela_book$desconline
    saida_ebook <- ebook_it(input_desc)

    # Create the editor type
    input_edtip <- tabela_book$tipo
    editor_tipo <- editor_tipe(input_edtip)

    # Create the table with the information of interest and change the format
    tabel_bibtexi <- cbind(autor_ano, equipesnomes, editor_tipo, titul_artic, subtitul_artic,
                           saida_orgtradu, edicao, local_bo, editora_bo, tabela_book$ano,
                           paginas, tabela_book$totalpagina, tabela_book$volume,
                           tabela_book$serie, saida_ebook, tabela_book$link,
                           data_acessos)
    tabel_bibtexi <- as.data.frame(tabel_bibtexi)

    # Change column names
    nomes <- c("fonte_ano","editores","editortipo","titulo","subtitulo","tradutor",
               "edicao","local","editora","ano","paginas","totalpagina",
               "volume","serie","desconline","link","acesso")
    colnames(tabel_bibtexi) <- nomes

    # Create the list
    tabela_bibtex <- list()
    tabela_bibtex <- list(tabel_bibtexi, tamanho)
    names(tabela_bibtex) <- c("tabel_bibtex", "tamanho")

    return(tabela_bibtex)

  }

  #Call the gerard_book function
  tabel_bibte <- gerad_bookorg(input_date)
  # Create the variables of interest
  tabel_bibtex <- tabel_bibte[[1]]
  tamanho <- tabel_bibte[[2]]

  # Function that generates the file in bibtex format
  bibtex_lister <- function(tabel_bibtex,indic){

    # Data entry
    input_tabel <- tabel_bibtex
    tabela <- input_tabel
    linha <- indic

    # Reduce database size
    tabela_redu <- tabela[linha,]
    tabela_ent <- tabela_redu

    # Assign input information to variables
    simbolo_entrada <- "@book{"
    fonte_ano <- tabela_ent$fonte_ano
    editor <- tabela_ent$editores
    editortipo <- tabela_ent$editortipo
    titulo <- tabela_ent$titulo
    subtitulo <- tabela_ent$subtitulo
    tradutor <- tabela_ent$tradutor
    edicao <- tabela_ent$edicao
    local <- tabela_ent$local
    editora <- tabela_ent$editora
    ano <- tabela_ent$ano
    pagina <- tabela_ent$paginas
    totalpagina <- tabela_ent$totalpagina
    volume <- tabela_ent$volume
    serie <- tabela_ent$serie
    desconline <- tabela_ent$desconline
    link <- tabela_ent$link
    acesso <- tabela_ent$acesso

    # Create the elements of the bibtex format structure
    espaco <- "  "
    virgula <- ","
    editorbook <- "editor="
    editortipobook <- "editortype="
    titulobook <- "title="
    subtitulobook <- "subtitle="
    tradutorbook <- "translator="
    edicaobook <- "edition="
    localbook <- "location="
    editorabook <- "publisher="
    anobook <- "year="
    paginabook <- "pages="
    totalpaginabook <- "pagetotal="
    volumebook <- "volume="
    seriebook <- "series="
    desconlinebook <- "note="
    linkbook <- "url="
    acessobook <- "urldate="
    chave_entrada <- "{"
    chave_saida <- "}"

    # Create the bibtex structure for each variable
    chave <- paste0(simbolo_entrada,fonte_ano,virgula)
    editor <- paste0(editorbook,chave_entrada,editor,
                     chave_saida)
    editortipo <- paste0(editortipobook,chave_entrada,editortipo,
                         chave_saida)
    titulo <- paste0(titulobook,chave_entrada,titulo,
                     chave_saida)
    subtitulo <- paste0(subtitulobook,chave_entrada,subtitulo,
                        chave_saida)
    tradutor <- paste0(tradutorbook,chave_entrada,tradutor,
                       chave_saida)
    edicao <- paste0(edicaobook,chave_entrada,edicao,
                     chave_saida)
    local <- paste0(localbook,chave_entrada,local,
                    chave_saida)
    editora <- paste0(editorabook,chave_entrada,editora,
                      chave_saida)
    ano <- paste0(anobook,chave_entrada,ano,
                  chave_saida)
    pagina <- paste0(paginabook,chave_entrada,pagina,
                     chave_saida)
    totalpagina <- paste0(totalpaginabook,chave_entrada,totalpagina,
                          chave_saida)
    volume <- paste0(volumebook,chave_entrada,volume,
                     chave_saida)
    serie <- paste0(seriebook,chave_entrada,serie,
                    chave_saida)
    desconline <- paste0(desconlinebook,chave_entrada,desconline,
                         chave_saida)
    link <- paste0(linkbook,chave_entrada,link,
                   chave_saida)
    acesso <- paste0(acessobook,chave_entrada,acesso,
                     chave_saida)

    # Create bibtex format for articles
    referen_bibtex <- paste0(chave,"\n",
                             espaco, editor, virgula, "\n",
                             espaco, editortipo, virgula, "\n",
                             espaco, titulo, virgula, "\n",
                             espaco, subtitulo, virgula, "\n",
                             espaco, tradutor, virgula, "\n",
                             espaco, edicao, virgula, "\n",
                             espaco, local, virgula, "\n",
                             espaco, editora, virgula,"\n",
                             espaco, ano, virgula, "\n",
                             espaco, pagina, virgula, "\n",
                             espaco, totalpagina, virgula, "\n",
                             espaco, volume, virgula, "\n",
                             espaco, serie, virgula, "\n",
                             espaco, desconline, virgula, "\n",
                             espaco, link, virgula, "\n",
                             espaco, acesso, "\n", chave_saida,
                             espaco, collapse = ",\n")

    # Returns the file in bibtex format
    return(referen_bibtex)
  }

  # Create the variable to store values
  arquivo_bibtex <- c()

  # Create the loop to generate the list of bibtex files
  for(i in 1:tamanho){

    # Create the parameter linked to the index
    indic <- i

    # Call the function that converts to bibtex format
    refer_bibtex <- bibtex_lister(tabel_bibtex,indic)
    arquivo_bibtex[i] <- refer_bibtex

  }

  # Return the value of the variable
  lista_final <- arquivo_bibtex

  return(lista_final)

}
