% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/visualise_methylation.R
\name{visualise_methylation}
\alias{visualise_methylation}
\title{Visualise methylation probabilities for many DNA sequences}
\usage{
visualise_methylation(
  modification_locations,
  modification_probabilities,
  sequence_lengths,
  low_colour = "blue",
  high_colour = "red",
  low_clamp = 0,
  high_clamp = 255,
  background_colour = "white",
  other_bases_colour = "grey",
  outline_colour = "black",
  outline_linewidth = 3,
  outline_join = "mitre",
  modified_bases_outline_colour = NA,
  modified_bases_outline_linewidth = NA,
  modified_bases_outline_join = NA,
  other_bases_outline_colour = NA,
  other_bases_outline_linewidth = NA,
  other_bases_outline_join = NA,
  margin = 0.5,
  return = TRUE,
  filename = NA,
  render_device = ragg::agg_png,
  pixels_per_base = 20
)
}
\arguments{
\item{modification_locations}{\verb{character vector}. One character value for each sequence, storing a condensed string (e.g. \code{"3,6,9,12"}, produced via \code{\link[=vector_to_string]{vector_to_string()}}) of the indices along the read at which modification was assessed. Indexing starts at 1.}

\item{modification_probabilities}{\verb{character vector}. One character value for each sequence, storing a condensed string (e.g. \code{"0,128,255,15"}, produced via \code{\link[=vector_to_string]{vector_to_string()}}) of the probability of methylation/modification at each assessed base.\cr\cr Assumed to be Nanopore > SAM style modification stored as an 8-bit integer from 0 to 255, but changing other arguments could make this work on other scales.}

\item{sequence_lengths}{\verb{numeric vector}. The length of each sequence.}

\item{low_colour}{\code{character}. The colour that should be used to represent minimum probability of methylation/modification (defaults to blue).}

\item{high_colour}{\code{character}. The colour that should be used to represent maximum probability of methylation/modification (defaults to red).}

\item{low_clamp}{\code{numeric}. The minimum probability below which all values are coloured \code{low_colour}. Defaults to \code{0} (i.e. no clamping). To specify a proportion probability in 8-bit form, multiply by 255 e.g. to low-clamp at 30\% probability, set this to \code{0.3*255}.}

\item{high_clamp}{\code{numeric}. The maximum probability above which all values are coloured \code{high_colour}. Defaults to \code{255} (i.e. no clamping, assuming Nanopore > SAM style modification calling where probabilities are 8-bit integers from 0 to 255).}

\item{background_colour}{\code{character}. The colour the background should be drawn (defaults to white).}

\item{other_bases_colour}{\code{character}. The colour non-assessed (e.g. non-CpG) bases should be drawn (defaults to grey).}

\item{outline_colour}{\code{character}. The colour of the box outlines. Defaults to black.}

\item{outline_linewidth}{\code{numeric}. The linewidth of the box outlines. Defaults to \code{3}. Set to \code{0} to disable box outlines.}

\item{outline_join}{\code{character}. One of \code{"mitre"}, \code{"round"}, or \code{"bevel"} specifying how outlines should be joined at the corners of boxes. Defaults to \code{"mitre"}. It would be unusual to need to change this.}

\item{modified_bases_outline_colour}{\code{character}. If \code{NA} (default), inherits from \code{outline_colour}. If not \code{NA}, overrides \code{outline_colour} for modification-assessed bases only.}

\item{modified_bases_outline_linewidth}{\code{numeric}. If \code{NA} (default), inherits from \code{outline_linewidth}. If not \code{NA}, overrides \code{outline_linewidth} for modification-assessed bases only.}

\item{modified_bases_outline_join}{\code{character}. If \code{NA} (default), inherits from \code{outline_join}. If not \code{NA}, overrides \code{outline_join} for modification-assessed bases only.}

\item{other_bases_outline_colour}{\code{character}. If \code{NA} (default), inherits from \code{outline_colour}. If not \code{NA}, overrides \code{outline_colour} for non-modification-assessed bases only.}

\item{other_bases_outline_linewidth}{\code{numeric}. If \code{NA} (default), inherits from \code{outline_linewidth}. If not \code{NA}, overrides \code{outline_linewidth} for non-modification-assessed bases only.}

\item{other_bases_outline_join}{\code{character}. If \code{NA} (default), inherits from \code{outline_join}. If not \code{NA}, overrides \code{outline_join} for non-modification-assessed bases only.}

\item{margin}{\code{numeric}. The size of the margin relative to the size of each base square. Defaults to \code{0.5} (half the side length of each base square).}

\item{return}{\code{logical}. Boolean specifying whether this function should return the ggplot object, otherwise it will return \code{invisible(NULL)}. Defaults to \code{TRUE}.}

\item{filename}{\code{character}. Filename to which output should be saved. If set to \code{NA} (default), no file will be saved. Recommended to end with \code{".png"}, but can change if render device is changed.}

\item{render_device}{\verb{function/character}. Device to use when rendering. See \code{\link[ggplot2:ggsave]{ggplot2::ggsave()}} documentation for options. Defaults to \code{\link[ragg:agg_png]{ragg::agg_png}}. Can be set to \code{NULL} to infer from file extension, but results may vary between systems.}

\item{pixels_per_base}{\code{integer}. How large each box should be in pixels, if file output is turned on via setting \code{filename}. Corresponds to dpi of the exported image. Defaults to \code{20}. Low values acceptable as currently this function does not write any text.}
}
\value{
A ggplot object containing the full visualisation, or \code{invisible(NULL)} if \code{return = FALSE}. It is often more useful to use \code{filename = "myfilename.png"}, because then the visualisation is exported at the correct aspect ratio.
}
\description{
This function takes vectors of modifications locations, modification probabilities,
and sequence lengths (e.g. created by \code{\link[=extract_methylation_from_dataframe]{extract_methylation_from_dataframe()}}) and
visualises the probability of methylation (or other modification) across each read.\cr\cr
Assumes that the three main input vectors are of equal length \emph{n} and represent \emph{n} sequences
(e.g. Nanopore reads), where \code{locations} are the indices along each read at which modification
was assessed, \code{probabilities} are the probability of modification at each assessed site, and
\code{lengths} are the lengths of each sequence.\cr\cr
For each sequence, renders non-assessed (e.g. non-CpG) bases as \code{other_bases_colour}, renders
background (including after the end of the sequence) as \code{background_colour}, and renders assessed
bases on a linear scale from \code{low_colour} to \code{high_colour}.\cr\cr
Clamping means that the endpoints of the colour gradient can be set some distance into the probability
space e.g. with Nanopore > SAM probability values from 0-255, the default is to render 0 as fully blue
(\verb{#0000FF}), 255 as fully red (\verb{#FF0000}), and values in between linearly interpolated. However, clamping with
\code{low_clamp = 100} and \code{high_clamp = 200} would set \emph{all probabilities up to 100} as fully blue,
\emph{all probabilities 200 and above} as fully red, and linearly interpolate only over the \code{100-200} range.\cr\cr
A separate scalebar plot showing the colours corresponding to each probability, with any/no clamping values,
can be produced via \code{\link[=visualise_methylation_colour_scale]{visualise_methylation_colour_scale()}}.
}
\examples{
\donttest{
## Extract info from dataframe
methylation_info <- extract_methylation_from_dataframe(example_many_sequences)

## Visualise example_many_sequences with all defaults
## This looks ugly because it isn't at the right scale/aspect ratio
visualise_methylation(
    methylation_info$locations,
    methylation_info$probabilities,
    methylation_info$lengths
)

## Export with all defaults rather than returning
visualise_methylation(
    methylation_info$locations,
    methylation_info$probabilities,
    methylation_info$lengths,
    filename = "example_vm_01.png",
    return = FALSE
)
## View exported image
image <- png::readPNG("example_vm_01.png")
unlink("example_vm_01.png")
grid::grid.newpage()
grid::grid.raster(image)

## Export with customisation
visualise_methylation(
    methylation_info$locations,
    methylation_info$probabilities,
    methylation_info$lengths,
    filename = "example_vm_02.png",
    return = FALSE,
    low_colour = "white",
    high_colour = "black",
    low_clamp = 0.3*255,
    high_clamp = 0.7*255,
    other_bases_colour = "lightblue1",
    other_bases_outline_linewidth = 1,
    other_bases_outline_colour = "grey",
    modified_bases_outline_linewidth = 3,
    modified_bases_outline_colour = "black",
    margin = 0.3
)
## View exported image
image <- png::readPNG("example_vm_02.png")
unlink("example_vm_02.png")
grid::grid.newpage()
grid::grid.raster(image)
}

}
