/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the COPYING file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
#ifndef H5Dpublic_H
#define H5Dpublic_H
#include "H5public.h"
#include "H5Ipublic.h"
#define H5D_CHUNK_CACHE_NSLOTS_DEFAULT SIZE_MAX
#define H5D_CHUNK_CACHE_NBYTES_DEFAULT SIZE_MAX
#define H5D_CHUNK_CACHE_W0_DEFAULT     (-1.0)
#define H5D_CHUNK_DONT_FILTER_PARTIAL_CHUNKS (0x0002u)
//! <!-- [H5D_layout_t_snip] -->
typedef enum H5D_layout_t {
    H5D_LAYOUT_ERROR = -1,
    H5D_COMPACT      = 0,
    H5D_CONTIGUOUS   = 1,
    H5D_CHUNKED      = 2,
    H5D_VIRTUAL      = 3,
    H5D_NLAYOUTS     = 4
} H5D_layout_t;
//! <!-- [H5D_layout_t_snip] -->
//! <!-- [H5D_chunk_index_t_snip] -->
typedef enum H5D_chunk_index_t {
    H5D_CHUNK_IDX_BTREE = 0,
    H5D_CHUNK_IDX_SINGLE =
        1,
    H5D_CHUNK_IDX_NONE   = 2,
    H5D_CHUNK_IDX_FARRAY = 3,
    H5D_CHUNK_IDX_EARRAY = 4,
    H5D_CHUNK_IDX_BT2    = 5,
    H5D_CHUNK_IDX_NTYPES
} H5D_chunk_index_t;
//! <!-- [H5D_chunk_index_t_snip] -->
//! <!-- [H5D_alloc_time_t_snip] -->
typedef enum H5D_alloc_time_t {
    H5D_ALLOC_TIME_ERROR   = -1,
    H5D_ALLOC_TIME_DEFAULT = 0,
    H5D_ALLOC_TIME_EARLY   = 1,
    H5D_ALLOC_TIME_LATE    = 2,
    H5D_ALLOC_TIME_INCR    = 3
} H5D_alloc_time_t;
//! <!-- [H5D_alloc_time_t_snip] -->
//! <!-- [H5D_space_status_t_snip] -->
typedef enum H5D_space_status_t {
    H5D_SPACE_STATUS_ERROR          = -1,
    H5D_SPACE_STATUS_NOT_ALLOCATED  = 0,
    H5D_SPACE_STATUS_PART_ALLOCATED = 1,
    H5D_SPACE_STATUS_ALLOCATED = 2
} H5D_space_status_t;
//! <!-- [H5D_space_status_t_snip] -->
//! <!-- [H5D_fill_time_t_snip] -->
typedef enum H5D_fill_time_t {
    H5D_FILL_TIME_ERROR = -1,
    H5D_FILL_TIME_ALLOC = 0,
    H5D_FILL_TIME_NEVER = 1,
    H5D_FILL_TIME_IFSET = 2
} H5D_fill_time_t;
//! <!-- [H5D_fill_time_t_snip] -->
//! <!-- [H5D_fill_value_t_snip] -->
typedef enum H5D_fill_value_t {
    H5D_FILL_VALUE_ERROR        = -1,
    H5D_FILL_VALUE_UNDEFINED    = 0,
    H5D_FILL_VALUE_DEFAULT      = 1,
    H5D_FILL_VALUE_USER_DEFINED = 2
} H5D_fill_value_t;
//! <!-- [H5D_fill_value_t_snip] -->
//! <!-- [H5D_vds_view_t_snip] -->
typedef enum H5D_vds_view_t {
    H5D_VDS_ERROR          = -1,
    H5D_VDS_FIRST_MISSING  = 0,
    H5D_VDS_LAST_AVAILABLE = 1
} H5D_vds_view_t;
//! <!-- [H5D_vds_view_t_snip] -->
//! <!-- [H5D_append_cb_t_snip] -->
typedef herr_t (*H5D_append_cb_t)(hid_t dataset_id, hsize_t *cur_dims, void *op_data);
//! <!-- [H5D_append_cb_t_snip] -->
//! <!-- [H5D_operator_t_snip] -->
typedef herr_t (*H5D_operator_t)(void *elem, hid_t type_id, unsigned ndim, const hsize_t *point,
                                 void *operator_data);
//! <!-- [H5D_operator_t_snip] -->
//! <!-- [H5D_scatter_func_t_snip] -->
typedef herr_t (*H5D_scatter_func_t)(const void **src_buf, size_t *src_buf_bytes_used,
                                     void *op_data);
//! <!-- [H5D_scatter_func_t_snip] -->
//! <!-- [H5D_gather_func_t_snip] -->
typedef herr_t (*H5D_gather_func_t)(const void *dst_buf, size_t dst_buf_bytes_used, void *op_data);
//! <!-- [H5D_gather_func_t_snip] -->
//! <!-- [H5D_chunk_iter_op_t_snip] -->
typedef int (*H5D_chunk_iter_op_t)(const hsize_t *offset, unsigned filter_mask, haddr_t addr, hsize_t size,
                                   void *op_data);
//! <!-- [H5D_chunk_iter_op_t_snip] -->
#ifdef __cplusplus
extern "C" {
#endif
H5_DLL hid_t H5Dcreate2(hid_t loc_id, const char *name, hid_t type_id, hid_t space_id, hid_t lcpl_id,
                        hid_t dcpl_id, hid_t dapl_id);
#ifndef H5_DOXYGEN
H5_DLL hid_t H5Dcreate_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                             const char *name, hid_t type_id, hid_t space_id, hid_t lcpl_id, hid_t dcpl_id,
                             hid_t dapl_id, hid_t es_id);
#else
H5_DLL hid_t H5Dcreate_async(hid_t loc_id, const char *name, hid_t type_id, hid_t space_id, hid_t lcpl_id,
                             hid_t dcpl_id, hid_t dapl_id, hid_t es_id);
#endif
H5_DLL hid_t H5Dcreate_anon(hid_t loc_id, hid_t type_id, hid_t space_id, hid_t dcpl_id, hid_t dapl_id);
H5_DLL hid_t H5Dopen2(hid_t loc_id, const char *name, hid_t dapl_id);
#ifndef H5_DOXYGEN
H5_DLL hid_t H5Dopen_async(const char *app_file, const char *app_func, unsigned app_line, hid_t loc_id,
                           const char *name, hid_t dapl_id, hid_t es_id);
#else
H5_DLL hid_t H5Dopen_async(hid_t loc_id, const char *name, hid_t dapl_id, hid_t es_id);
#endif
H5_DLL hid_t H5Dget_space(hid_t dset_id);
#ifndef H5_DOXYGEN
H5_DLL hid_t H5Dget_space_async(const char *app_file, const char *app_func, unsigned app_line, hid_t dset_id,
                                hid_t es_id);
#else
H5_DLL hid_t H5Dget_space_async(hid_t dset_id, hid_t es_id);
#endif
H5_DLL herr_t H5Dget_space_status(hid_t dset_id, H5D_space_status_t *allocation);
H5_DLL hid_t H5Dget_type(hid_t dset_id);
H5_DLL hid_t H5Dget_create_plist(hid_t dset_id);
H5_DLL hid_t H5Dget_access_plist(hid_t dset_id);
H5_DLL hsize_t H5Dget_storage_size(hid_t dset_id);
H5_DLL herr_t H5Dget_chunk_storage_size(hid_t dset_id, const hsize_t *offset, hsize_t *chunk_bytes);
H5_DLL herr_t H5Dget_num_chunks(hid_t dset_id, hid_t fspace_id, hsize_t *nchunks);
H5_DLL herr_t H5Dget_chunk_info_by_coord(hid_t dset_id, const hsize_t *offset, unsigned *filter_mask,
                                         haddr_t *addr, hsize_t *size);
H5_DLL herr_t H5Dchunk_iter(hid_t dset_id, hid_t dxpl_id, H5D_chunk_iter_op_t cb, void *op_data);
H5_DLL herr_t H5Dget_chunk_info(hid_t dset_id, hid_t fspace_id, hsize_t chk_idx, hsize_t *offset,
                                unsigned *filter_mask, haddr_t *addr, hsize_t *size);
H5_DLL haddr_t H5Dget_offset(hid_t dset_id);
H5_DLL herr_t H5Dread(hid_t dset_id, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id,
                      hid_t dxpl_id, void *buf);
H5_DLL herr_t H5Dread_multi(size_t count, hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                            hid_t file_space_id[], hid_t dxpl_id, void *buf[]);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dread_async(const char *app_file, const char *app_func, unsigned app_line, hid_t dset_id,
                            hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t dxpl_id,
                            void *buf, hid_t es_id);
#else
H5_DLL herr_t H5Dread_async(hid_t dset_id, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id,
                            hid_t dxpl_id, void *buf, hid_t es_id);
#endif
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dread_multi_async(const char *app_file, const char *app_func, unsigned app_line, size_t count,
                                  hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                                  hid_t file_space_id[], hid_t dxpl_id, void *buf[], hid_t es_id);
#else
H5_DLL herr_t H5Dread_multi_async(size_t count, hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                                  hid_t file_space_id[], hid_t dxpl_id, void *buf[], hid_t es_id);
#endif
H5_DLL herr_t H5Dwrite(hid_t dset_id, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id,
                       hid_t dxpl_id, const void *buf);
H5_DLL herr_t H5Dwrite_multi(size_t count, hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                             hid_t file_space_id[], hid_t dxpl_id, const void *buf[]);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dwrite_async(const char *app_file, const char *app_func, unsigned app_line, hid_t dset_id,
                             hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id, hid_t dxpl_id,
                             const void *buf, hid_t es_id);
#else
H5_DLL herr_t H5Dwrite_async(hid_t dset_id, hid_t mem_type_id, hid_t mem_space_id, hid_t file_space_id,
                             hid_t dxpl_id, const void *buf, hid_t es_id);
#endif
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dwrite_multi_async(const char *app_file, const char *app_func, unsigned app_line,
                                   size_t count, hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                                   hid_t file_space_id[], hid_t dxpl_id, const void *buf[], hid_t es_id);
#else
H5_DLL herr_t H5Dwrite_multi_async(size_t count, hid_t dset_id[], hid_t mem_type_id[], hid_t mem_space_id[],
                                   hid_t file_space_id[], hid_t dxpl_id, const void *buf[], hid_t es_id);
#endif
H5_DLL herr_t H5Dwrite_chunk(hid_t dset_id, hid_t dxpl_id, uint32_t filters, const hsize_t *offset,
                             size_t data_size, const void *buf);
H5_DLL herr_t H5Dread_chunk(hid_t dset_id, hid_t dxpl_id, const hsize_t *offset, uint32_t *filters,
                            void *buf);
H5_DLL herr_t H5Diterate(void *buf, hid_t type_id, hid_t space_id, H5D_operator_t op, void *operator_data);
H5_DLL herr_t H5Dvlen_get_buf_size(hid_t dset_id, hid_t type_id, hid_t space_id, hsize_t *size);
H5_DLL herr_t H5Dfill(const void *fill, hid_t fill_type_id, void *buf, hid_t buf_type_id, hid_t space_id);
H5_DLL herr_t H5Dset_extent(hid_t dset_id, const hsize_t size[]);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dset_extent_async(const char *app_file, const char *app_func, unsigned app_line,
                                  hid_t dset_id, const hsize_t size[], hid_t es_id);
#else
H5_DLL herr_t H5Dset_extent_async(hid_t dset_id, const hsize_t size[], hid_t es_id);
#endif
H5_DLL herr_t H5Dflush(hid_t dset_id);
H5_DLL herr_t H5Drefresh(hid_t dset_id);
H5_DLL herr_t H5Dscatter(H5D_scatter_func_t op, void *op_data, hid_t type_id, hid_t dst_space_id,
                         void *dst_buf);
H5_DLL herr_t H5Dgather(hid_t src_space_id, const void *src_buf, hid_t type_id, size_t dst_buf_size,
                        void *dst_buf, H5D_gather_func_t op, void *op_data);
H5_DLL herr_t H5Dclose(hid_t dset_id);
#ifndef H5_DOXYGEN
H5_DLL herr_t H5Dclose_async(const char *app_file, const char *app_func, unsigned app_line, hid_t dset_id,
                             hid_t es_id);
#else
H5_DLL herr_t H5Dclose_async(hid_t dset_id, hid_t es_id);
#endif
/// \cond DEV
H5_DLL herr_t H5Ddebug(hid_t dset_id);
H5_DLL herr_t H5Dformat_convert(hid_t dset_id);
H5_DLL herr_t H5Dget_chunk_index_type(hid_t did, H5D_chunk_index_t *idx_type);
/// \endcond
/// \cond DEV
#ifndef H5D_MODULE
#define H5Dcreate_async(...)      H5Dcreate_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dopen_async(...)        H5Dopen_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dget_space_async(...)   H5Dget_space_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dread_async(...)        H5Dread_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dread_multi_async(...)  H5Dread_multi_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dwrite_async(...)       H5Dwrite_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dwrite_multi_async(...) H5Dwrite_multi_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dset_extent_async(...)  H5Dset_extent_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dclose_async(...)       H5Dclose_async(__FILE__, __func__, __LINE__, __VA_ARGS__)
#define H5Dcreate_async_wrap     H5_NO_EXPAND(H5Dcreate_async)
#define H5Dopen_async_wrap       H5_NO_EXPAND(H5Dopen_async)
#define H5Dget_space_async_wrap  H5_NO_EXPAND(H5Dget_space_async)
#define H5Dread_async_wrap       H5_NO_EXPAND(H5Dread_async)
#define H5Dwrite_async_wrap      H5_NO_EXPAND(H5Dwrite_async)
#define H5Dset_extent_async_wrap H5_NO_EXPAND(H5Dset_extent_async)
#define H5Dclose_async_wrap      H5_NO_EXPAND(H5Dclose_async)
#endif
/// \endcond
#ifndef H5_NO_DEPRECATED_SYMBOLS
#define H5D_CHUNK_BTREE H5D_CHUNK_IDX_BTREE
#define H5D_XFER_DIRECT_CHUNK_WRITE_FLAG_NAME     "direct_chunk_flag"
#define H5D_XFER_DIRECT_CHUNK_WRITE_FILTERS_NAME  "direct_chunk_filters"
#define H5D_XFER_DIRECT_CHUNK_WRITE_OFFSET_NAME   "direct_chunk_offset"
#define H5D_XFER_DIRECT_CHUNK_WRITE_DATASIZE_NAME "direct_chunk_datasize"
#define H5D_XFER_DIRECT_CHUNK_READ_FLAG_NAME    "direct_chunk_read_flag"
#define H5D_XFER_DIRECT_CHUNK_READ_OFFSET_NAME  "direct_chunk_read_offset"
#define H5D_XFER_DIRECT_CHUNK_READ_FILTERS_NAME "direct_chunk_read_filters"
H5_DLL hid_t H5Dcreate1(hid_t loc_id, const char *name, hid_t type_id, hid_t space_id, hid_t dcpl_id);
H5_DLL hid_t H5Dopen1(hid_t loc_id, const char *name);
H5_DLL herr_t H5Dextend(hid_t dset_id, const hsize_t size[]);
H5_DLL herr_t H5Dvlen_reclaim(hid_t type_id, hid_t space_id, hid_t dxpl_id, void *buf);
#endif
#ifdef __cplusplus
}
#endif
#endif
