/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Tpkg.h"      

static H5T_t *H5T__get_native_type(H5T_t *dt, H5T_direction_t direction, size_t *struct_align, size_t *offset,
                                   size_t *comp_size);
static H5T_t *H5T__get_native_integer(size_t prec, H5T_sign_t sign, H5T_direction_t direction,
                                      size_t *struct_align, size_t *offset, size_t *comp_size);
static H5T_t *H5T__get_native_float(const H5T_t *dtype, H5T_direction_t direction, size_t *struct_align,
                                    size_t *offset, size_t *comp_size);
static H5T_t *H5T__get_native_bitfield(size_t prec, H5T_direction_t direction, size_t *struct_align,
                                       size_t *offset, size_t *comp_size);
static herr_t H5T__cmp_offset(size_t *comp_size, size_t *offset, size_t elem_size, size_t nelems,
                              size_t align, size_t *struct_align);

hid_t
H5Tget_native_type(hid_t type_id, H5T_direction_t direction)
{
    H5T_t *dt;               
    H5T_t *new_dt    = NULL; 
    size_t comp_size = 0;    
    hid_t  ret_value;        

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a data type");
    if (direction != H5T_DIR_DEFAULT && direction != H5T_DIR_ASCEND && direction != H5T_DIR_DESCEND)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not valid direction value");

    
    if (NULL == (new_dt = H5T__get_native_type(dt, direction, NULL, NULL, &comp_size)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "cannot retrieve native type");

    
    if ((ret_value = H5I_register(H5I_DATATYPE, new_dt, true)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register data type");

done:
    
    if (ret_value < 0)
        if (new_dt && H5T_close_real(new_dt) < 0)
            HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, H5I_INVALID_HID, "unable to release datatype");

    FUNC_LEAVE_API(ret_value)
} 

static H5T_t *
H5T__get_native_type(H5T_t *dtype, H5T_direction_t direction, size_t *struct_align, size_t *offset,
                     size_t *comp_size)
{
    H5T_t  *super_type;       
    H5T_t  *nat_super_type;   
    H5T_t  *new_type  = NULL; 
    H5T_t  *memb_type = NULL; 
    H5T_t **memb_list = NULL; 
    size_t *memb_offset =
        NULL; 
    char      **comp_mname     = NULL; 
    char       *memb_name      = NULL; 
    void       *memb_value     = NULL; 
    void       *tmp_memb_value = NULL; 
    hsize_t    *dims           = NULL; 
    H5T_class_t h5_class;              
    size_t      size;                  
    size_t      prec;                  
    int         snmemb;                
    unsigned    nmemb = 0;             
    unsigned    u;                     
    H5T_t      *ret_value = NULL;      

    FUNC_ENTER_PACKAGE

    assert(dtype);

    if (H5T_NO_CLASS == (h5_class = H5T_get_class(dtype, false)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a valid class");

    if (0 == (size = H5T_get_size(dtype)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a valid size");

    switch (h5_class) {
        case H5T_INTEGER: {
            H5T_sign_t sign; 

            if (H5T_SGN_ERROR == (sign = H5T_get_sign(dtype)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a valid signess");

            prec = dtype->shared->u.atomic.prec;

            if (NULL ==
                (ret_value = H5T__get_native_integer(prec, sign, direction, struct_align, offset, comp_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve integer type");
        } 
        break;

        case H5T_FLOAT:
            if (NULL ==
                (ret_value = H5T__get_native_float(dtype, direction, struct_align, offset, comp_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve float type");

            break;

        case H5T_STRING:
            if (NULL == (ret_value = H5T_copy(dtype, H5T_COPY_TRANSIENT)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve float type");

            if (H5T_IS_VL_STRING(dtype->shared)) {
                
                if (H5T__cmp_offset(comp_size, offset, sizeof(char *), (size_t)1, H5T_POINTER_ALIGN_g,
                                    struct_align) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");
            } 
            else {
                
                if (H5T__cmp_offset(comp_size, offset, sizeof(char), size, H5T_NATIVE_SCHAR_ALIGN_g,
                                    struct_align) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");
            } 
            break;

        
        case H5T_TIME:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "time type is not supported yet");

        case H5T_BITFIELD: {
            prec = dtype->shared->u.atomic.prec;

            if (NULL ==
                (ret_value = H5T__get_native_bitfield(prec, direction, struct_align, offset, comp_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve integer for bitfield type");
        } 
        break;

        case H5T_OPAQUE:
            if (NULL == (ret_value = H5T_copy(dtype, H5T_COPY_TRANSIENT)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve float type");

            
            if (H5T__cmp_offset(comp_size, offset, sizeof(char), size, H5T_NATIVE_SCHAR_ALIGN_g,
                                struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");
            break;

        case H5T_REFERENCE: {
            H5T_t *dt; 
            size_t align;
            size_t ref_size;

            if (NULL == (ret_value = H5T_copy(dtype, H5T_COPY_TRANSIENT)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot copy reference type");

            
            if (NULL == (dt = (H5T_t *)H5I_object(H5T_STD_REF_OBJ_g)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

            
            if (0 == H5T_cmp(ret_value, dt, false)) {
                align    = H5T_HOBJREF_ALIGN_g;
                ref_size = sizeof(hobj_ref_t);
            } 
            else {
                
                if (NULL == (dt = (H5T_t *)H5I_object(H5T_STD_REF_DSETREG_g)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

                if (0 == H5T_cmp(ret_value, dt, false)) {
                    align    = H5T_HDSETREGREF_ALIGN_g;
                    ref_size = sizeof(hdset_reg_ref_t);
                } 
                else {
                    
                    align    = H5T_REF_ALIGN_g;
                    ref_size = sizeof(H5R_ref_t);
                } 
            }     

            if (H5T__cmp_offset(comp_size, offset, ref_size, (size_t)1, align, struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");
        } 
        break;

        case H5T_COMPOUND: {
            size_t children_size = 0; 
            size_t children_st_align =
                0; 

            if ((snmemb = H5T_get_nmembers(dtype)) <= 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "compound data type doesn't have any member");
            H5_CHECKED_ASSIGN(nmemb, unsigned, snmemb, int);

            if (NULL == (memb_list = (H5T_t **)H5MM_calloc(nmemb * sizeof(H5T_t *))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");
            if (NULL == (memb_offset = (size_t *)H5MM_calloc(nmemb * sizeof(size_t))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");
            if (NULL == (comp_mname = (char **)H5MM_calloc(nmemb * sizeof(char *))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");

            
            for (u = 0; u < nmemb; u++) {
                if (NULL == (memb_type = H5T_get_member_type(dtype, u)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "member type retrieval failed");

                if (NULL == (comp_mname[u] = H5T__get_member_name(dtype, u)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "member type retrieval failed");

                if (NULL == (memb_list[u] = H5T__get_native_type(memb_type, direction, &children_st_align,
                                                                 &(memb_offset[u]), &children_size)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "member identifier retrieval failed");

                if (H5T_close_real(memb_type) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot close datatype");
            } 

            
            if (children_st_align && children_size % children_st_align)
                children_size += children_st_align - (children_size % children_st_align);

            
            if (NULL == (new_type = H5T__create(H5T_COMPOUND, children_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot create a compound type");

            
            for (u = 0; u < nmemb; u++)
                if (H5T__insert(new_type, comp_mname[u], memb_offset[u], memb_list[u]) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot insert member to compound datatype");

            
            if (H5T__cmp_offset(comp_size, offset, children_size, (size_t)1, children_st_align,
                                struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

            
            for (u = 0; u < nmemb; u++) {
                if (H5T_close_real(memb_list[u]) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot close datatype");

                
                comp_mname[u] = (char *)H5MM_xfree(comp_mname[u]);
            } 

            
            memb_list   = (H5T_t **)H5MM_xfree(memb_list);
            memb_offset = (size_t *)H5MM_xfree(memb_offset);
            comp_mname  = (char **)H5MM_xfree(comp_mname);

            ret_value = new_type;
        } 
        break;

        case H5T_ENUM: {
            H5T_path_t *tpath; 

            

            
            if (NULL == (super_type = H5T_get_super(dtype)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to get base type for enumerate type");
            if (NULL == (nat_super_type =
                             H5T__get_native_type(super_type, direction, struct_align, offset, comp_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "base native type retrieval failed");

            
            if (NULL == (tmp_memb_value = H5MM_calloc(H5T_get_size(super_type))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");
            if (NULL == (memb_value = H5MM_calloc(H5T_get_size(nat_super_type))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");

            
            if (NULL == (new_type = H5T__enum_create(nat_super_type)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to create enum type");

            
            if (NULL == (tpath = H5T_path_find(super_type, nat_super_type)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL,
                            "unable to convert between src and dst data types");

            
            if ((snmemb = H5T_get_nmembers(dtype)) <= 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "enumerate data type doesn't have any member");
            H5_CHECKED_ASSIGN(nmemb, unsigned, snmemb, int);
            for (u = 0; u < nmemb; u++) {
                if (NULL == (memb_name = H5T__get_member_name(dtype, u)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot get member name");
                if (H5T__get_member_value(dtype, u, tmp_memb_value) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot get member value");
                H5MM_memcpy(memb_value, tmp_memb_value, H5T_get_size(super_type));

                if (H5T_convert(tpath, super_type, nat_super_type, (size_t)1, (size_t)0, (size_t)0,
                                memb_value, NULL) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot get member value");

                if (H5T__enum_insert(new_type, memb_name, memb_value) < 0)
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot insert member");
                memb_name = (char *)H5MM_xfree(memb_name);
            }
            memb_value     = H5MM_xfree(memb_value);
            tmp_memb_value = H5MM_xfree(tmp_memb_value);

            if (H5T_close(nat_super_type) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "can't close datatype");
            if (H5T_close(super_type) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, NULL, "can't close datatype");

            ret_value = new_type;
        } 
        break;

        case H5T_ARRAY: {
            int      sarray_rank; 
            unsigned array_rank;  
            hsize_t  nelems       = 1;
            size_t   super_offset = 0;
            size_t   super_size   = 0;
            size_t   super_align  = 0;

            
            if ((sarray_rank = H5T__get_array_ndims(dtype)) <= 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot get dimension rank");
            H5_CHECKED_ASSIGN(array_rank, unsigned, sarray_rank, int);
            if (NULL == (dims = (hsize_t *)H5MM_malloc(array_rank * sizeof(hsize_t))))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot allocate memory");
            if (H5T__get_array_dims(dtype, dims) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot get dimension size");

            
            if (NULL == (super_type = H5T_get_super(dtype)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to get parent type for array type");
            if (NULL == (nat_super_type = H5T__get_native_type(super_type, direction, &super_align,
                                                               &super_offset, &super_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "parent native type retrieval failed");

            
            if (H5T_close_real(super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            
            if (NULL == (new_type = H5T__array_create(nat_super_type, array_rank, dims)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to create array type");

            
            if (H5T_close_real(nat_super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            for (u = 0; u < array_rank; u++)
                nelems *= dims[u];
            H5_CHECK_OVERFLOW(nelems, hsize_t, size_t);
            if (H5T__cmp_offset(comp_size, offset, super_size, (size_t)nelems, super_align, struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

            dims = (hsize_t *)H5MM_xfree(dims);

            ret_value = new_type;
        } 
        break;

        case H5T_VLEN: {
            size_t vl_align   = 0;
            size_t vl_size    = 0;
            size_t super_size = 0;

            
            if (NULL == (super_type = H5T_get_super(dtype)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to get parent type for VL type");
            
            if (NULL ==
                (nat_super_type = H5T__get_native_type(super_type, direction, NULL, NULL, &super_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "parent native type retrieval failed");

            
            if (H5T_close_real(super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            
            if (NULL == (new_type = H5T__vlen_create(nat_super_type)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to create VL type");

            
            if (H5T_close_real(nat_super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            
            vl_align = H5T_HVL_ALIGN_g;
            vl_size  = sizeof(hvl_t);

            if (H5T__cmp_offset(comp_size, offset, vl_size, (size_t)1, vl_align, struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

            ret_value = new_type;
        } 
        break;

        case H5T_COMPLEX: {
            size_t super_offset = 0;
            size_t super_size   = 0;
            size_t super_align  = 0;

            
            if (NULL == (super_type = H5T_get_super(dtype)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to get parent type for complex number type");

            if (NULL == (nat_super_type = H5T__get_native_type(super_type, direction, &super_align,
                                                               &super_offset, &super_size)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "parent native type retrieval failed");

            
            if (H5T_close_real(super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            
            if (NULL == (new_type = H5T__complex_create(nat_super_type)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "unable to create complex number type");

            
            if (H5T_close_real(nat_super_type) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_CLOSEERROR, NULL, "cannot close datatype");

            if (H5T__cmp_offset(comp_size, offset, new_type->shared->size, 1, super_align, struct_align) < 0)
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

            ret_value = new_type;
        } 
        break;

        case H5T_NO_CLASS:
        case H5T_NCLASSES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "data type doesn't match any native type");
    } 

done:
    
    if (NULL == ret_value) {
        if (new_type)
            if (H5T_close_real(new_type) < 0)
                HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, NULL, "unable to release datatype");

        
        if (memb_list) {
            for (u = 0; u < nmemb; u++)
                if (memb_list[u] && H5T_close_real(memb_list[u]) < 0)
                    HDONE_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot close datatype");

            memb_list = (H5T_t **)H5MM_xfree(memb_list);
        } 
        memb_offset = (size_t *)H5MM_xfree(memb_offset);
        if (comp_mname) {
            for (u = 0; u < nmemb; u++)
                if (comp_mname[u])
                    H5MM_xfree(comp_mname[u]);
            comp_mname = (char **)H5MM_xfree(comp_mname);
        } 
        memb_name      = (char *)H5MM_xfree(memb_name);
        memb_value     = H5MM_xfree(memb_value);
        tmp_memb_value = H5MM_xfree(tmp_memb_value);
        dims           = (hsize_t *)H5MM_xfree(dims);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__get_native_integer(size_t prec, H5T_sign_t sign, H5T_direction_t direction, size_t *struct_align,
                        size_t *offset, size_t *comp_size)
{
    H5T_t *dt;                 
    hid_t  tid         = (-1); 
    size_t align       = 0;    
    size_t native_size = 0;    
    enum match_type {          
                      H5T_NATIVE_INT_MATCH_CHAR,
                      H5T_NATIVE_INT_MATCH_SHORT,
                      H5T_NATIVE_INT_MATCH_INT,
                      H5T_NATIVE_INT_MATCH_LONG,
                      H5T_NATIVE_INT_MATCH_LLONG,
                      H5T_NATIVE_INT_MATCH_UNKNOWN
    } match          = H5T_NATIVE_INT_MATCH_UNKNOWN;
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    H5_WARN_DUPLICATED_BRANCHES_OFF
    if (direction == H5T_DIR_DEFAULT || direction == H5T_DIR_ASCEND) {
        if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_SCHAR_g))) {
            match       = H5T_NATIVE_INT_MATCH_CHAR;
            native_size = sizeof(char);
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_SHORT_g))) {
            match       = H5T_NATIVE_INT_MATCH_SHORT;
            native_size = sizeof(short);
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_INT_g))) {
            match       = H5T_NATIVE_INT_MATCH_INT;
            native_size = sizeof(int);
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_LONG_g))) {
            match       = H5T_NATIVE_INT_MATCH_LONG;
            native_size = sizeof(long);
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_LLONG_g))) {
            match       = H5T_NATIVE_INT_MATCH_LLONG;
            native_size = sizeof(long long);
        }
        else { 
            match       = H5T_NATIVE_INT_MATCH_LLONG;
            native_size = sizeof(long long);
        }
    }
    else if (direction == H5T_DIR_DESCEND) {
        if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_LONG_g))) {
            match       = H5T_NATIVE_INT_MATCH_LLONG;
            native_size = sizeof(long long);
        }
        else if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_INT_g))) {
            match       = H5T_NATIVE_INT_MATCH_LONG;
            native_size = sizeof(long);
        }
        else if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_SHORT_g))) {
            match       = H5T_NATIVE_INT_MATCH_INT;
            native_size = sizeof(int);
        }
        else if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_SCHAR_g))) {
            match       = H5T_NATIVE_INT_MATCH_SHORT;
            native_size = sizeof(short);
        }
        else {
            match       = H5T_NATIVE_INT_MATCH_CHAR;
            native_size = sizeof(char);
        }
    }
    H5_WARN_DUPLICATED_BRANCHES_ON

    
    switch (match) {
        case H5T_NATIVE_INT_MATCH_CHAR:
            if (sign == H5T_SGN_2)
                tid = H5T_NATIVE_SCHAR;
            else
                tid = H5T_NATIVE_UCHAR;

            align = H5T_NATIVE_SCHAR_ALIGN_g;
            break;

        case H5T_NATIVE_INT_MATCH_SHORT:
            if (sign == H5T_SGN_2)
                tid = H5T_NATIVE_SHORT;
            else
                tid = H5T_NATIVE_USHORT;
            align = H5T_NATIVE_SHORT_ALIGN_g;
            break;

        case H5T_NATIVE_INT_MATCH_INT:
            if (sign == H5T_SGN_2)
                tid = H5T_NATIVE_INT;
            else
                tid = H5T_NATIVE_UINT;

            align = H5T_NATIVE_INT_ALIGN_g;
            break;

        case H5T_NATIVE_INT_MATCH_LONG:
            if (sign == H5T_SGN_2)
                tid = H5T_NATIVE_LONG;
            else
                tid = H5T_NATIVE_ULONG;

            align = H5T_NATIVE_LONG_ALIGN_g;
            break;

        case H5T_NATIVE_INT_MATCH_LLONG:
            if (sign == H5T_SGN_2)
                tid = H5T_NATIVE_LLONG;
            else
                tid = H5T_NATIVE_ULLONG;

            align = H5T_NATIVE_LLONG_ALIGN_g;
            break;

        case H5T_NATIVE_INT_MATCH_UNKNOWN:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "Unknown native integer match");
    } 

    
    assert(tid >= 0);
    if (NULL == (dt = (H5T_t *)H5I_object(tid)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

    if (NULL == (ret_value = H5T_copy(dt, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot copy type");

    
    if (H5T__cmp_offset(comp_size, offset, native_size, (size_t)1, align, struct_align) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__get_native_float(const H5T_t *dtype, H5T_direction_t direction, size_t *struct_align, size_t *offset,
                      size_t *comp_size)
{
    H5T_t *dt          = NULL; 
    hid_t  tid         = (-1); 
    size_t size        = 0;    
    size_t align       = 0;    
    size_t native_size = 0;    
    enum match_type {          
                      H5T_NATIVE_FLOAT_MATCH_FLOAT16,
                      H5T_NATIVE_FLOAT_MATCH_FLOAT,
                      H5T_NATIVE_FLOAT_MATCH_DOUBLE,
                      H5T_NATIVE_FLOAT_MATCH_LDOUBLE,
                      H5T_NATIVE_FLOAT_MATCH_UNKNOWN
    } match          = H5T_NATIVE_FLOAT_MATCH_UNKNOWN;
    H5T_t *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    if (0 == (size = H5T_get_size(dtype)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a valid size");

    H5_WARN_DUPLICATED_BRANCHES_OFF
    if (direction == H5T_DIR_DEFAULT || direction == H5T_DIR_ASCEND) {
        if (size == 1) {
#ifdef H5_HAVE__FLOAT16
            
            H5T_t *f8_e4m3_dt = NULL; 
            H5T_t *f8_e5m2_dt = NULL; 

            if (NULL == (f8_e4m3_dt = H5I_object(H5T_FLOAT_F8E4M3)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
            if (NULL == (f8_e5m2_dt = H5I_object(H5T_FLOAT_F8E5M2)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

            if (0 == H5T_cmp(dtype, f8_e4m3_dt, false) || 0 == H5T_cmp(dtype, f8_e5m2_dt, false)) {
                match       = H5T_NATIVE_FLOAT_MATCH_FLOAT16;
                native_size = sizeof(H5__Float16);
            }
            else {
                match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
                native_size = sizeof(float);
            }
#else
            
            match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
            native_size = sizeof(float);
#endif
        }
        else if (size == 2) {
#ifdef H5_HAVE__FLOAT16
            
            H5T_t *f16_le_dt = NULL; 
            H5T_t *f16_be_dt = NULL; 

            if (NULL == (f16_le_dt = H5I_object(H5T_IEEE_F16LE)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
            if (NULL == (f16_be_dt = H5I_object(H5T_IEEE_F16BE)))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

            if (0 == H5T_cmp(dtype, f16_le_dt, false) || 0 == H5T_cmp(dtype, f16_be_dt, false)) {
                match       = H5T_NATIVE_FLOAT_MATCH_FLOAT16;
                native_size = sizeof(H5__Float16);
            }
            else {
                match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
                native_size = sizeof(float);
            }
#else
            
            match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
            native_size = sizeof(float);
#endif
        }
        else if (size <= sizeof(float)) {
            match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
            native_size = sizeof(float);
        }
        else if (size <= sizeof(double)) {
            match       = H5T_NATIVE_FLOAT_MATCH_DOUBLE;
            native_size = sizeof(double);
        }
        else if (size <= sizeof(long double)) {
            match       = H5T_NATIVE_FLOAT_MATCH_LDOUBLE;
            native_size = sizeof(long double);
        }
        else { 
            match       = H5T_NATIVE_FLOAT_MATCH_LDOUBLE;
            native_size = sizeof(long double);
        }
    }
    else {
        if (size > sizeof(double)) {
            match       = H5T_NATIVE_FLOAT_MATCH_LDOUBLE;
            native_size = sizeof(long double);
        }
        else if (size > sizeof(float)) {
            match       = H5T_NATIVE_FLOAT_MATCH_DOUBLE;
            native_size = sizeof(double);
        }
        else {
#ifdef H5_HAVE__FLOAT16
            if (size > sizeof(H5__Float16)) {
                match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
                native_size = sizeof(float);
            }
            else if (size > 1) {
                
                H5T_t *f16_le_dt = NULL; 
                H5T_t *f16_be_dt = NULL; 

                if (NULL == (f16_le_dt = H5I_object(H5T_IEEE_F16LE)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
                if (NULL == (f16_be_dt = H5I_object(H5T_IEEE_F16BE)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

                if (0 == H5T_cmp(dtype, f16_le_dt, false) || 0 == H5T_cmp(dtype, f16_be_dt, false)) {
                    match       = H5T_NATIVE_FLOAT_MATCH_FLOAT16;
                    native_size = sizeof(H5__Float16);
                }
                else {
                    match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
                    native_size = sizeof(float);
                }
            }
            else {
                
                H5T_t *f8_e4m3_dt = NULL; 
                H5T_t *f8_e5m2_dt = NULL; 

                if (NULL == (f8_e4m3_dt = H5I_object(H5T_FLOAT_F8E4M3)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
                if (NULL == (f8_e5m2_dt = H5I_object(H5T_FLOAT_F8E5M2)))
                    HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

                if (0 == H5T_cmp(dtype, f8_e4m3_dt, false) || 0 == H5T_cmp(dtype, f8_e5m2_dt, false)) {
                    match       = H5T_NATIVE_FLOAT_MATCH_FLOAT16;
                    native_size = sizeof(H5__Float16);
                }
                else {
                    match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
                    native_size = sizeof(float);
                }
            }
#else
            
            match       = H5T_NATIVE_FLOAT_MATCH_FLOAT;
            native_size = sizeof(float);
#endif
        }
    }
    H5_WARN_DUPLICATED_BRANCHES_ON

    
    switch (match) {
        case H5T_NATIVE_FLOAT_MATCH_FLOAT16:
            tid   = H5T_NATIVE_FLOAT16;
            align = H5T_NATIVE_FLOAT16_ALIGN_g;
            break;

        case H5T_NATIVE_FLOAT_MATCH_FLOAT:
            tid   = H5T_NATIVE_FLOAT;
            align = H5T_NATIVE_FLOAT_ALIGN_g;
            break;

        case H5T_NATIVE_FLOAT_MATCH_DOUBLE:
            tid   = H5T_NATIVE_DOUBLE;
            align = H5T_NATIVE_DOUBLE_ALIGN_g;
            break;

        case H5T_NATIVE_FLOAT_MATCH_LDOUBLE:
            tid   = H5T_NATIVE_LDOUBLE;
            align = H5T_NATIVE_LDOUBLE_ALIGN_g;
            break;

        case H5T_NATIVE_FLOAT_MATCH_UNKNOWN:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "Unknown native floating-point match");
    } 

    
    assert(tid >= 0);
    if (NULL == (dt = (H5T_t *)H5I_object(tid)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");
    if ((ret_value = H5T_copy(dt, H5T_COPY_TRANSIENT)) == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot retrieve float type");

    
    if (H5T__cmp_offset(comp_size, offset, native_size, (size_t)1, align, struct_align) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static H5T_t *
H5T__get_native_bitfield(size_t prec, H5T_direction_t direction, size_t *struct_align, size_t *offset,
                         size_t *comp_size)
{
    H5T_t *dt;                 
    hid_t  tid         = (-1); 
    size_t align       = 0;    
    size_t native_size = 0;    
    H5T_t *ret_value   = NULL; 

    FUNC_ENTER_PACKAGE

    H5_WARN_DUPLICATED_BRANCHES_OFF
    if (direction == H5T_DIR_DEFAULT || direction == H5T_DIR_ASCEND) {
        if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B8_g))) {
            tid         = H5T_NATIVE_B8;
            native_size = 1;
            align       = H5T_NATIVE_UINT8_ALIGN_g;
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B16_g))) {
            tid         = H5T_NATIVE_B16;
            native_size = 2;
            align       = H5T_NATIVE_UINT16_ALIGN_g;
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B32_g))) {
            tid         = H5T_NATIVE_B32;
            native_size = 4;
            align       = H5T_NATIVE_UINT32_ALIGN_g;
        }
        else if (prec <= H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B64_g))) {
            tid         = H5T_NATIVE_B64;
            native_size = 8;
            align       = H5T_NATIVE_UINT64_ALIGN_g;
        }
        else { 
            tid         = H5T_NATIVE_B64;
            native_size = 8;
            align       = H5T_NATIVE_UINT64_ALIGN_g;
        }
    }
    else if (direction == H5T_DIR_DESCEND) {
        if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B32_g))) {
            tid         = H5T_NATIVE_B64;
            native_size = 8;
            align       = H5T_NATIVE_UINT64_ALIGN_g;
        }
        else if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B16_g))) {
            tid         = H5T_NATIVE_B32;
            native_size = 4;
            align       = H5T_NATIVE_UINT32_ALIGN_g;
        }
        else if (prec > H5T_get_precision((H5T_t *)H5I_object(H5T_NATIVE_B8_g))) {
            tid         = H5T_NATIVE_B16;
            native_size = 2;
            align       = H5T_NATIVE_UINT16_ALIGN_g;
        }
        else {
            tid         = H5T_NATIVE_B8;
            native_size = 1;
            align       = H5T_NATIVE_UINT8_ALIGN_g;
        }
    }
    H5_WARN_DUPLICATED_BRANCHES_ON

    
    assert(tid >= 0);
    if (NULL == (dt = (H5T_t *)H5I_object(tid)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a data type");

    if ((ret_value = H5T_copy(dt, H5T_COPY_TRANSIENT)) == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot copy type");

    
    if (H5T__cmp_offset(comp_size, offset, native_size, (size_t)1, align, struct_align) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "cannot compute compound offset");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5T__cmp_offset(size_t *comp_size, size_t *offset, size_t elem_size, size_t nelems, size_t align,
                size_t *struct_align)
{
    FUNC_ENTER_PACKAGE_NOERR

    if (offset && comp_size) {
        if (align > 1 && *comp_size % align) {
            
            *offset = *comp_size + (align - *comp_size % align);
            *comp_size += (align - *comp_size % align);
        } 
        else
            *offset = *comp_size;

        
        *comp_size += nelems * elem_size;
    } 

    if (struct_align && *struct_align < align)
        *struct_align = align;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

#define TAG_ALIGNMENT(tag) (offsetof(alignments_t, tag.x) - offsetof(alignments_t, tag))

#define NATIVE_ENTRY_INITIALIZER(tag, type, precision, has_sign) {  \
  .alignmentp = &H5T_NATIVE_##tag##_ALIGN_g                         \
, .alignment = TAG_ALIGNMENT(tag)                                   \
, .hidp = &H5T_NATIVE_##tag##_g                                     \
, .size = sizeof(type)                                              \
, .atomic = {                                                       \
      .offset   = 0                                                 \
    , .prec     = (precision != 0) ? precision : (sizeof(type) * 8) \
    , .lsb_pad  = H5T_PAD_ZERO                                      \
    , .msb_pad  = H5T_PAD_ZERO                                      \
    , .u.i.sign = has_sign ? H5T_SGN_2 : H5T_SGN_NONE               \
    }                                                               \
}

static H5T_order_t
get_host_byte_order(void)
{
    static const union {
        uint64_t u64;
        char     byte[8];
    } endian_exemplar = {.byte = {1}};

    return (endian_exemplar.u64 == 1) ? H5T_ORDER_LE : H5T_ORDER_BE;
}

herr_t
H5T__init_native_internal(void)
{
    
    typedef struct {
        struct {
            char        c;
            signed char x;
        } SCHAR;
        struct {
            char          c;
            unsigned char x;
        } UCHAR;
        struct {
            char  c;
            short x;
        } SHORT;
        struct {
            char           c;
            unsigned short x;
        } USHORT;
        struct {
            char c;
            int  x;
        } INT;
        struct {
            char         c;
            unsigned int x;
        } UINT;
        struct {
            char c;
            long x;
        } LONG;
        struct {
            char          c;
            unsigned long x;
        } ULONG;
        struct {
            char      c;
            long long x;
        } LLONG;
        struct {
            char               c;
            unsigned long long x;
        } ULLONG;
        struct {
            char   c;
            int8_t x;
        } INT8;
        struct {
            char    c;
            uint8_t x;
        } UINT8;
        struct {
            char         c;
            int_least8_t x;
        } INT_LEAST8;
        struct {
            char          c;
            uint_least8_t x;
        } UINT_LEAST8;
        struct {
            char        c;
            int_fast8_t x;
        } INT_FAST8;
        struct {
            char         c;
            uint_fast8_t x;
        } UINT_FAST8;
        struct {
            char    c;
            int16_t x;
        } INT16;
        struct {
            char     c;
            uint16_t x;
        } UINT16;
        struct {
            char          c;
            int_least16_t x;
        } INT_LEAST16;
        struct {
            char           c;
            uint_least16_t x;
        } UINT_LEAST16;
        struct {
            char         c;
            int_fast16_t x;
        } INT_FAST16;
        struct {
            char          c;
            uint_fast16_t x;
        } UINT_FAST16;
        struct {
            char    c;
            int32_t x;
        } INT32;
        struct {
            char     c;
            uint32_t x;
        } UINT32;
        struct {
            char          c;
            int_least32_t x;
        } INT_LEAST32;
        struct {
            char           c;
            uint_least32_t x;
        } UINT_LEAST32;
        struct {
            char         c;
            int_fast32_t x;
        } INT_FAST32;
        struct {
            char          c;
            uint_fast32_t x;
        } UINT_FAST32;
        struct {
            char    c;
            int64_t x;
        } INT64;
        struct {
            char     c;
            uint64_t x;
        } UINT64;
        struct {
            char          c;
            int_least64_t x;
        } INT_LEAST64;
        struct {
            char           c;
            uint_least64_t x;
        } UINT_LEAST64;
        struct {
            char         c;
            int_fast64_t x;
        } INT_FAST64;
        struct {
            char          c;
            uint_fast64_t x;
        } UINT_FAST64;
        struct {
            char  c;
            void *x;
        } pointer;
        struct {
            char  c;
            hvl_t x;
        } hvl;
        struct {
            char       c;
            hobj_ref_t x;
        } hobjref;
        struct {
            char            c;
            hdset_reg_ref_t x;
        } hdsetregref;
        struct {
            char      c;
            H5R_ref_t x;
        } ref;
    } alignments_t;

    
    typedef struct {
        
        size_t *alignmentp;
        size_t  alignment; // natural alignment of `type`
        
        hid_t       *hidp;
        size_t       size;   // sizeof(`type`)
        H5T_atomic_t atomic; // `type` facts such as signedness
    } native_int_t;

    typedef struct {
        const native_int_t *table;
        size_t              nelmts;
    } native_int_table_t;

    

    
#if defined(__PGIC__) && __PGIC__ == 19 && __PGIC_MINOR__ == 10
#   define static_unless_buggy_pgic
#else
#   define static_unless_buggy_pgic static
#endif

    
    static_unless_buggy_pgic const native_int_t table1[] = {
      NATIVE_ENTRY_INITIALIZER(SCHAR, signed char, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UCHAR, unsigned char, 0, false)
    , NATIVE_ENTRY_INITIALIZER(SHORT, short, 0, true)
    , NATIVE_ENTRY_INITIALIZER(USHORT, unsigned short, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT, int, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT, unsigned int, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT, int, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT, unsigned int, 0, false)
    , NATIVE_ENTRY_INITIALIZER(LONG, long, 0, true)
    , NATIVE_ENTRY_INITIALIZER(ULONG, unsigned long, 0, false)
    , NATIVE_ENTRY_INITIALIZER(LLONG, long long, 0, true)
    , NATIVE_ENTRY_INITIALIZER(ULLONG, unsigned long long, 0, false)
    };
    static_unless_buggy_pgic const native_int_t table2[] = {
      NATIVE_ENTRY_INITIALIZER(INT8, int8_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT8, uint8_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_LEAST8, int_least8_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_LEAST8, uint_least8_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_FAST8, int_fast8_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_FAST8, uint_fast8_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT16, int16_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT16, uint16_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_LEAST16, int_least16_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_LEAST16, uint_least16_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_FAST16, int_fast16_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_FAST16, uint_fast16_t, 0, false)
    };
    static_unless_buggy_pgic const native_int_t table3[] = {
      NATIVE_ENTRY_INITIALIZER(INT32, int32_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT32, uint32_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_LEAST32, int_least32_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_LEAST32, uint_least32_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_FAST32, int_fast32_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_FAST32, uint_fast32_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT64, int64_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT64, uint64_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_LEAST64, int_least64_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_LEAST64, uint_least64_t, 0, false)
    , NATIVE_ENTRY_INITIALIZER(INT_FAST64, int_fast64_t, 0, true)
    , NATIVE_ENTRY_INITIALIZER(UINT_FAST64, uint_fast64_t, 0, false)
    };
    static_unless_buggy_pgic const native_int_table_t table_table[] = {
      {table1, NELMTS(table1)}
    , {table2, NELMTS(table2)}
    , {table3, NELMTS(table3)}
    };
#undef static_unless_buggy_pgic
    

    size_t      i, j;
    H5T_order_t byte_order = get_host_byte_order();

    for (i = 0; i < NELMTS(table_table); i++) {
        const native_int_t *table  = table_table[i].table;
        size_t              nelmts = table_table[i].nelmts;

        
        for (j = 0; j < nelmts; j++) {
            H5T_t *dt;

            if (NULL == (dt = H5T__alloc()))
                return FAIL;

            dt->shared->state          = H5T_STATE_IMMUTABLE;
            dt->shared->type           = H5T_INTEGER;
            dt->shared->size           = table[j].size;
            dt->shared->u.atomic       = table[j].atomic;
            dt->shared->u.atomic.order = byte_order;
            *table[j].alignmentp       = table[j].alignment;

            if ((*table[j].hidp = H5I_register(H5I_DATATYPE, dt, false)) < 0)
                return FAIL;
        }
    }

    H5T_POINTER_ALIGN_g     = TAG_ALIGNMENT(pointer);
    H5T_HVL_ALIGN_g         = TAG_ALIGNMENT(hvl);
    H5T_HOBJREF_ALIGN_g     = TAG_ALIGNMENT(hobjref);
    H5T_HDSETREGREF_ALIGN_g = TAG_ALIGNMENT(hdsetregref);
    H5T_REF_ALIGN_g         = TAG_ALIGNMENT(ref);

    return SUCCEED;
}

#ifdef H5_HAVE_COMPLEX_NUMBERS

herr_t
H5T__init_native_complex_types(void)
{
    H5T_t *native_float   = NULL; 
    H5T_t *native_double  = NULL; 
    H5T_t *native_ldouble = NULL; 
    H5T_t *dt             = NULL;
    herr_t ret_value      = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(H5T_NATIVE_FLOAT_g != H5I_INVALID_HID);
    assert(H5T_NATIVE_DOUBLE_g != H5I_INVALID_HID);
    assert(H5T_NATIVE_LDOUBLE_g != H5I_INVALID_HID);

    
    typedef struct {
        struct {
            char             c;
            H5_float_complex x;
        } FLOAT_COMPLEX;
        struct {
            char              c;
            H5_double_complex x;
        } DOUBLE_COMPLEX;
        struct {
            char               c;
            H5_ldouble_complex x;
        } LDOUBLE_COMPLEX;
    } alignments_t;

    if (NULL == (native_float = (H5T_t *)H5I_object(H5T_NATIVE_FLOAT_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get datatype structure for native float type");
    if (NULL == (native_double = (H5T_t *)H5I_object(H5T_NATIVE_DOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get datatype structure for native double type");
    if (NULL == (native_ldouble = (H5T_t *)H5I_object(H5T_NATIVE_LDOUBLE_g)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get datatype structure for native long double type");

    

    if (NULL == (dt = H5T__complex_create(native_float)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't create native float complex datatype");
    dt->shared->state = H5T_STATE_IMMUTABLE;

    
    if ((H5T_NATIVE_FLOAT_COMPLEX_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't register ID for native float complex datatype");
    H5T_NATIVE_FLOAT_COMPLEX_ALIGN_g = TAG_ALIGNMENT(FLOAT_COMPLEX);

    dt = NULL;

    

    if (NULL == (dt = H5T__complex_create(native_double)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't create native double complex datatype");
    dt->shared->state = H5T_STATE_IMMUTABLE;

    
    if ((H5T_NATIVE_DOUBLE_COMPLEX_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't register ID for native double complex datatype");
    H5T_NATIVE_DOUBLE_COMPLEX_ALIGN_g = TAG_ALIGNMENT(DOUBLE_COMPLEX);

    dt = NULL;

    

    if (NULL == (dt = H5T__complex_create(native_ldouble)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't create native long double complex datatype");
    dt->shared->state = H5T_STATE_IMMUTABLE;

    
    if ((H5T_NATIVE_LDOUBLE_COMPLEX_g = H5I_register(H5I_DATATYPE, dt, false)) < 0)
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL,
                    "can't register ID for native long double complex datatype");
    H5T_NATIVE_LDOUBLE_COMPLEX_ALIGN_g = TAG_ALIGNMENT(LDOUBLE_COMPLEX);

    dt = NULL;

done:
    if (ret_value < 0) {
        if (dt && (H5T_close(dt) < 0))
            HDONE_ERROR(H5E_DATATYPE, H5E_CANTCLOSEOBJ, FAIL, "can't close datatype");
    }

    FUNC_LEAVE_NOAPI(ret_value)
}
#endif
