/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5module.h" 

#include "H5private.h"

#define H5TIMER_TIME_STRING_LEN 1536

#define H5_SEC_PER_DAY  (24.0 * 60.0 * 60.0)
#define H5_SEC_PER_HOUR (60.0 * 60.0)
#define H5_SEC_PER_MIN  (60.0)

void
H5_bandwidth(char *buf , size_t bufsize, double nbytes, double nseconds)
{
    double bw;

    if (nseconds <= 0.0)
        strcpy(buf, "       NaN");
    else {
        bw = nbytes / nseconds;
        if (H5_DBL_ABS_EQUAL(bw, 0.0))
            strcpy(buf, "0.000  B/s");
        else if (bw < 1.0)
            snprintf(buf, bufsize, "%10.4e", bw);
        else if (bw < (double)H5_KB) {
            snprintf(buf, bufsize, "%05.4f", bw);
            strcpy(buf + 5, "  B/s");
        }
        else if (bw < (double)H5_MB) {
            snprintf(buf, bufsize, "%05.4f", bw / (double)H5_KB);
            strcpy(buf + 5, " kB/s");
        }
        else if (bw < (double)H5_GB) {
            snprintf(buf, bufsize, "%05.4f", bw / (double)H5_MB);
            strcpy(buf + 5, " MB/s");
        }
        else if (bw < (double)H5_TB) {
            snprintf(buf, bufsize, "%05.4f", bw / (double)H5_GB);
            strcpy(buf + 5, " GB/s");
        }
        else if (bw < (double)H5_PB) {
            snprintf(buf, bufsize, "%05.4f", bw / (double)H5_TB);
            strcpy(buf + 5, " TB/s");
        }
        else if (bw < (double)H5_EB) {
            snprintf(buf, bufsize, "%05.4f", bw / (double)H5_PB);
            strcpy(buf + 5, " PB/s");
        }
        else {
            snprintf(buf, bufsize, "%10.4e", bw);
            if (strlen(buf) > 10)
                snprintf(buf, bufsize, "%10.3e", bw);
        }
    }
} 

time_t
H5_now(void)
{
    time_t now; 

#ifdef H5_HAVE_GETTIMEOFDAY
    {
        struct timeval now_tv;

        HDgettimeofday(&now_tv, NULL);
        now = now_tv.tv_sec;
    }
#else
    now = time(NULL);
#endif

    return now;
} 

uint64_t
H5_now_usec(void)
{
    uint64_t now; 

#if defined(H5_HAVE_CLOCK_GETTIME)
    {
        struct timespec ts;

        clock_gettime(CLOCK_MONOTONIC, &ts);

        
        now = ((uint64_t)ts.tv_sec * ((uint64_t)1000 * (uint64_t)1000)) +
              ((uint64_t)ts.tv_nsec / (uint64_t)1000);
    }
#elif defined(H5_HAVE_GETTIMEOFDAY)
    {
        struct timeval now_tv;

        HDgettimeofday(&now_tv, NULL);

        
        now = ((uint64_t)now_tv.tv_sec * ((uint64_t)1000 * (uint64_t)1000)) + (uint64_t)now_tv.tv_usec;
    }
#else
    
    now = ((uint64_t)time(NULL) * ((uint64_t)1000 * (uint64_t)1000));
#endif

    return now;
} 

double
H5_get_time(void)
{
    double ret_value = 0.0;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

#if defined(H5_HAVE_CLOCK_GETTIME)
    {
        struct timespec ts;

        clock_gettime(CLOCK_MONOTONIC, &ts);
        ret_value = (double)ts.tv_sec + ((double)ts.tv_nsec / 1000000000.0);
    }
#elif defined(H5_HAVE_GETTIMEOFDAY)
    {
        struct timeval now_tv;

        HDgettimeofday(&now_tv, NULL);
        ret_value = (double)now_tv.tv_sec + ((double)now_tv.tv_usec / 1000000.0);
    }
#else
    ret_value = (double)time(NULL);
#endif

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5__timer_get_timevals(H5_timevals_t *times )
{
    assert(times);

    
#ifdef H5_HAVE_WIN32_API
    if (H5_get_win32_times(times) < 0) {
        times->elapsed = -1.0;
        times->system  = -1.0;
        times->user    = -1.0;

        return -1;
    }
#else 

    
#if defined(H5_HAVE_GETRUSAGE)
    {
        struct rusage res;

        if (getrusage(RUSAGE_SELF, &res) < 0)
            return -1;
        times->system = (double)res.ru_stime.tv_sec + ((double)res.ru_stime.tv_usec / 1.0E6);
        times->user   = (double)res.ru_utime.tv_sec + ((double)res.ru_utime.tv_usec / 1.0E6);
    }
#else
    
    
    times->system = -1.0;
    times->user   = -1.0;
#endif

    

    times->elapsed = H5_get_time();

#endif 

    return 0;
} 

herr_t
H5_timer_init(H5_timer_t *timer )
{
    assert(timer);

    
    memset(timer, 0, sizeof(H5_timer_t));

    return 0;
} 

herr_t
H5_timer_start(H5_timer_t *timer )
{
    assert(timer);

    
    if (H5__timer_get_timevals(&(timer->initial)) < 0)
        return -1;

    timer->is_running = true;

    return 0;
} 

herr_t
H5_timer_stop(H5_timer_t *timer )
{
    assert(timer);

    
    if (H5__timer_get_timevals(&(timer->final_interval)) < 0)
        return -1;

    
    timer->final_interval.elapsed = timer->final_interval.elapsed - timer->initial.elapsed;
    timer->final_interval.system  = timer->final_interval.system - timer->initial.system;
    timer->final_interval.user    = timer->final_interval.user - timer->initial.user;

    
    timer->total.elapsed += timer->final_interval.elapsed;
    timer->total.system += timer->final_interval.system;
    timer->total.user += timer->final_interval.user;

    timer->is_running = false;

    return 0;
} 

herr_t
H5_timer_get_times(H5_timer_t timer, H5_timevals_t *times )
{
    assert(times);

    if (timer.is_running) {
        H5_timevals_t now;

        
        if (H5__timer_get_timevals(&now) < 0)
            return -1;

        times->elapsed = now.elapsed - timer.initial.elapsed;
        times->system  = now.system - timer.initial.system;
        times->user    = now.user - timer.initial.user;
    }
    else {
        times->elapsed = timer.final_interval.elapsed;
        times->system  = timer.final_interval.system;
        times->user    = timer.final_interval.user;
    }

    return 0;
} 

herr_t
H5_timer_get_total_times(H5_timer_t timer, H5_timevals_t *times )
{
    assert(times);

    if (timer.is_running) {
        H5_timevals_t now;

        
        if (H5__timer_get_timevals(&now) < 0)
            return -1;

        times->elapsed = timer.total.elapsed + (now.elapsed - timer.initial.elapsed);
        times->system  = timer.total.system + (now.system - timer.initial.system);
        times->user    = timer.total.user + (now.user - timer.initial.user);
    }
    else {
        times->elapsed = timer.total.elapsed;
        times->system  = timer.total.system;
        times->user    = timer.total.user;
    }

    return 0;
} 

char *
H5_timer_get_time_string(double seconds)
{
    char *s; 

    
    double days          = 0.0;
    double hours         = 0.0;
    double minutes       = 0.0;
    double remainder_sec = 0.0;

    
    if (seconds > 60.0) {
        
        remainder_sec = seconds;

        
        days = floor(remainder_sec / H5_SEC_PER_DAY);
        remainder_sec -= (days * H5_SEC_PER_DAY);

        
        hours = floor(remainder_sec / H5_SEC_PER_HOUR);
        remainder_sec -= (hours * H5_SEC_PER_HOUR);

        
        minutes = floor(remainder_sec / H5_SEC_PER_MIN);
        remainder_sec -= (minutes * H5_SEC_PER_MIN);

        
    }

    
    if (NULL == (s = (char *)calloc(H5TIMER_TIME_STRING_LEN, sizeof(char))))
        return NULL;

    
    if (seconds < 0.0)
        snprintf(s, H5TIMER_TIME_STRING_LEN, "N/A");
    else if (H5_DBL_ABS_EQUAL(0.0, seconds))
        snprintf(s, H5TIMER_TIME_STRING_LEN, "0.0 s");
    else if (seconds < 1.0E-6)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.f ns", seconds * 1.0E9);
    else if (seconds < 1.0E-3)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.1f us", seconds * 1.0E6);
    else if (seconds < 1.0)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.1f ms", seconds * 1.0E3);
    else if (seconds < H5_SEC_PER_MIN)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.2f s", seconds);
    else if (seconds < H5_SEC_PER_HOUR)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.f m %.f s", minutes, remainder_sec);
    else if (seconds < H5_SEC_PER_DAY)
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.f h %.f m %.f s", hours, minutes, remainder_sec);
    else
        
        snprintf(s, H5TIMER_TIME_STRING_LEN, "%.f d %.f h %.f m %.f s", days, hours, minutes, remainder_sec);

    return s;
} 
