% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tpmatrix.R
\name{apply_rr}
\alias{apply_rr}
\title{Apply relative risks to transition probability matrices}
\usage{
apply_rr(x, rr, index, complement = NULL)
}
\arguments{
\item{x}{A 3-dimensional array where each slice is a square transition
probability matrix.}

\item{rr}{A 2-dimensional tabular object such as a matrix or data frame where each
column is a vector of relative risks to apply to each transition matrix in \code{x}.}

\item{index}{The indices of the transition probability matrices that \code{rr}  is applied to.
Should either be a matrix where the first column denotes a transition probability matrix row
and the second column denotes a transition probability matrix column or a list
where each element is a vector of length 2 with the first element denoting
a transition probability matrix row and the second column denoting a transition
probability matrix column.}

\item{complement}{Denotes indices of transition probability matrices that are
"complements" (i.e.,  computed as \eqn{1} less the sum of all other
elements in that row). Should be in the same format as \code{index}. There can be
at most one complementary column in each row of a transition probability
matrix. If \code{NULL}, then the diagonals are assumed to be the complements.}
}
\value{
A 3-dimensional array where each slice contains matrices of the same
dimension as each matrix in \code{x} and the number of slices is equal to the number
of rows in \code{rr}.
}
\description{
Elements of transition probability matrices are multiplied by relative risks
and the transition probability matrices are adjusted so that rows sum to 1.
Operations are vectorized and each relative risk is multiplied by every
transition matrix (stored in 3-dimensional arrays).
}
\details{
This function is useful for applying relative treatment effects measured
using relative risks to an existing transition probability matrix. For example,
a transition probability matrix for the reference treatment strategy may exist or
have been estimated from the data. Relative risks estimated from a meta-analysis
or network meta-analysis can then be applied to the reference transition probability
matrix. If the number of rows in \code{rr} exceeds \code{x}, then the arrays in \code{x} are
recycled to the number of rows in \code{rr}, which facilitates the application of
relative risks from multiple treatment strategies to a reference treatment.
}
\examples{
p_12 <- c(.7, .5)
p_23 <- c(.1, .2)
x <- as_array3(tpmatrix(
  C, p_12, .1,
  0, C,     p_23,
  0, 0,     1
))

# There are the same number of relative risk rows and transition probability matrices
rr_12 <- runif(2, .8, 1)
rr_13 <- runif(2, .9, 1)
rr <- cbind(rr_12, rr_13)
apply_rr(x, rr, 
         index = list(c(1, 2), c(1, 3)),
         complement = list(c(1, 1), c(2, 2)))
         
# There are more relative risk rows than transition probability matrices
rr_12 <- runif(4, .8, 1)
rr_13 <- runif(4, .9, 1)
rr <- cbind(rr_12, rr_13)
apply_rr(x, rr, 
         index = list(c(1, 2), c(1, 3)),
         complement = list(c(1, 1), c(2, 2)))
}
