% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/export_table.R
\name{export_table}
\alias{export_table}
\title{Data frame and Tables Pretty Formatting}
\usage{
export_table(
  x,
  sep = " | ",
  header = "-",
  cross = NULL,
  empty_line = NULL,
  digits = 2,
  protect_integers = TRUE,
  missing = "",
  width = NULL,
  format = NULL,
  title = NULL,
  caption = title,
  subtitle = NULL,
  footer = NULL,
  column_names = NULL,
  align = NULL,
  by = NULL,
  zap_small = FALSE,
  table_width = "auto",
  remove_duplicates = FALSE,
  row_groups = NULL,
  column_groups = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{x}{A data frame. May also be a list of data frames, to export multiple
data frames into multiple tables.}

\item{sep}{Column separator.}

\item{header}{Header separator. Can be \code{NULL}.}

\item{cross}{Character that is used where separator and header lines cross.}

\item{empty_line}{Separator used for empty lines. If \code{NULL}, line remains
empty (i.e. filled with whitespaces).}

\item{digits}{Number of digits for rounding or significant figures. May also
be \code{"signif"} to return significant figures or \code{"scientific"}
to return scientific notation. Control the number of digits by adding the
value as suffix, e.g. \code{digits = "scientific4"} to have scientific
notation with 4 decimal places, or \code{digits = "signif5"} for 5
significant figures (see also \code{\link[=signif]{signif()}}).}

\item{protect_integers}{Should integers be kept as integers (i.e., without
decimals)?}

\item{missing}{Value by which \code{NA} values are replaced. By default, an
empty string (i.e. \code{""}) is returned for \code{NA}.}

\item{width}{Refers to the width of columns (with numeric values). Can be
either \code{NULL}, a number or a named numeric vector. If \code{NULL}, the width for
each column is adjusted to the minimum required width. If a number, columns
with numeric values will have the minimum width specified in \code{width}. If
a named numeric vector, value names are matched against column names, and
for each match, the specified width is used (see 'Examples'). Only applies
to text-format (see \code{format}).}

\item{format}{Name of output-format, as string. If \code{NULL} (or \code{"text"}),
returned output is used for basic printing. Can be one of \code{NULL} (the
default) resp. \code{"text"} for plain text, \code{"markdown"} (or \code{"md"}) for
markdown and \code{"html"} for HTML output. A special option is \code{"tt"}, which
creates a \code{\link[tinytable:tt]{tinytable::tt()}} object, where the output format is dependent
on the context where the table is used, i.e. it can be markdown format when
\code{export_table()} is used in markdown files, or LaTeX format when creating
PDFs etc.}

\item{title, caption, subtitle}{Table title (same as caption) and subtitle, as
strings. If \code{NULL}, no title or subtitle is printed, unless it is stored as
attributes (\code{table_title}, or its alias \code{table_caption}, and
\code{table_subtitle}). If you want to force that no title is printed, even if
present as attribute, use \code{""}, which will never print titles. If \code{x} is a
list of data frames, \code{caption} may be a list of table captions, one for
each table.}

\item{footer}{Table footer, as string. For markdown-formatted tables, table
footers, due to the limitation in markdown rendering, are actually just a
new text line under the table. If \code{x} is a list of data frames, \code{footer}
may be a list of table captions, one for each table. If \code{NULL}, no footer
is printed, unless it is stored as attributes (\code{table_footer}). If you want
to force that no footer is printed, even if present as attribute, use \code{""},
which will never print footers.}

\item{column_names}{Character vector of names that will be used as column
names in the table. Must either be of same length as columns in the table,
or a named vector, where names (LHS) indicate old column names, and values
(RHS) are used as new column names.}

\item{align}{Column alignment. For markdown-formatted tables, the default
\code{align = NULL} will right-align numeric columns, while all other columns
will be left-aligned. If \code{format = "html"}, the default is left-align first
column and center all remaining. May be a string to indicate alignment
rules for the complete table, like \code{"left"}, \code{"right"}, \code{"center"} or
\code{"firstleft"} (to left-align first column, center remaining); or a string
with abbreviated alignment characters, where the length of the string must
equal the number of columns. For instance, \code{align = "lccrl"} would
left-align the first column, center the second and third, right-align
column four and left-align the fifth column.}

\item{by}{Name of column(s) in \code{x} that indicates grouping for tables.
When \code{format = "html"}, \code{by} is passed down to \code{gt::gt(groupname_col = by)}.
Likewise, for \code{format = "tt"}, \code{by} indicates the name of the variable in
the data frame, which is then used to create row headers in the table.
For markdown and text format, \code{x} is internally split into a list of data
frames. See also \code{row_groups} to group rows in the printed output.}

\item{zap_small}{Logical, if \code{TRUE}, small values are rounded after
\code{digits} decimal places. If \code{FALSE}, values with more decimal
places than \code{digits} are printed in scientific notation.}

\item{table_width}{Numeric,\code{"auto"}, \code{NULL} or \code{Inf}, indicating the width of
the complete table.
\itemize{
\item If \code{table_width = "auto"} (default) and the table is wider than the
current width (i.e. line length) of the console (or any other source for
textual output, like markdown files), the table is split into multiple
parts.
\item Else, if \code{table_width} is numeric and table rows are larger than
\code{table_width}, the table is split into multiple parts. For each new table,
the first column is repeated for better orientation.
\item Use \code{NULL} or \code{Inf} to turn off automatic splitting of the table.
\item \verb{options(easystats_table_width = <value>)} can be used to set a default
width for tables.
}}

\item{remove_duplicates}{Logical, if \code{TRUE} and table is split into multiple
parts, duplicated ("empty") rows will be removed. If \code{FALSE}, empty rows
will be preserved. Only applies when \code{table_width} is \emph{not} \code{NULL} (or
\code{Inf}) \emph{and} table is split into multiple parts.}

\item{row_groups}{Named list, can be used as alternative to \code{by} to group
rows in the printed output, but in a more flexible way. List elements may
either be character vectors that match the names of values in the first
column of the data frame that belong to one group, or list elements can be
row numbers of those value rows that should belong to one group. The names
of the list elements will be used as group names, which will be inserted as
"header row". Rows will be re-ordered according to the order used in
\code{row_groups}, while all rows with non-matching values will be added to the
end.}

\item{column_groups}{Named list, can be used to group columns in the printed
output. List elements must indicate column indices for columns that should
belong to one group. The names of the list elements will be used as group
names, which will be inserted as "column header row". Currently only
works for \code{format = "tt"} or \code{format = "html"}.}

\item{verbose}{Toggle messages and warnings.}

\item{...}{Arguments passed to \code{\link[tinytable:tt]{tinytable::tt()}} and \code{\link[tinytable:style_tt]{tinytable::style_tt()}}
when \code{format = "tt"}.}
}
\value{
If \code{format = "text"} (or \code{NULL}), a formatted character string is
returned. \code{format = "markdown"} (or \code{"md"}) returns a character string of
class \code{knitr_kable}, which renders nicely in markdown files. \code{format = "html"}
returns an \code{gt} object (created by the \strong{gt} package), which - by default -
is displayed in the IDE's viewer pane or default browser. This object can be
further modified with the various gt-functions. \code{format = "tt"} returns a
\code{\link[tinytable:tt]{tinytable::tt()}} object, which is a lightweight table format that can be
used in markdown, LaTeX, HTML and other formats, depending on the context
where the table is used.
}
\description{
Function to export data frames into tables, which can be printed
to the console, or displayed in markdown or HTML format (and thereby, exported
to other formats like Word or PDF). The table width is automatically adjusted
to fit into the width of the display device (e.g., width of console). Use
the \code{table_width} argument to control this behaviour.
}
\note{
The values for \code{caption}, \code{subtitle} and \code{footer} can also be provided
as attributes of \code{x}, e.g. if \code{caption = NULL} and \code{x} has attribute
\code{table_caption}, the value for this attribute will be used as table caption.
\code{table_subtitle} is the attribute for \code{subtitle}, and \code{table_footer} for
\code{footer}.
}
\examples{
export_table(head(iris))
export_table(head(iris), cross = "+")
export_table(head(iris), sep = " ", header = "*", digits = 1)

# split longer tables
export_table(head(iris), table_width = 30)

# group (split) tables by variables
export_table(head(mtcars, 8), by = "cyl")

\donttest{
# colored footers
data(iris)
x <- as.data.frame(iris[1:5, ])
attr(x, "table_footer") <- c("This is a yellow footer line.", "yellow")
export_table(x)

attr(x, "table_footer") <- list(
  c("\nA yellow line", "yellow"),
  c("\nAnd a red line", "red"),
  c("\nAnd a blue line", "blue")
)
export_table(x)

attr(x, "table_footer") <- list(
  c("Without the ", "yellow"),
  c("new-line character ", "red"),
  c("we can have multiple colors per line.", "blue")
)
export_table(x)

# rename column names
export_table(x, column_names = letters[1:5])
export_table(x, column_names = c(Species = "a"))
}

# column-width
d <- data.frame(
  x = c(1, 2, 3),
  y = c(100, 200, 300),
  z = c(10000, 20000, 30000)
)
export_table(d)
export_table(d, width = 8)
export_table(d, width = c(x = 5, z = 10))
export_table(d, width = c(x = 5, y = 5, z = 10), align = "lcr")

# group rows in the table
\dontrun{
data(mtcars)

# fit model
mtcars$cyl <- as.factor(mtcars$cyl)
mtcars$gear <- as.factor(mtcars$gear)
model <- lm(mpg ~ hp + gear * vs + cyl + drat, data = mtcars)

# model summary, don't select "Intercept" parameter
mp <- as.data.frame(format(
  parameters::model_parameters(model, drop = "^\\\\(Intercept")
))

# define groups for the table
groups <- list(
  Engine = c("cyl [6]", "cyl [8]", "vs", "hp"),
  Interactions = c(8, 9),
  Controls = c(2, 3, 7)
)

# export table with groups, using tinytable format
export_table(mp, format = "tt", row_groups = groups)
}
}
\seealso{
Vignettes
\href{https://easystats.github.io/insight/articles/display.html}{Formatting, printing and exporting tables}
and \href{https://easystats.github.io/parameters/articles/model_parameters_formatting.html}{Formatting model parameters}.
}
