% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/islasso-package.R, R/islasso.R
\docType{package}
\name{islasso}
\alias{islasso-package}
\alias{islasso}
\alias{print.islasso}
\alias{islasso.fit}
\alias{vcov.islasso}
\alias{deviance.islasso}
\alias{residuals.islasso}
\alias{logLik.islasso}
\alias{model.matrix.islasso}
\alias{cooks.distance.islasso}
\alias{extractAIC.islasso}
\alias{family.islasso}
\alias{formula.islasso}
\alias{influence.islasso}
\alias{model.frame.islasso}
\alias{nobs.islasso}
\alias{rstandard.islasso}
\alias{rstudent.islasso}
\alias{variable.names.islasso}
\alias{weights.islasso}
\title{The Induced Smoothed Lasso: A practical framework for hypothesis testing in high dimensional regression}
\usage{
islasso(
  formula,
  family = gaussian,
  lambda,
  alpha = 1,
  data,
  weights,
  subset,
  offset,
  unpenalized,
  contrasts = NULL,
  control = is.control()
)
}
\arguments{
\item{formula}{A symbolic formula describing the model.}

\item{family}{Response distribution. Can be \code{gaussian}, \code{binomial}, \code{poisson}, or \code{Gamma}.}

\item{lambda}{Regularization parameter. If missing, it is estimated via \code{\link[glmnet]{cv.glmnet}}.}

\item{alpha}{Elastic-net mixing parameter (\eqn{0 \le \alpha \le 1}).}

\item{data}{A data frame or environment containing the variables in the model.}

\item{weights}{Observation weights. Defaults to 1.}

\item{subset}{Optional vector specifying a subset of rows to include.}

\item{offset}{Optional numeric vector of offsets in the linear predictor.}

\item{unpenalized}{Vector indicating variables (by name or index) to exclude from penalization.}

\item{contrasts}{Optional contrasts specification for factor variables.}

\item{control}{A list of parameters to control model fitting. See \code{\link{is.control}}.}
}
\value{
A list with components such as:
  \item{coefficients}{Estimated coefficients}
  \item{se}{Standard errors}
  \item{fitted.values}{Fitted values}
  \item{deviance, aic, null.deviance}{Model diagnostic metrics}
  \item{residuals, weights}{IWLS residuals and weights}
  \item{df.residual, df.null, rank}{Degrees of freedom}
  \item{converged}{Logical; convergence status}
  \item{model, call, terms, formula, data, offset}{Model objects}
  \item{xlevels, contrasts}{Factor handling details}
  \item{lambda, alpha, dispersion}{Model parameters}
  \item{internal}{Other internal values}
}
\description{
This package implements an induced smoothed approach for hypothesis testing in Lasso regression.

Fits regression models with a smoothed L1 penalty under the induced smoothing paradigm.
Supports linear, logistic, Poisson, and Gamma responses. Enables reliable standard errors
and Wald-based inference.
}
\details{
\tabular{ll}{
Package: \tab islasso\cr
Type: \tab Package\cr
Version: \tab 1.6.1\cr
Date: \tab 2025-11-13\cr
License: \tab GPL-2\cr
}

\code{\link{islasso}} fits generalized linear models with an L1 penalty on selected coefficients.
It returns both point estimates and full covariance matrices, enabling standard error-based inference.
Related methods include: \code{\link{summary.islasso}}, \code{\link{predict.islasso}}, \code{\link{logLik.islasso}}, \code{\link{deviance.islasso}}, and \code{\link{residuals.islasso}}.

\code{\link{islasso.path}} fits regularization paths using the Induced Smoothed Lasso.
It computes coefficients and standard errors across a grid of \code{lambda} values.
Companion methods include: \code{\link{summary.islasso.path}}, \code{\link{predict.islasso.path}}, \code{\link{logLik.islasso.path}}, \code{\link{residuals.islasso.path}}, \code{\link{coef.islasso.path}}, and \code{\link{fitted.islasso.path}}.

The non-smooth L1 penalty is replaced by a smooth approximation, enabling inference through
standard errors and Wald tests. The approach controls type-I error and shows strong power
in various simulation settings.
}
\examples{
n <- 100; p <- 100

beta <- c(rep(1, 5), rep(0, p - 5))
sim1 <- simulXy(n = n, p = p, beta = beta, seed = 1, family = gaussian())
o <- islasso(y ~ ., data = sim1$data, family = gaussian())

summary(o, pval = 0.05)
coef(o)
fitted(o)
predict(o, type="response")
plot(o)
residuals(o)
deviance(o)
AIC(o)
logLik(o)
summary(relax.islasso(o), pval = 0.05)

\dontrun{
# for the interaction
o <- islasso(y ~ X1 * X2, data = sim1$data, family = gaussian())

##### binomial ######
beta <- c(c(1,1,1), rep(0, p-3))
sim2 <- simulXy(n = n, p = p, beta = beta, interc = 1, seed = 1,
                size = 100, family = binomial())
o2 <- islasso(cbind(y.success, y.failure) ~ .,
              data = sim2$data, family = binomial())
summary(o2, pval = 0.05)

##### poisson ######
beta <- c(c(1,1,1), rep(0, p-3))
sim3 <- simulXy(n = n, p = p, beta = beta, interc = 1, seed = 1,
                family = poisson())
o3 <- islasso(y ~ ., data = sim3$data, family = poisson())
summary(o3, pval = 0.05)

##### Gamma ######
beta <- c(c(1,1,1), rep(0, p-3))
sim4 <- simulXy(n = n, p = p, beta = beta, interc = -1, seed = 1,
                dispersion = 0.1, family = Gamma(link = "log"))
o4 <- islasso(y ~ ., data = sim4$data, family = Gamma(link = "log"))
summary(o4, pval = 0.05)
}

}
\references{
Cilluffo, G., Sottile, G., La Grutta, S., Muggeo, VMR (2019). *The Induced Smoothed lasso: A practical framework for hypothesis testing in high dimensional regression*, Statistical Methods in Medical Research.
DOI: \doi{10.1177/0962280219842890}

Sottile, G., Cilluffo, G., Muggeo, VMR (2019). *The R package islasso: estimation and hypothesis testing in lasso regression*. Technical Report on ResearchGate.
DOI: \doi{10.13140/RG.2.2.16360.11521}

Cilluffo G., Sottile G., La Grutta S., Muggeo V.M.R. (2019)
\emph{The Induced Smoothed Lasso: A practical framework for hypothesis testing in high dimensional regression}.
Statistical Methods in Medical Research. DOI: 10.1177/0962280219842890

Sottile G., Cilluffo G., Muggeo V.M.R. (2019)
\emph{The R package islasso: estimation and hypothesis testing in lasso regression}.
Technical Report. DOI: 10.13140/RG.2.2.16360.11521
}
\seealso{
Useful links:
\itemize{
  \item \url{https://gianluca-sottile.github.io/islasso/}
  \item Report bugs at \url{https://github.com/gianluca-sottile/islasso/issues}
}


\code{\link{confint.islasso}}, \code{\link{plot.islasso}}, \code{\link{predict.islasso}},
         \code{\link{summary.islasso}}, \code{\link{is.control}}, \code{\link{aic.islasso}},
         \code{\link{anova.islasso}}, \code{\link{relax.islasso}}, \code{\link{islasso.path}},
         \code{\link{simulXy}}
}
\author{
Gianluca Sottile, based on preliminary work by Vito Muggeo.
Maintainer: \email{gianluca.sottile@unipa.it}

Gianluca Sottile \email{gianluca.sottile@unipa.it}
}
\keyword{models}
\keyword{package}
\keyword{regression}
