% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimizers-schedules.R
\name{learning_rate_schedule_inverse_time_decay}
\alias{learning_rate_schedule_inverse_time_decay}
\title{A \code{LearningRateSchedule} that uses an inverse time decay schedule.}
\usage{
learning_rate_schedule_inverse_time_decay(
  initial_learning_rate,
  decay_steps,
  decay_rate,
  staircase = FALSE,
  name = "InverseTimeDecay"
)
}
\arguments{
\item{initial_learning_rate}{A float. The initial learning rate.}

\item{decay_steps}{How often to apply decay.}

\item{decay_rate}{A number.  The decay rate.}

\item{staircase}{Whether to apply decay in a discrete staircase, as o
pposed to continuous, fashion.}

\item{name}{String.  Optional name of the operation.  Defaults to
\code{"InverseTimeDecay"}.}
}
\value{
A 1-arg callable learning rate schedule that takes the current optimizer
step and outputs the decayed learning rate, a scalar tensor of the
same type as \code{initial_learning_rate}.
}
\description{
When training a model, it is often useful to lower the learning rate as
the training progresses. This schedule applies the inverse decay function
to an optimizer step, given a provided initial learning rate.
It requires a \code{step} value to compute the decayed learning rate. You can
just pass a backend variable that you increment at each training step.

The schedule is a 1-arg callable that produces a decayed learning
rate when passed the current optimizer step. This can be useful for changing
the learning rate value across different invocations of optimizer functions.
It is computed as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  initial_learning_rate / (1 + decay_rate * step / decay_step)
\}
}\if{html}{\out{</div>}}

or, if \code{staircase} is \code{TRUE}, as:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{decayed_learning_rate <- function(step) \{
  initial_learning_rate /
           (1 + decay_rate * floor(step / decay_step))
\}
}\if{html}{\out{</div>}}

You can pass this schedule directly into a \verb{optimizer_*}
as the learning rate.
}
\section{Examples}{
Fit a Keras model when decaying 1/t with a rate of 0.5:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{...
initial_learning_rate <- 0.1
decay_steps <- 1.0
decay_rate <- 0.5
learning_rate_fn <- learning_rate_schedule_inverse_time_decay(
    initial_learning_rate, decay_steps, decay_rate)

model \%>\% compile(
  optimizer = optimizer_sgd(learning_rate=learning_rate_fn),
  loss = 'sparse_categorical_crossentropy',
  metrics = 'accuracy')
)

model \%>\% fit(data, labels, epochs=5)
}\if{html}{\out{</div>}}
}

\seealso{
\itemize{
\item \url{https://keras.io/api/optimizers/learning_rate_schedules/inverse_time_decay#inversetimedecay-class}
}

Other optimizer learning rate schedules: \cr
\code{\link{LearningRateSchedule}()} \cr
\code{\link{learning_rate_schedule_cosine_decay}()} \cr
\code{\link{learning_rate_schedule_cosine_decay_restarts}()} \cr
\code{\link{learning_rate_schedule_exponential_decay}()} \cr
\code{\link{learning_rate_schedule_piecewise_constant_decay}()} \cr
\code{\link{learning_rate_schedule_polynomial_decay}()} \cr
}
\concept{optimizer learning rate schedules}
