% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/primersearch.R
\name{primersearch}
\alias{primersearch}
\title{Use EMBOSS primersearch for in silico PCR}
\usage{
primersearch(obj, seqs, forward, reverse, mismatch = 5, clone = TRUE)
}
\arguments{
\item{obj}{A \code{\link{taxmap}} object.}

\item{seqs}{The sequences to do in silico PCR on. This can be any variable in
\code{obj$data} listed in \code{all_names(obj)} or an external variable. If
an external variable (i.e. not in \code{obj$data}), it must be named by
taxon IDs or have the same length as the number of taxa in \code{obj}.
Currently, only character vectors are accepted.}

\item{forward}{(\code{character} of length 1) The forward primer sequence}

\item{reverse}{(\code{character} of length 1) The reverse primer sequence}

\item{mismatch}{An integer vector of length 1. The percentage of mismatches allowed.}

\item{clone}{If \code{TRUE}, make a copy of the input object and add on the results (like most R
functions). If \code{FALSE}, the input will be changed without saving the result, which uses less RAM.}
}
\value{
A copy of the input \code{\link{taxmap}} object with two tables added. One table contains amplicon information with one row per predicted amplicon with the following info:

\preformatted{
           (f_primer)
   5' AAGTACCTTAACGGAATTATAG ->        (r_primer)
                               <- TAAGCAAAGCATCCACCTCG 5'
5' ...AAGTACCTTAACGGAATTATAG......ATTCGTTTCGTAGGTGGAGC... 3'
      ^                    ^      ^                  ^
   f_start              f_end   r_rtart             r_end
     
      |--------------------||----||------------------|
             f_match       amplicon       r_match  
      |----------------------------------------------|
                           product
                           
}

 \describe{
    \item{taxon_id:}{The taxon IDs for the sequence.} 
    \item{seq_index:}{The index of the input sequence.} 
    \item{f_primer:}{The sequence of the forward primer.} 
    \item{r_primer:}{The sequence of the reverse primer.} 
    \item{f_mismatch:}{The number of mismatches on the forward primer.} 
    \item{r_mismatch:}{The number of mismatches on the reverse primer.} 
    \item{f_start:}{The start location of the forward primer.} 
    \item{f_end:}{The end location of the forward primer.} 
    \item{r_start:}{The start location of the reverse primer.}  
    \item{r_end:}{The end location of the reverse primer.} 
    \item{f_match:}{The sequence matched by the forward primer.} 
    \item{r_match:}{The sequence matched by the reverse primer.} 
    \item{amplicon:}{The sequence amplified by the primers, not including the primers.} 
    \item{product:}{The sequence amplified by the primers including the primers. This simulates a real PCR product.} 
  }
  
The other table contains per-taxon information about the PCR, with one row per taxon. It has the following columns:

 \describe{
    \item{taxon_ids:}{Taxon IDs.}
    \item{query_count:}{The number of sequences used as input.}
    \item{seq_count:}{The number of sequences that had at least one amplicon.}
    \item{amp_count:}{The number of amplicons. Might be more than one per sequence.}
    \item{amplified:}{If at least one sequence of that taxon had at least one amplicon.}
    \item{multiple:}{If at least one sequences had at least two amplicons.}
    \item{prop_amplified:}{The proportion of sequences with at least one amplicon.}
    \item{med_amp_len:}{The median amplicon length.}
    \item{min_amp_len:}{The minimum amplicon length.}
    \item{max_amp_len:}{The maximum amplicon length.}
    \item{med_prod_len:}{The median product length.}
    \item{min_prod_len:}{The minimum product length.}
    \item{max_prod_len:}{The maximum product length.}
  }
}
\description{
A pair of primers are aligned against a set of sequences. A
\code{\link{taxmap}} object with two tables is returned: a table with
information for each predicted amplicon, quality of match, and predicted
amplicons, and a table with per-taxon amplification statistics. Requires the
EMBOSS tool kit (\url{https://emboss.sourceforge.net/}) to be installed.
}
\details{
It can be confusing how the primer sequence relates to the binding sites on a
reference database sequence. A simplified diagram can help. For example, if
the top strand below (5' -> 3') is the database sequence, the forward primer
has the same sequence as the target region, since it will bind to the other
strand (3' -> 5') during PCR and extend on the 3' end. However, the reverse
primer must bind to the database strand, so it will have to be the complement
of the reference sequence. It also has to be reversed to make it in the
standard 5' -> 3' orientation. Therefore, the reverse primer must be the
reverse complement of its binding site on the reference sequence.
\preformatted{
Primer 1: 5' AAGTACCTTAACGGAATTATAG 3'
Primer 2: 5' GCTCCACCTACGAAACGAAT   3'
 
                               <- TAAGCAAAGCATCCACCTCG 5'
5' ...AAGTACCTTAACGGAATTATAG......ATTCGTTTCGTAGGTGGAGC... 3'

3' ...TTCATGGAATTGCCTTAATATC......TAAGCAAAGCATCCACCTCG... 5'
   5' AAGTACCTTAACGGAATTATAG ->
}
However, a database might have either the top or the bottom strand as a
reference sequence. Since one implies the sequence of the other, either is
valid, but this is another source of confusion. If we take the diagram above
and rotate it 180 degrees, it would mean the same thing, but which primer we would
want to call "forward" and which we would want to call "reverse" would
change. Databases of a single locus (e.g. Greengenes) will likely have a
convention for which strand will be present, so relative to this convention,
there is a distinct "forward" and "reverse". However, computers dont know
about this convention, so the "forward" primer is whichever primer has the
same sequence as its binding region in the database (as opposed to the
reverse complement). For this reason, primersearch will redefine which primer
is "forward" and which is "reverse" based on how it binds the reference
sequence. See the example code in \code{\link{primersearch_raw}} for a
demonstration of this.
}
\section{Installing EMBOSS}{


The command-line tool "primersearch" from the EMBOSS tool kit is needed to use this function.
How you install EMBOSS will depend on your operating system:

\strong{Linux:}

Open up a terminal and type:

\code{sudo apt-get install emboss}

\strong{Mac OSX:}

The easiest way to install EMBOSS on OSX is to use \href{https://brew.sh/}{homebrew}.
After installing homebrew, open up a terminal and type:

\code{brew install homebrew/science/emboss}

\strong{Windows:}

There is an installer for Windows here:

ftp://emboss.open-bio.org/pub/EMBOSS/windows/mEMBOSS-6.5.0.0-setup.exe
}

\examples{
\donttest{
# Get example FASTA file
fasta_path <- system.file(file.path("extdata", "silva_subset.fa"),
                          package = "metacoder")

# Parse the FASTA file as a taxmap object
obj <- parse_silva_fasta(file = fasta_path)

# Simulate PCR with primersearch
# Have to replace Us with Ts in sequences since primersearch
#   does not understand Us.
obj <- primersearch(obj,
                    gsub(silva_seq, pattern = "U", replace = "T"), 
                    forward = c("U519F" = "CAGYMGCCRCGGKAAHACC"),
                    reverse = c("Arch806R" = "GGACTACNSGGGTMTCTAAT"),
                    mismatch = 10)
                           
# Plot what did not ampilify                          
obj \%>\%
  filter_taxa(prop_amplified < 1) \%>\%
  heat_tree(node_label = taxon_names, 
            node_color = prop_amplified, 
            node_color_range = c("grey", "red", "purple", "green"),
            node_color_trans = "linear",
            node_color_axis_label = "Proportion amplified",
            node_size = n_obs,
            node_size_axis_label = "Number of sequences",
            layout = "da", 
            initial_layout = "re")
} 

}
