% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mmrm-methods.R
\name{stats_anova}
\alias{stats_anova}
\alias{anova.mmrm}
\title{Analysis of Variance for \code{mmrm} Fits}
\usage{
\method{anova}{mmrm}(object, ..., test = TRUE, refit = FALSE)
}
\arguments{
\item{object}{(\code{mmrm})\cr an \code{mmrm} model fit.}

\item{...}{(\code{mmrm})\cr optional \code{mmrm} model fits. If left empty, the
significance of each term in \code{object} will be calculated.}

\item{test}{(\code{flag})\cr indicating whether the output should include
likelihood ratio test (LRT) results comparing the model fits to one
another. Defaults to \code{TRUE}. Ignored if \code{...} is empty.}

\item{refit}{(\code{flag})\cr indicating whether the models should be refitted
with the dataset consisting of their shared set of observations before
performing diagnostics and testing. This is ignored if the models already
share the same dataset. If \code{refit = FALSE} and the models have different
underlying data sets, an error will be thrown. Defaults to \code{FALSE}. Ignored
if \code{...} is empty.}
}
\value{
A data frame with a row for each supplied model. If \code{...} is empty,
this will be the the returned value of \code{\link[=h_anova_single_mmrm_model]{h_anova_single_mmrm_model()}} with
\code{object} supplied as its argument. Otherwise, the resulting data frame will
have the following columns:
\itemize{
\item \code{Model}: the sequence number of the model according to the order in which
the models were supplied to this function.
\item \code{refit}: logical, indicating whether or not the model was refitted. If
the \code{refit} argument was \code{FALSE}, all values will be \code{FALSE}.
\item \code{REML}: logical, indicating whether or not the model was fitted using
restricted maximum likelihood (REML) estimation. If \code{FALSE}, the model was
fitted using maximum likelihood (ML) estimation.
\item \code{n_param}: the number of variance parameters in the model fit, obtained
via \code{\link[=logLik.mmrm_tmb]{logLik.mmrm_tmb()}}.
\item \code{n_coef}: the number of estimated coefficients in the model fit, obtained
via \code{\link[=logLik.mmrm_tmb]{logLik.mmrm_tmb()}}.
\item \code{df}: degrees of freedom of the model fit, obtained via
\code{\link[=logLik.mmrm_tmb]{logLik.mmrm_tmb()}}.
\item \code{AIC}: Akaike's "An Information Criterion" of the model fit, obtained via
\code{\link[stats:AIC]{stats::AIC()}}.
\item \code{BIC}: the Bayesian Information Criterion of the model fit, obtained via
\code{\link[stats:AIC]{stats::BIC()}}.
\item \code{logLik}: the log likelihood of the model fit, obtained via
\code{\link[=logLik.mmrm_tmb]{logLik.mmrm_tmb()}}.
\item \code{call}: the \link{call} that created the model fit, obtained via \code{\link[=component]{component()}}
with \code{name = "call"}, which is passed to \code{\link[=deparse1]{deparse1()}}. If the model was
refitted (i.e., if its \code{refit} entry in this table is \code{TRUE}), this \code{call}
will be different from the \code{call} component of the pre-refitted model fit.
}

The data frame will have these additional columns inserted before \code{call} if
\code{test = TRUE}. Note that since each of these columns describe the results
of a likelihood ratio test (LRT) between the previous row's and current
row's model fits, the first element of each of these columns will be \code{NA}.
\itemize{
\item \code{test}: character, indicating which two model fits were compared. Values
are of the form \verb{\{Model - 1\} vs \{Model\}}.
\item \code{log_likelihood_ratio}: the logarithm of the likelihood ratio between the two
models being compared.
\item \code{p_value}: the p-value of the \code{log_likelihood_ratio}.
}
}
\description{
If supplied only one model fit, the function will calculate and return the
significance of the model terms. If supplied more than one model fit,
standard diagnostics will be returned for each model, optionally including
likelihood ratio test (LRT) results for adjacent models.
}
\details{
When \code{test = FALSE} (or, when only one model is supplied), this
function will process any \code{mmrm} fits, related or unrelated.

When supplying multiple models and \code{test = TRUE}, adjacent pairs of models
are tested sequentially. In other words, the order of the supplied models
matters. Furthermore, there are are multiple requirements for successful
LRT. See the section "Requirements for LRT" below.
}
\section{Requirements for LRT}{
\enumerate{
\item Each supplied model fit must have more degrees of freedom than
the preceding model fit.
\item If all supplied models were estimated using maximum likelihood (ML), the
models must have nested covariates in order to undergo LRT. In other words,
the set of covariates for each model must be a subset of the covariates of
the next model. However, if any of the supplied models were estimated using
restricted maximum likelihood (REML), all models must have the same
covariates.
\item The covariance structure of each model must be either (a) the same as
that of the next model or (b) a special case of that of the next model. See
the section "Covariance structure nesting hierarchy" below.
\item All supplied model fits must either already use the same data or be
refitted using \code{refit = TRUE}, which refits all models to the dataset of
common observations between all models' respective data sets.
}
}

\section{Covariance structure nesting hierarchy}{
\subsection{Structured nests within unstructured}{

Tautologically, all covariance structures are special cases of an
unstructured covariance, and a model \emph{with} a covariance structure can be
considered "nested" within an model \emph{without} a covariance structure
(assuming that the covariates are also nested).
}

\subsection{Homogeneous nests within analogous heterogeneous}{

All homogeneous covariance structures are nested within their corresponding
heterogeneous counterparts. For instance, the homogeneous Toeplitz
covariance structure is nested within the heterogeneous Toeplitz covariance
structure.
}

\subsection{Other nested structures}{

Some different covariance structure types are also nested:
\itemize{
\item First-order auto-regressive (\code{ar1} / \code{ar1h}) is nested within:
\itemize{
\item ante-dependence (\code{ad} / \code{adh})
\item Toeplitz (\code{toep} / \code{toeph})
}
\item Compound symmetry (\code{cs} / \code{csh}) is nested within Toeplitz (\code{toep} / \code{toeph})
}
}
}

\examples{
# Create a few model fits, only adding terms from one to the next.
# Notice also that each covariance structure is a special case of the one
# that follows.
fit_sex_ar1 <-
  mmrm(FEV1 ~ FEV1_BL + SEX + ARMCD + ar1(AVISIT | USUBJID),
    data = fev_data,
    reml = FALSE
  )
fit_sex_race_toeph <-
  mmrm(
    FEV1 ~ FEV1_BL + SEX + RACE + ARMCD + toeph(AVISIT | USUBJID),
    data = fev_data,
    reml = FALSE
  )
fit_interaction_us <-
  mmrm(
    FEV1 ~ FEV1_BL + SEX * RACE + ARMCD + us(AVISIT | USUBJID),
    data = fev_data,
    reml = FALSE
  )

# Single model fit, showing significance of model terms:
anova(fit_interaction_us)

# Multiple model fits, with diagnostics for each fit and likelihood ratio
# testing (LRT) for each adjacent pair. LRT is possible because when the fits
# are in this order, their covariates and covariance structures are nested.
anova(fit_sex_ar1, fit_sex_race_toeph, fit_interaction_us)

# We can only change the order if we forego LRT using test = FALSE.
anova(fit_sex_race_toeph, fit_interaction_us, fit_sex_ar1, test = FALSE)

# Create a subset of fev_data set with the 4th visit removed.
fev_subset <- droplevels(fev_data[fev_data$VISITN < 4, ])

# Recreate fit_sex_race_toeph but this time based off fev_subset:
fit_sex_race_toeph_sub <-
  mmrm(
    FEV1 ~ FEV1_BL + SEX + RACE + ARMCD + toeph(AVISIT | USUBJID),
    data = fev_subset,
    reml = FALSE
  )

# If a model was created with a different data set, refit = TRUE is needed.
anova(fit_sex_ar1, fit_sex_race_toeph_sub, fit_interaction_us, refit = TRUE)
}
\seealso{
For details on the single model operation of this function, see
\code{\link[=h_anova_single_mmrm_model]{h_anova_single_mmrm_model()}}. For details on the generic, see
\code{\link[stats:anova]{stats::anova()}}.
}
