% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/log_likelihood.R
\name{log_likelihood}
\alias{log_likelihood}
\title{Calculate log-likelihood}
\usage{
log_likelihood(
  lambda,
  pop_data,
  curve_params,
  noise_params,
  antigen_isos = get_biomarker_levels(pop_data),
  verbose = FALSE,
  ...
)
}
\arguments{
\item{lambda}{a \link{numeric} vector of incidence parameters,
in events per person-year}

\item{pop_data}{a \code{\link[=data.frame]{data.frame()}} with cross-sectional serology data
by antibody and age, and additional columns}

\item{curve_params}{a \code{\link[=data.frame]{data.frame()}} containing MCMC samples of parameters
from the Bayesian posterior distribution of a longitudinal decay curve model.
The parameter columns must be named:
\itemize{
\item \code{antigen_iso}: a \code{\link[=character]{character()}} vector indicating antigen-isotype
combinations
\item \code{iter}: an \code{\link[=integer]{integer()}} vector indicating MCMC sampling iterations
\item \code{y0}: baseline antibody level at $t=0$ ($y(t=0)$)
\item \code{y1}: antibody peak level (ELISA units)
\item \code{t1}: duration of infection
\item \code{alpha}: antibody decay rate
(1/days for the current longitudinal parameter sets)
\item \code{r}: shape factor of antibody decay
}}

\item{noise_params}{a \code{\link[=data.frame]{data.frame()}} (or \code{\link[tibble:tibble]{tibble::tibble()}})
containing the following variables,
specifying noise parameters for each antigen isotype:
\itemize{
\item \code{antigen_iso}: antigen isotype whose noise parameters are being specified
on each row
\item \code{nu}: biological noise
\item \code{eps}: measurement noise
\item \code{y.low}: lower limit of detection for the current antigen isotype
\item \code{y.high}: upper limit of detection for the current antigen isotype
}}

\item{antigen_isos}{Character vector listing one or more antigen isotypes.
Values must match \code{pop_data}.}

\item{verbose}{logical: if TRUE, print verbose log information to console}

\item{...}{additional arguments passed to other functions
(not currently used).}
}
\value{
the log-likelihood of the data with the current parameter values
}
\description{
Calculates the log-likelihood of a set of cross-sectional antibody response
data, for a given incidence rate (\code{lambda}) value.
}
\examples{
library(dplyr)
library(tibble)

# Load cross-sectional data
xs_data <-
  sees_pop_data_pk_100

# Load curve parameters and subset for the purposes of this example
curve <-
  typhoid_curves_nostrat_100 \%>\%
  filter(antigen_iso \%in\% c("HlyE_IgA", "HlyE_IgG"))

# Load noise params
cond <- tibble(
  antigen_iso = c("HlyE_IgG", "HlyE_IgA"),
  nu = c(0.5, 0.5), # Biologic noise (nu)
  eps = c(0, 0), # M noise (eps)
  y.low = c(1, 1), # low cutoff (llod)
  y.high = c(5e6, 5e6)
) # high cutoff (y.high)

# Calculate log-likelihood
ll_AG <- log_likelihood(
  pop_data = xs_data,
  curve_params = curve,
  noise_params = cond,
  antigen_isos = c("HlyE_IgG", "HlyE_IgA"),
  lambda = 0.1
) \%>\% print()

}
