test_that("calc_performance_metrics calculates correct RMSE, MAPE and coverage", {
  # Create a mock data.table
  date <- as.POSIXct("2021-01-01 00:00:00")

  predictions <- data.table::data.table(
    date = date + (1:10) * 3600 * 24,
    value = 1:10,
    prediction = c(rep(2.5, 5), rep(7.5, 5)),
    prediction_lower = rep(0, 10),
    prediction_upper = rep(10, 10)
  )
  performance_metrics <- calc_performance_metrics(predictions)

  # Check if the cleaned data has the expected structure and content
  expect_vector(performance_metrics, ptype = numeric())
  expect_equal(performance_metrics["RMSE"], c("RMSE" = 1.5))
  expect_equal(performance_metrics["Bias"], c("Bias" = -0.5))
  expect_equal(performance_metrics["Coverage"], c("Coverage" = 1))
})

test_that("calc_summary_statistics calculates correct means", {
  # Create a mock data.table
  date <- as.POSIXct("2021-01-01 00:00:00")

  predictions <- data.table::data.table(
    date = date + (1:10) * 3600 * 24,
    value = c(rep(5, 5), rep(10, 5)),
    prediction = rep(7.5, 10),
    prediction_min = rep(5, 10),
    prediction_max = rep(10, 10)
  )
  summary_statistics <- calc_summary_statistics(predictions)

  # Check if the cleaned data has the expected structure and content
  expect_s3_class(summary_statistics, "data.frame")
  expect_equal(summary_statistics["mean", "true"], 7.5)
  expect_equal(summary_statistics["mean", "prediction"], 7.5)
  expect_equal(summary_statistics["5-percentile", "true"], 5)
  expect_equal(summary_statistics["5-percentile", "prediction"], 7.5)
})

test_that("calc_performance_metrics and calc_summary_statistics correctly exclude the effect date range", {
  # Create a mock data.table
  date <- as.POSIXct("2021-01-01 00:00:00")

  predictions <- data.table::data.table(
    date = date + (1:10) * 3600 * 24,
    value = 1:10,
    prediction = c(rep(2.5, 5), rep(7.5, 5)),
    prediction_min = rep(0, 10),
    prediction_max = rep(10, 10)
  )

  date_effect_start <- date + as.difftime(8, units = "days")
  buffer <- 24
  performance_metrics <- calc_performance_metrics(predictions, date_effect_start, buffer)
  summary_statistics <- calc_summary_statistics(predictions, date_effect_start, buffer)

  expect_equal(summary_statistics["max", ], data.frame(true = 6, prediction = 7.5, row.names = c("max")))
  expect_equal(performance_metrics["MSE"], c("MSE" = 2.25))
})
