% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/G_tuneup.R
\name{G.tuneup}
\alias{G.tuneup}
\title{Tune-up the the genomic relationship matrix G}
\usage{
G.tuneup(
  G = NULL,
  A = NULL,
  blend = FALSE,
  pblend = 0.02,
  bend = FALSE,
  eig.tol = 1e-06,
  align = FALSE,
  rcn = TRUE,
  digits = 8,
  sparseform = FALSE,
  determinant = TRUE,
  message = TRUE
)
}
\arguments{
\item{G}{Input of the genomic matrix \eqn{\boldsymbol{G}} to tune-up
in full form (\eqn{n \times n}) (default = \code{NULL}).}

\item{A}{Input of the matching pedigree relationship matrix \eqn{\boldsymbol{A}}
in full form (\eqn{n \times n}) (default = \code{NULL}).}

\item{blend}{If \code{TRUE} a \emph{blending} with identity matrix \eqn{\boldsymbol{I}} or pedigree relationship matrix
\eqn{\boldsymbol{A}} (if provided) is performed (default = \code{FALSE}).}

\item{pblend}{If blending is requested this is the proportion of the identity matrix \eqn{\boldsymbol{I}} or
pedigree-based relationship matrix \eqn{\boldsymbol{A}} to blend for (default = \code{0.02}).}

\item{bend}{If \code{TRUE} a \emph{bending} is performed by making the matrix near positive definite (default = \code{FALSE}).}

\item{eig.tol}{Defines relative positiveness (\emph{i.e.}, non-zero) of eigenvalues compared to the largest one.
It determines which threshold of eigenvalues will be treated as zero (default = \code{1e-06}).}

\item{align}{If \code{TRUE} the genomic matrix \eqn{\boldsymbol{G}} is \emph{aligned} to the
matching pedigree relationship matrix \eqn{\boldsymbol{A}} (default = \code{FALSE}).}

\item{rcn}{If \code{TRUE} the reciprocal conditional number of the original and the bended,
blended or aligned matrix will be calculated (default = \code{TRUE}).}

\item{digits}{Set up the number of digits used to round the output matrix (default = \code{8}).}

\item{sparseform}{If \code{TRUE} it generates an inverse matrix in sparse form to be used directly in \pkg{asreml} with
required attributes (default = \code{FALSE}).}

\item{determinant}{If \code{TRUE} the determinant will be calculated, otherwise, this is obtained for
matrices of a dimension of less than 1,500 \eqn{\times} 1,500 (default = \code{TRUE}).}

\item{message}{If \code{TRUE} diagnostic messages are printed on screen (default = \code{TRUE}).}
}
\value{
A list with six of the following elements:
\itemize{
\item \code{Gb}: the inverse of \eqn{\boldsymbol{G}} matrix in full form
(only if sparseform = \code{FALSE}).
\item \code{Gb.sparse}: if requested, the inverse of \eqn{\boldsymbol{G}} matrix in
sparse form (only if sparseform = \code{TRUE}).
\item \code{rcn0}: the reciprocal conditional number of the original matrix.
Values near zero are associated with an ill-conditioned matrix.
\item \code{rcnb}: the reciprocal conditional number of the blended, bended or aligned
matrix. Values near zero are associated with an ill-conditioned matrix.
\item \code{det0}: if requested, the determinant of the original matrix.
\item \code{blend}: if the matrix was \emph{blended}.
\item \code{bend}: if the matrix was \emph{bended}.
\item \code{align}: if the matrix was \emph{aligned}.
}
}
\description{
Generates a new matrix that can be \emph{blended}, \emph{bended} or \emph{aligned}
in order to make it stable for future use or matrix inversion. The input matrix should
be of the full form (\eqn{n \times n}) with individual names assigned to
\code{rownames} and \code{colnames}.

This routine provides three options of tune-up:
\itemize{
 \item{\emph{Blend}}{. The \eqn{\boldsymbol{G}} matrix is blended (or averaged) with another matrix.
 \eqn{\boldsymbol{G}^\ast=(1-p) \boldsymbol{G} + p \boldsymbol{A}}, where
 \eqn{\boldsymbol{G}^\ast} is the blended matrix, \eqn{\boldsymbol{G}}
 is the original matrix,  \eqn{p} is the
 proportion of the identity matrix or pedigree-based relationship matrix to consider,
 and \eqn{\boldsymbol{A}} is the matrix to blend.
 Ideally, the pedigree-based relationship matrix should be used, but if this is not available (or it is
 of poor quality), then it is replaced by an identity matrix \eqn{\boldsymbol{I}}.}
 \item{\emph{Bend}}{. It consists on adjusting the original \eqn{\boldsymbol{G}} matrix to obtain a
 near positive definite matrix, which is done by making all negative or very small eigenvalues
 slightly positive.}
 \item{\emph{Align}}{. The original \eqn{\boldsymbol{G}} matrix is aligned to the
 matching pedigree relationship
 matrix \eqn{\boldsymbol{A}} where an \eqn{\alpha} and \eqn{\beta} parameters are obtained.
 More information can be found in the manual or in Christensen \emph{et al.} (2012).}
}

The user should provide the matrices \eqn{\boldsymbol{G}} and
\eqn{\boldsymbol{A}} in full form (\eqn{n \times n})
and matching individual names should be
assigned to the \code{rownames} and \code{colnames} of the matrices.

Based on procedures published by Nazarian and Gezan \emph{et al.} (2016).
}
\examples{
# Example: Apple dataset.

# Get G matrix.
G <- G.matrix(M = geno.apple, method = "VanRaden")$G
G[1:5, 1:5]

# Blend G matrix.
G_blended <- G.tuneup(G = G, blend = TRUE, pblend = 0.05)
G_blended$Gb[1:5, 1:5]

# Bend G matrix.
G_bended <- G.tuneup(G = G, bend = TRUE, eig.tol = 1e-03)
G_bended$Gb[1:5, 1:5]

\donttest{
# Example: Loblolly Pine dataset with pedigree - Aligned G matrix.

A <- AGHmatrix::Amatrix(ped.pine)
dim(A)

# Read and filter genotypic data.
M.clean <- qc.filtering(
 M = geno.pine655,
 maf = 0.05,
 marker.callrate = 0.2, ind.callrate = 0.20,
 na.string = "-9")$M.clean

# Get G matrix.
G <- G.matrix(M = M.clean, method = "VanRaden", na.string = "-9")$G
G[1:5, 1:5]
dim(G)

# Match G and A.
Aclean <- match.G2A(A = A, G = G, clean = TRUE, ord = TRUE, mism = TRUE)$Aclean

# Align G with A.
G_align <- G.tuneup(G = G, A = Aclean, align = TRUE)
G_align$Gb[1:5, 1:5]
}

}
\references{
Christensen, O.F., Madsen, P., Nielsen, B., Ostersen, T. and Su, G. 2012.
Single-step methods for genomic evaluation in pigs. Animal 6:1565-1571.
doi:10.1017/S1751731112000742.

Nazarian A., Gezan S.A. 2016. GenoMatrix: A software package for pedigree-based
and genomic prediction analyses on complex traits. Journal of Heredity 107:372-379.
}
