% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/file_client_funcs.R
\name{list_azure_files}
\alias{list_azure_files}
\alias{upload_azure_file}
\alias{multiupload_azure_file}
\alias{download_azure_file}
\alias{multidownload_azure_file}
\alias{delete_azure_file}
\alias{create_azure_dir}
\alias{delete_azure_dir}
\alias{azure_file_exists}
\title{Operations on a file share}
\usage{
list_azure_files(share, dir = "/", info = c("all", "name"), prefix = NULL,
  recursive = FALSE)

upload_azure_file(share, src, dest = basename(src), create_dir = FALSE,
  blocksize = 2^22, use_azcopy = FALSE)

multiupload_azure_file(share, src, dest, recursive = FALSE,
  create_dir = recursive, blocksize = 2^22, use_azcopy = FALSE,
  max_concurrent_transfers = 10)

download_azure_file(share, src, dest = basename(src), blocksize = 2^22,
  overwrite = FALSE, use_azcopy = FALSE)

multidownload_azure_file(share, src, dest, recursive = FALSE,
  blocksize = 2^22, overwrite = FALSE, use_azcopy = FALSE,
  max_concurrent_transfers = 10)

delete_azure_file(share, file, confirm = TRUE)

create_azure_dir(share, dir, recursive = FALSE)

delete_azure_dir(share, dir, recursive = FALSE, confirm = TRUE)

azure_file_exists(share, file)
}
\arguments{
\item{share}{A file share object.}

\item{dir, file}{A string naming a directory or file respectively.}

\item{info}{Whether to return names only, or all information in a directory listing.}

\item{prefix}{For \code{list_azure_files}, filters the result to return only files and directories whose name begins with this prefix.}

\item{recursive}{For the multiupload/download functions, whether to recursively transfer files in subdirectories. For \code{list_azure_dir}, whether to include the contents of any subdirectories in the listing. For \code{create_azure_dir} and \code{delete_azure_dir}, whether to recursively create/delete each component of a nested directory path. Note that in all cases this can be slow, so try to use a non-recursive solution if possible.}

\item{src, dest}{The source and destination files for uploading and downloading. See 'Details' below.}

\item{create_dir}{For the uploading functions, whether to create the destination directory if it doesn't exist. Again for the file storage API this can be slow, hence is optional.}

\item{blocksize}{The number of bytes to upload/download per HTTP(S) request.}

\item{use_azcopy}{Whether to use the AzCopy utility from Microsoft to do the transfer, rather than doing it in R.}

\item{max_concurrent_transfers}{For \code{multiupload_azure_file} and \code{multidownload_azure_file}, the maximum number of concurrent file transfers. Each concurrent file transfer requires a separate R process, so limit this if you are low on memory.}

\item{overwrite}{When downloading, whether to overwrite an existing destination file.}

\item{confirm}{Whether to ask for confirmation on deleting a file or directory.}
}
\value{
For \code{list_azure_files}, if \code{info="name"}, a vector of file/directory names. If \code{info="all"}, a data frame giving the file size and whether each object is a file or directory.

For \code{download_azure_file}, if \code{dest=NULL}, the contents of the downloaded file as a raw vector.

For \code{azure_file_exists}, either TRUE or FALSE.
}
\description{
Upload, download, or delete a file; list files in a directory; create or delete directories; check file existence.
}
\details{
\code{upload_azure_file} and \code{download_azure_file} are the workhorse file transfer functions for file storage. They each take as inputs a \emph{single} filename as the source for uploading/downloading, and a single filename as the destination. Alternatively, for uploading, \code{src} can be a \link{textConnection} or \link{rawConnection} object; and for downloading, \code{dest} can be NULL or a \code{rawConnection} object. If \code{dest} is NULL, the downloaded data is returned as a raw vector, and if a raw connection, it will be placed into the connection. See the examples below.

\code{multiupload_azure_file} and \code{multidownload_azure_file} are functions for uploading and downloading \emph{multiple} files at once. They parallelise file transfers by using the background process pool provided by AzureRMR, which can lead to significant efficiency gains when transferring many small files. There are two ways to specify the source and destination for these functions:
\itemize{
\item Both \code{src} and \code{dest} can be vectors naming the individual source and destination pathnames.
\item The \code{src} argument can be a wildcard pattern expanding to one or more files, with \code{dest} naming a destination directory. In this case, if \code{recursive} is true, the file transfer will replicate the source directory structure at the destination.
}

\code{upload_azure_file} and \code{download_azure_file} can display a progress bar to track the file transfer. You can control whether to display this with \code{options(azure_storage_progress_bar=TRUE|FALSE)}; the default is TRUE.
}
\section{AzCopy}{

\code{upload_azure_file} and \code{download_azure_file} have the ability to use the AzCopy commandline utility to transfer files, instead of native R code. This can be useful if you want to take advantage of AzCopy's logging and recovery features; it may also be faster in the case of transferring a very large number of small files. To enable this, set the \code{use_azcopy} argument to TRUE.

Note that AzCopy only supports SAS and AAD (OAuth) token as authentication methods. AzCopy also expects a single filename or wildcard spec as its source/destination argument, not a vector of filenames or a connection.
}

\examples{
\dontrun{

share <- file_share("https://mystorage.file.core.windows.net/myshare", key="access_key")

list_azure_files(share, "/")
list_azure_files(share, "/", recursive=TRUE)

create_azure_dir(share, "/newdir")

upload_azure_file(share, "~/bigfile.zip", dest="/newdir/bigfile.zip")
download_azure_file(share, "/newdir/bigfile.zip", dest="~/bigfile_downloaded.zip")

delete_azure_file(share, "/newdir/bigfile.zip")
delete_azure_dir(share, "/newdir")

# uploading/downloading multiple files at once
multiupload_azure_file(share, "/data/logfiles/*.zip")
multidownload_azure_file(share, "/monthly/jan*.*", "/data/january")

# you can also pass a vector of file/pathnames as the source and destination
src <- c("file1.csv", "file2.csv", "file3.csv")
dest <- paste0("uploaded_", src)
multiupload_azure_file(share, src, dest)

# uploading serialized R objects via connections
json <- jsonlite::toJSON(iris, pretty=TRUE, auto_unbox=TRUE)
con <- textConnection(json)
upload_azure_file(share, con, "iris.json")

rds <- serialize(iris, NULL)
con <- rawConnection(rds)
upload_azure_file(share, con, "iris.rds")

# downloading files into memory: as a raw vector, and via a connection
rawvec <- download_azure_file(share, "iris.json", NULL)
rawToChar(rawvec)

con <- rawConnection(raw(0), "r+")
download_azure_file(share, "iris.rds", con)
unserialize(con)

}
}
\seealso{
\link{file_share}, \link{az_storage}, \link{storage_download}, \link{call_azcopy}

\href{https://github.com/Azure/azure-storage-azcopy}{AzCopy version 10 on GitHub}
}
