% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dlm_close.R
\name{dlm_close}
\alias{dlm_close}
\title{DLM: Fitting the advanced ages of the life tables.}
\usage{
dlm_close(fit, method = c("linear", "gompertz", "plateau"),
          x0 = max(fit$info$ages), max_age = 120, k = 7,
          weights = seq(from = 0, to = 1, length.out = k),
          new_data = NULL)
}
\arguments{
\item{fit}{Object of the class \code{DLM} adjusted by the dlm() function.}

\item{method}{Character string specifying the closing method to be fitted, with them being: 'plateau', 'linear' or 'gompertz'.}

\item{x0}{Integer with the starting age the closing method will be fitted from. Default is the last age fitted by the 'DLM' object.}

\item{max_age}{Integer with the maximum age the closing method will be fitted. Default age is '120'.}

\item{k}{Integer representing the size of the age-interval to be mixed with the 'linear' or 'gompertz' closing methods for a smooth graduation. If k = 0, no mixing will be made. Default: 7.}

\item{weights}{Vector of weights to be applied in the mixing of the life tables. Vector's size should be equal to k.}

\item{new_data}{Vector containing the log mortality rates in the period after the x0 input. This argument is necessary in the 'linear' and 'gompertz' closing methods.}
}
\value{
Returns a \code{ClosedDLM} class object with the predictive chains of the death probability
(qx) from first fitted age to max_age argument, the data information utilized by the function and the
closing method chosen.
}
\description{
This function receives an object of the class \code{DLM} fitted by the dlm() function
and fits a closing method to expand the life tables dataset to a maximum age argument inputed
by the user.
There are three closing methods available: 'linear', 'gompertz' and 'plateau'.
}
\details{
The three closing methods implemented by the function are:
1.'linear' method: Fits a linear regression starting at age x0 - k until the last age with data available
2.'gompertz' method: Used as the closing method of the 2010-2012 English Life Table No. 17, fits the Gompertz mortality law via SIR using the same available data as the 'linear' method.
3.'plateau' method: Keeps the death probability (qx) constant after the x0 argument.
}
\examples{
## Importing mortality data from the USA available on the Human Mortality Database (HMD):
data(USA)

## Selecting the exposure and the death count of the year 2010, ranging from 0 to 90 years old:
USA2010 = USA[USA$Year == 2010,]
x = 0:100
Ex = USA2010$Ex.Male[x+1]
Dx = USA2010$Dx.Male[x+1]

qx_t <- Dx/Ex
qx_t <- 1 - exp(-qx_t)
y <- log(qx_t)

fit <- dlm(y, M = 100, bn = 20, thin = 1)

## Applying the closing function with different methods:
close1 = dlm_close(fit, method = "plateau")

### Getting new data for the linear and gompertz methods:::
x2 = 101:110
Ex2 = USA2010$Ex.Male[x2+1]
Dx2 = USA2010$Dx.Male[x2+1]

qx2_t <- Dx2/Ex2
qx2_t <- 1 - exp(-qx2_t)
y2 <- log(qx2_t)

close2 = dlm_close(fit, method = "linear",
                  new_data = y2)

#### Using the other functions available in the package with the 'ClosedDLM' object:

## credible intervals (See "?qx_ci" for more options):
qx_ci(close1)

## qx estimation (See "?fitted" in the BayesMortalityPlus package for more options):
fitted(close2)

## life expectancy (See "?expectancy.DLM" for more options)
expectancy(close2, age = seq(0,120,by=20), graph = FALSE)

## plotting (See "?plot" in the BayesMortalityPlus package for more options):
plot(list(close1, close2, fit),
     colors = c("red4","seagreen", "blue"),
     labels = c("Plateau method","Linear method", "DLM fitted"),
     plotData = FALSE)


}
\references{
Dodd, Erengul, Forster, Jonathan, Bijak, Jakub, & Smith, Peter 2018. “Smoothing mortality data: the English life table, 2010-12.” \emph{Journal of the Royal Statistical Society: Series A (Statistics in Society)}, 181(3), 717-735.
}
\seealso{
\code{\link[=fitted.DLM]{fitted.DLM()}}, \code{\link[=plot.DLM]{plot.DLM()}}, \code{\link[=print.DLM]{print.DLM()}} and \code{\link[=summary.DLM]{summary.DLM()}} for \code{ClosedDLM} methods to native R functions \code{\link[=fitted]{fitted()}},
\code{\link[=plot]{plot()}}, \code{\link[=print]{print()}} and \code{\link[=summary]{summary()}}.

\code{\link[=expectancy.DLM]{expectancy.DLM()}} and \code{\link[=Heatmap.DLM]{Heatmap.DLM()}} for \code{ClosedDLM} methods to compute and visualise the truncated life expectancy
via \code{\link[=expectancy]{expectancy()}} and \code{\link[=Heatmap]{Heatmap()}} functions.

\code{\link[=qx_ci]{qx_ci()}} to compute credible intervals.
}
