% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MEM.R
\name{MEM}
\alias{MEM}
\title{Modal EM algorithm (MEM)}
\usage{
MEM(
  mcmc,
  data,
  pars_names,
  dist = "NA",
  pdf_func = NULL,
  tol_x = sd(data)/10,
  tol_conv = 1e-08,
  show_plot = FALSE
)
}
\arguments{
\item{mcmc}{Vector of estimated mixture parameters.}

\item{data}{Vector of observations used for estimating the mixture.}

\item{pars_names}{Names of the mixture parameters; the first element of 
this vector should be the name of the mixture proportions. If you have used 
the skew normal of Azzalini, then the second element should correspond to the location,
the third to the scale and the fourth to the shape.}

\item{dist}{String indicating the distribution of the mixture components; default is "NA".
Currently supports "normal" and "skew_normal"; not needed if pdf_func is provided.}

\item{pdf_func}{Pdf of the mixture components associated with the mcmc draws
(if mcmc estimation has not been carried out with BayesMultiMode); default is null.}

\item{tol_x}{Tolerance parameter for distance in-between modes; default is sd(data)/10; if two modes are closer than \code{tol_x}, only the first estimated mode is kept.}

\item{tol_conv}{Tolerance parameter for convergence of the algorithm; default is 1e-8.}

\item{show_plot}{If true show the data and estimated modes; default is false}
}
\value{
Vector of estimated modes.
}
\description{
Algorithm from Li and Lindsay (2007) to find modes in mixture of continuous distributions.
}
\details{
This algorithm returns the local maxima of the mixture
\deqn{p(x) = \sum_{k=1}^{K}\pi_k p_k(x),}
where \eqn{p_k} is a density function.
Following Li and Lindsay (2007), a mode \eqn{x} is found by iterating the two steps:
\deqn{(i) \quad p(k|x^{(n)}) = \frac{\pi_k p_k(x^{(n)})}{p(x^{(n)})},}
\deqn{(ii) \quad x^{(n+1)} = \text{argmax}_x  \sum_k p(k|x) \text{log} p_k(x^{(n)}),}
until convergence, that is, until \eqn{abs(x^{(n+1)}-x^{(n)})< \text{tol}_\text{conv}},
where \eqn{\text{tol}_\text{conv}} is an argument with default value \eqn{1e-8}.
The algorithm is started at each component location.
Separately, it is necessary to identify identical modes which diverge only up to
a small value. By default modes which are closer
than \eqn{sd(y)/10} are merged; this tolerance value can be controlled with the argument
\code{tol_x}.

While it is also possible to use the MEM algorithm for Normal mixtures, 
this is not recommended because the algorithm is less efficient than the
fixed-point method in this particular case.
}
\examples{

# Example with the skew normal =============================================
xi = c(0,6)
omega = c(1,2)
alpha = c(0,0)
p = c(0.8,0.2)
params = c(eta = p, xi = xi, omega = omega, alpha = alpha)
pars_names = c("eta", "xi", "omega", "alpha")
dist = "skew_normal"

data = c(sn::rsn(p[1]*100, xi[1], omega[1], alpha[1]),
         sn::rsn(p[2]*100, xi[2], omega[2], alpha[2]))

modes = MEM(params, data = data, pars_names = pars_names, dist = dist)

# Example with an arbitrary distribution ===================================
xi = c(0,6)
omega = c(1,2)
alpha = c(0,0)
nu = c(3,100)
p = c(0.8,0.2)
params = c(eta = p, mu = xi, sigma = omega, xi = alpha, nu = nu)
pars_names = c("eta", "mu", "sigma", "xi", "nu")

pdf_func <- function(x, pars) {
  sn::dst(x, pars["mu"], pars["sigma"], pars["xi"], pars["nu"])
}

data = c(sn::rst(p[1]*100, xi[1], omega[1], alpha = alpha[1], nu = nu[1]),
         sn::rst(p[2]*100, xi[2], omega[2], alpha = alpha[2], nu = nu[2]))

modes = MEM(params, pars_names = pars_names, data = data, pdf_func = pdf_func)

}
\references{
\insertRef{li_nonparametric_2007}{BayesMultiMode}\cr\cr
}
