% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SimulatedDataGenerator.r
\name{SimulatedDataGenerator}
\alias{SimulatedDataGenerator}
\title{Generate simulated data with either ARMA or MCD correlation structures.}
\usage{
SimulatedDataGenerator(
  Num.of.Obs,
  Num.of.TimePoints,
  Fixed.Effs,
  Random.Effs,
  Cor.in.DesignMat,
  Missing,
  Cor.Str,
  HSD.DesignMat.para,
  ARMA.para
)
}
\arguments{
\item{Num.of.Obs}{the number of subjects will be simulated.}

\item{Num.of.TimePoints}{the maximum number of time points among all subjects.}

\item{Fixed.Effs}{a vector of regression coefficients.}

\item{Random.Effs}{a list of covariance matrix and the degree of freedom,  e.g., \code{list(Sigma = 0.5*diag(1), df=3)}.}

\item{Cor.in.DesignMat}{the correlation of covariates in the design matrix.}

\item{Missing}{a list of the missing mechanism of observations, 0: data is complete, 1: missing complete at random, 2: missing at random related to responses , and 3: 2: missing at random related to covariates and the corresponding regression coefficients (weights) on the previous observed values either responses or covariates, e.g.,  \code{Missing = list(Missing.Mechanism = 3, RegCoefs = c(0.4, 1.2, -2.1))}.}

\item{Cor.Str}{the model for correlation structure, \verb{ARMA'' or }HSD''.}

\item{HSD.DesignMat.para}{if \code{Cor.Str="HSD"}, you need to specify the list of parameters in HSD correlation structure, e.g., \code{HSD.DesignMat.para = list(HSD.para = HSD.para, DesignMat = w)}.}

\item{ARMA.para}{if \code{Cor.Str="ARMA"}, you need to specify the list of parameters in AMRA correlation structure, e.g., \code{ARMA.para = list(AR.para=0.1, MA.para=0.2)}.}
}
\value{
a list containing the following components:
\describe{
\item{sim.data}{The simulated response variables \eqn{y}, covariates \eqn{x}'s, and subject identifcation \samp{id}.}
\item{beta.true}{The given values of fixed effects .}
\item{ARMA.para}{The given values of parameters in ARMA model.}
\item{HSD.para}{The given values of parameters in ARMA model.}
}
}
\description{
This function is used to generate simulated data for simulation studies with ARMA and MCD correlation structures.
}
\examples{
library(BayesRGMM)
rm(list=ls(all=TRUE))
Fixed.Effs = c(-0.2, -0.3, 0.8, -0.4) #c(-0.2,-0.8, 1.0, -1.2)
P = length(Fixed.Effs) 
q = 1 #number of random effects
T = 5 #time points
N = 100 #number of subjects
num.of.iter = 100 #number of iterations
HSD.para = c(-0.5,  -0.3) #the parameters in HSD model
a = length(HSD.para)
w = array(runif(T*T*a), c(T, T, a)) #design matrix in HSD model

for(time.diff in 1:a)
	w[, , time.diff] = 1*(as.matrix(dist(1:T, 1:T, method="manhattan")) ==time.diff)

#Generate a data with HSD model
HSD.sim.data = SimulatedDataGenerator(Num.of.Obs = N, Num.of.TimePoints = T, 
Fixed.Effs = Fixed.Effs, Random.Effs = list(Sigma = 0.5*diag(1), df=3), 
Cor.in.DesignMat = 0., Missing = list(Missing.Mechanism = 2, RegCoefs = c(-1.5, 1.2)), 
Cor.Str = "HSD", HSD.DesignMat.para = list(HSD.para = HSD.para, DesignMat = w))

#the proportion of 1's
print(sum(HSD.sim.data$sim.data$y==1, na.rm=T)/sum(!is.na(HSD.sim.data$sim.data$y))) 

#the missing rate in the simulated data 
print(sum(is.na(HSD.sim.data$sim.data$y)))


#=====================================================================================#
#Generate a data with ARMA model
ARMA.sim.data = SimulatedDataGenerator(Num.of.Obs = N, Num.of.TimePoints = T, 
	Fixed.Effs = Fixed.Effs, Random.Effs = list(Sigma = 0.5*diag(1), df=3), 
	Cor.in.DesignMat = 0., list(Missing.Mechanism = 2, RegCoefs = c(-1.5, 1.2)), 
	Cor.Str = "ARMA", ARMA.para=list(AR.para = 0.8))
}
