### Support functions for BigVAR Package
### These are mostly utility functions that will not be seen by the user


                                        # mean benchmark
.evalMean <- function(Y,T1,T2,h=1)
{

    if(class(Y)!="matrix"){
        Y <- matrix(Y,ncol=1)
    }

    MSFE <- c()

    k <- ncol(Y)

    for (u in (T1-h+2):T2) {

        if(h+u-1>T2){break}
        
        trainY1 <- Y[1:(u-1), ]

        if(k>1){

            ypred <- colMeans(trainY1)
        }else{
            ypred <- mean(trainY1)
        }

        uhat <- matrix(Y[u+h-1, ] - ypred, 
                       ncol = k)

        MSFE <- c(MSFE,norm2(uhat)^2)

    }
    return(list(Mean=mean(na.omit(MSFE)),SD=sd(na.omit(MSFE))/sqrt(length(na.omit(MSFE)))))
}

                                        # random walk benchmark
.evalRW <- function(Y,T1,T2,h=1)
{

    if(class(Y)!="matrix"){
        Y <- matrix(Y,ncol=1)
    }
    
    MSFE <- c()
    
    k <- ncol(Y)
    
    for (u in (T1-h+2):T2) {
        
        if(h+u-1>T2){break}
        
        trainY1 <- Y[u-1, ]
        
        uhat <- matrix(Y[u+h-1, ] - trainY1, 
                       ncol = k)
        
        MSFE <- c(MSFE,norm2(uhat)^2)
    }
    return(list(Mean=mean(na.omit(MSFE)),SD=sd(na.omit(MSFE))/sqrt(length(na.omit(MSFE)))))
}

                                        # Constructs the Grid of Lambda Values: VAR
.LambdaGridE<- function (gran1, gran2, jj = jj, Y, Z, group,p,k,MN,alpha,C) 
{

    if (group == "Lag") {
        mat <- list()
        for (i in 1:length(jj)) {
            if(k>1){
                mat[[i]] <- norm2(Z[jj[[i]], ]%*%Y)
            }

            else{
                mat[[i]] <- norm2(t(Y)%*%Z[jj[[i]],])
            }
        }
        gamstart <- max(unlist(mat))
    }
    if (group == "Basic"|group=="Tapered") {

        gamstart <- max(t(Y) %*% t(Z))

    }

    if (group == "SparseLag") {

        mat <- list()

        if(alpha>0){
            for (i in 1:length(jj)) {

                if(k>1){

                    mat[[i]] <- norm2(Z[jj[[i]], ]%*%Y)*(1/(alpha))

                }

                else{mat[[i]] <- norm2(t(Y)%*%Z[jj[[i]],])

                 }

            }

            gamstart <- max(unlist(mat))

        }else{
            gamstart <- max(t(Y) %*% t(Z))

        }
    }

    if (group == "OwnOther") {

        mat <- list()

        ZZ <- kronecker(t(Z), diag(k))

        for (i in 1:length(jj)) {

            mat[[i]] <- norm(as.vector(t(Y)) %*% ZZ[, jj[[i]]]/sqrt(length(jj[[i]])), 
                             "F")

        }

        gamstart <- max(unlist(mat))

    }
    if (group == "SparseOO") {

        mat <- list()

        ZZ <- kronecker(t(Z), diag(k))

        if(alpha>0){
            for (i in 1:length(jj)) {

                mat[[i]] <- norm(1/(k + 1) * as.vector(t(Y)) %*% 
                                 ZZ[, jj[[i]]]/sqrt(length(jj[[i]])), "F")

            }

            gamstart <- max(unlist(mat))

        }else{
            gamstart <- max(t(Y) %*% t(Z))

        }
    }

    if (group == "HVARC"|group=="HVAROO"|group=="HVARELEM") {

        gmax <- c()

        for (i in 1:k) {

            gmax[i]  <- norm2(Z %*% Y[, i])

        }

        gamstart <- max(gmax)

    }

    if(group=="Tapered"){

        beta <- array(0,dim=c(k,k*p+1,10))

    }else{
        beta <- array(0,dim=c(k,k*p+1,1))
    }

    gamstart <- LGSearch(gamstart,Y,Z,beta,group,k,p,jj,MN,alpha,C)
    
    gamm <- exp(seq(from = log(gamstart), to = log(gamstart/gran1), 
                    length = gran2))

    return(gamm)

}
                                        # Constructs penalty grid for each value of alpha in case of dual cv
.LambdaGridEDual <- function(gran1,gran2,jj,Y,Z,group,p,k,MN,alpha,C){

    Lambda <- matrix(0,nrow=gran2,ncol=length(alpha))
    
    for(i in 1:length(alpha)){

        Lambda[,i] <- .LambdaGridE(gran1,gran2,jj,Y,Z,group,p,k,MN,alpha[i],C)

    }
    return(Lambda)

    


}

                                        # Construct Lambda Grid: VARX
.LambdaGridX<- function (gran1, gran2, jj = jj, Y, Z, group,p,k1,s,m,k,MN,alpha,C) 
{

    if (group == "Lag") {

        mat  <- list()

        for (i in 1:length(jj)) {

            if(k>1){

                mat[[i]] <- norm2(Z[jj[[i]], ]%*%Y)

            }

            else{
                mat[[i]] <- norm2(t(Y)%*%Z[jj[[i]],])
            }

        }

        gamstart <- max(unlist(mat))

    }

    if (group == "Basic") {

        gamstart <- max(t(Y)%*%t(Z))

    }

    if (group == "SparseLag") {

        mat  <- list()

        if(alpha>0){
            
            for (i in 1:length(jj)) {

                if(k>1){

                    mat[[i]] <- norm2(Z[jj[[i]], ]%*%Y*(1/(alpha)))

                }

                else{

                    mat[[i]] <- norm2(t(Y)%*%Z[jj[[i]],])

                }

            }

            gamstart <- max(unlist(mat))

        }else{

            gamstart <- max(t(Y)%*%t(Z))




        }


    }

    if (group == "OwnOther") {

        mat <- list()

        ZZ <- kronecker(t(Z), diag(k1))

        for (i in 1:length(jj)) {

            mat[[i]] <- norm(as.vector(t(Y)) %*% ZZ[, jj[[i]]]/sqrt(length(jj[[i]])), 
                             "F")

        }

        gamstart <- max(unlist(mat))

    }

    if (group == "SparseOO") {

        mat <- list()

        ZZ <- kronecker(t(Z), diag(k1))

        if(alpha>0){
            for (i in 1:length(jj)) {

                mat[[i]] <- norm(1/(alpha) * as.vector(t(Y)) %*% 
                                 ZZ[, jj[[i]]]/sqrt(length(jj[[i]])), "F")
            }

            gamstart <- max(unlist(mat))

        }else{

            gamstart <- max(t(Y)%*%t(Z))

        }

    }

    if (group=="EFX") {

        gmax <- c()

        for (i in 1:k1) {

            gmax[i] <- norm2(Z %*% Y[, i])/sqrt(k*p)

        }
        
        gamstart <- max(gmax)

    }
    

    beta <- array(0,dim=c(k1,k1*p+s*m+1,1))
    gamstart <- LGSearchX(gamstart,Y,Z,beta,group,k1,p,s,m,jj,k,MN,alpha,C)
    gamm <- exp(seq(from = log(gamstart), to = log(gamstart/gran1), 
                    length = gran2))

    return(gamm)

}

.LambdaGridXDual <- function(gran1,gran2,jj,Y,Z,group,p,k1,s,m,k,MN,alpha,C){

    Lambda <- matrix(0,nrow=gran2,ncol=length(alpha))
    for(i in 1:length(alpha)){

        Lambda[,i] <- .LambdaGridX(gran1,gran2,jj,Y,Z,group,p,k1,s,m,k,MN,alpha[i],C)

    }
    return(Lambda)

    

}

                                        # Forecast evaluation: VARX (called in cv.bigvar)
.BigVAREVALX <- function(ZFull,gamopt,k,p,group,h,MN,verbose,RVAR,palpha,T2,T,k1,s,m,contemp,alpha,C)
    {

        if(contemp){
            s1 <- 1

        }else{

            s1 <- 0

        }

        preds <- matrix(NA,nrow=length((T2+1):T),ncol=k1)

        gran2 <- 1	

        gamm <- gamopt

        Y <- ZFull$Y

        MSFE <- rep(NA,length((T2+1):T))

        beta <- array(0,dim=c(k1,k1*p+(k-k1)*(s+s1)+1,1))  

        if (group == "Lag") {

            jj <- groupfunVARX(p,k,k1,s+s1)

            jjcomp <- groupfunVARXcomp(p,k,k1,s+s1)

            activeset <- rep(list(rep(rep(list(0), length(jj)))), 
                             gran2)

        }

        if (group == "SparseLag") {

            jj <- groupfunVARX(p, k,k1,s+s1)

            q1a <- list()

            for (i in 1:(p+s+s1)) {

                q1a[[i]] <- matrix(runif(k1, -1, 1), ncol = 1)

            }

            activeset <- rep(list(rep(rep(list(0), length(jj)))), 
                             gran2)


        }

        if (group == "OwnOther") {

            kk <- diaggroupfunVARX(p, k,k1,s+s1)
            activeset <- rep(list(rep(rep(list(0), length(kk)))), 
                             gran2)
        }

        if (group == "SparseOO") {

            kk <- diaggroupfunVARX(p, k,k1,s+s1)

            activeset <- rep(list(rep(rep(list(0), length(kk)))), 
                             gran2)
        }


        if(verbose){

            print("Evaluation Stage")

            pb <- txtProgressBar(min = T2-h+2, max = T, style = 3)
        }

        for (v in (T2-h+2):T) {

            if(v+h-1>T){
                break
            }

            trainY <- ZFull$Y[h:(v-1), ]

            trainZ <- ZFull$Z[,1:(v-h)]

            if (group == "Basic") {

                beta <- .lassoVARFistX(beta, trainZ, trainY,gamm, 1e-6,p,MN,k,k1,s+s1,m,C)
            }

            if (group == "Lag") {

                GG <- .GroupLassoVAR1(beta,jj,jjcomp,trainY,trainZ,gamm,activeset,1e-04,p,MN,k,k1,s+s1,C)
                

                beta <- GG$beta

                activeset <- GG$active

            }

            if (group == "SparseLag") {

                GG <- .SparseGroupLassoVARX(beta, jj, trainY, trainZ, 
                                            gamm, alpha, INIactive = activeset, 1e-04, q1a,p,MN,k,s+s1,k1,C)

                beta <- GG$beta

                activeset <- GG$active

                q1a <- GG$q1

            }

            if (group == "OwnOther") {

                GG <- .GroupLassoOOX(beta, kk, trainY, trainZ, gamm, 
                                     activeset, 1e-04,p,MN,k,k1,s+s1,C)

                beta <- GG$beta

                activeset <- GG$active

            }
            if (group == "SparseOO") {
                

                GG <- .SparseGroupLassoVAROOX(beta, kk, trainY, trainZ, 
                                              gamm, alpha, INIactive = activeset, 1e-04,p,MN,k1,s+s1,k,FALSE,C)

                beta <- GG$beta

                activeset <- GG$active

            }


            if(group=="EFX"){
                
                beta <- .EFVARX(beta,trainY,trainZ,gamm,1e-4,MN,k1,s,m,p,C)
            }
            
            betaEVAL <- matrix(beta[,,1],nrow=k1,ncol=(k1*p+(k-k1)*(s+s1)+1))

            if (RVAR) {

                betaEVAL <- RelaxedLS(cbind(t(trainZ),trainY),betaEVAL,k,p,k1,s+s1)
                
            }
            

            if(MN){

                eZ <- matrix(ZFull$Z[,v],ncol=1)

            }else{

                eZ <- matrix(c(1,ZFull$Z[,v]),ncol=1)

            }

            if(MN){

                MSFE[v-T2+h-1] <- norm2(ZFull$Y[v+h-1,1:k1] - betaEVAL[,2:ncol(betaEVAL)] %*% eZ)^2

                preds[v-T2+h-1,] <-  betaEVAL[,2:ncol(betaEVAL)] %*% eZ

                diag(beta[,2:(k1+1),1]) <- diag(beta[,2:(k1+1),1])-C # subtract one for warm start purposes 
                
            }else{
                MSFE[v-T2+h-1] <- norm2(ZFull$Y[v+h-1,1:k1] - betaEVAL %*% eZ)^2

                preds[v-T2+h-1,] <- betaEVAL %*% eZ
                
            }

            if(verbose){

                setTxtProgressBar(pb, v)

            }
        }
                                        # Parameter estimates using all available data
        if (group == "Basic") {

            betaPred <- .lassoVARFistX(beta, ZFull$Z, ZFull$Y,gamm, 1e-05,p,MN,k,k1,s+s1,m,C)            
        }
        if (group == "Lag") {

            GG <- .GroupLassoVAR1(beta,jj,jjcomp,ZFull$Y,ZFull$Z,gamm,activeset,1e-04,p,MN,k,k1,s+s1,C)

            betaPred <- GG$beta

        }

        if (group == "SparseLag") {

            GG <- .SparseGroupLassoVARX(beta, jj, ZFull$Y, ZFull$Z, 
                                        gamm, alpha, INIactive = activeset, 1e-04, q1a,p,MN,k,s+s1,k1,C)

            betaPred <- GG$beta

        }

        if (group == "OwnOther") {

            GG <- .GroupLassoOOX(beta, kk, ZFull$Y, ZFull$Z, gamm, 
                                 activeset, 1e-04,p,MN,k,k1,s+s1,C)

            betaPred <- GG$beta

        }
        if (group == "SparseOO") {

            GG <- .SparseGroupLassoVAROOX(beta, kk, ZFull$Y, ZFull$Z, 
                                          gamm, alpha, INIactive = activeset, 1e-04,p,MN,k1,s+s1,k,FALSE,C)

            betaPred <- GG$beta
            

        }

        if(group=="EFX")

            {


                betaPred <- .EFVARX(beta,ZFull$Y,ZFull$Z,gamm,1e-4,MN,k1,s+s1,m,p,C)


            }
        

        betaPred <- as.matrix(betaPred[,,1])


        return(list(MSFE=MSFE,betaPred=betaPred,predictions=preds))
        

    }


                                        # Forecast evaluation: VAR (called in cv.bigvar)
.BigVAREVAL <- function(ZFull,gamopt,k,p,group,h,MN,verbose,RVAR,palpha,T1,T2,alpha,recursive,C)
    {
        

        gran2 <- 1

        preds <- matrix(NA,nrow=length((T1+1):T2),ncol=k)


        gamm <- gamopt

        Y <- ZFull$Y

        s <- p
        k1 <- k

        MSFE <- rep(NA,length((T1+1):T2))

        
        beta <- array(0,dim=c(k,p*k+1,1))

        if (group == "Lag")
            {

                jj <- .groupfuncpp(p,k)

                jjcomp <- .groupfuncomp(p,k)

                activeset <- rep(list(rep(rep(list(0), length(jj)))), 
                                 gran2)

            }
        
        if (group == "SparseLag")
            {

                jj <- .groupfun(p, k)

                q1a <- list()

                for (i in 1:(p)) {

                    q1a[[i]] <- matrix(runif(k, -1, 1), ncol = 1)

                }

                activeset <- rep(list(rep(rep(list(0), length(jj)))), 
                                 gran2)

                

            }
        

        if (group == "OwnOther")
            {


                kk <- .lfunction3cpp(p, k)

                activeset <- rep(list(rep(rep(list(0), length(kk)))), 
                                 gran2)

            }

        if (group == "SparseOO") {

            kk <- .lfunction3cpp(p, k)

            jjcomp <- .lfunctioncomp(p,k)

            jj=.lfunction3(p,k)

            activeset <- rep(list(rep(rep(list(0), length(kk)))), 
                             gran2)

            q1a <- list()

            for (i in 1:(2*p))
                {

                    q1a[[i]] <- matrix(runif(length(kk[[i]]), -1, 1), ncol = 1)

                }

        }


        if(verbose)
            {

                print("Evaluation Stage")

                pb <- txtProgressBar(min = T1-h+2, max = T2, style = 3)

            }
        
        for (v in (T1-h+2):T2)
            {

                if(h>1 & ! recursive){


                    trainY <- ZFull$Y[h:(v-1), ]


                    trainZ <- ZFull$Z[,1:(v-h)]

                }else{


                    trainY <- ZFull$Y[1:(v-1), ]

                    trainZ <- ZFull$Z[,1:(v-1)]
                }
                if(v+h-1>T2){
                    break
                }
                if (group == "Basic") {

                    beta <- .lassoVARFist(beta, trainZ, trainY,gamm, 1e-05,p,MN,C)

                }

                if (group == "Lag") {

                    GG <- .GroupLassoVAR1(beta,jj,jjcomp,trainY,trainZ,gamm,activeset,1e-04,p,MN,k,k1,s,C)

                    beta <- GG$beta

                    activeset <- GG$active
                }
                

                if (group == "SparseLag") {

                    GG <- .SparseGroupLassoVAR(beta, jj, trainY, trainZ, 
                                               gamm, alpha, INIactive = activeset, 1e-04, q1a,p,MN,C)

                    beta <- GG$beta

                    activeset <- GG$active

                    q1a <- GG$q1

                }
                

                if (group == "OwnOther") {

                    GG <- .GroupLassoOO(beta, kk, trainY, trainZ, gamm, 
                                        activeset, 1e-04,p,MN,C)

                    beta <- GG$beta

                    activeset <- GG$active

                }

                if (group == "SparseOO") {
                    
                    GG <- .SparseGroupLassoVAROO(beta, kk, trainY, trainZ, 
                                                 gamm, alpha, INIactive = activeset, 1e-04,q1a,p,MN,FALSE,C)

                    beta <- GG$beta

                    activeset <- GG$active

                    q1a <- GG$q1

                }

                if (group=="HVARC"){

                    beta <- .HVARCAlg(beta,trainY,trainZ,gamm,1e-05,p,MN,C)
                    
                }


                if(group=="HVAROO"){

                    beta <- .HVAROOAlg(beta,trainY,trainZ,gamm,1e-05,p,MN,C)

                }

                if(group=="HVARELEM"){

                    beta <- .HVARElemAlg(beta,trainY,trainZ,gamm,1e-05,p,MN,C) 	


                }

                if(group=="Tapered"){

                    beta <- .lassoVARTL(beta, trainZ, trainY,gamm, 1e-4,p,MN,palpha,C)
                }
                

                betaEVAL <- matrix(beta[,,1],nrow=k,ncol=(k*p+1))

                if (RVAR) {

                    betaEVAL <- RelaxedLS(cbind(t(trainZ),trainY),betaEVAL,k,p,k1,s)      
                }
                
                if(MN){
                    eZ <- matrix(ZFull$Z[,v],ncol=1)

                }else{


                    eZ <- matrix(c(1,ZFull$Z[,v]),ncol=1)


                }
                                        # We don't consider an intercept for the MN lasso
                if(MN){

                    preds[v-T1+h-1,] <- betaEVAL[,2:ncol(betaEVAL)] %*% eZ

                    if(h>1 & recursive){

                        pred <- matrix(preds[v-T1,],nrow=1)
                        
                        preds[v-T1+h-1,] <- predictMS(pred,trainY,h-1,betaEVAL[,2:ncol(betaEVAL)],p,TRUE)

                    }else{
                    
                    MSFE[v-T1+h-1] <- norm2(ZFull$Y[v+h-1, ] - preds[v-T1-h+1,])^2

                    diag(beta[,2:(k+1),1]) <- diag(beta[,2:(k+1),1])-C # subtract one for warm start purposes 

                    }
                    
                }else{
                    preds[v-T1+h-1,] <- betaEVAL %*% eZ                       
                    if(h>1 & recursive){

                        pred <- matrix(preds[v-T1+h-1,],nrow=1)

                        preds[v-T1+h-1,] <- predictMS(pred,trainY,h-1,betaEVAL,p,FALSE)

                    }
                    
                    MSFE[v-T1+h-1] <- norm2(ZFull$Y[v+h-1,] - preds[v-T1+h-1,])^2                    
                    
                }

                if(verbose){


                    setTxtProgressBar(pb, v)

                }
            }

        if (group == "Basic") {

            betaPred <- .lassoVARFist(beta, ZFull$Z, ZFull$Y,gamm, 1e-05,p,MN,C)

        }

        if (group == "Lag") {

            GG <- .GroupLassoVAR1(beta,jj,jjcomp,ZFull$Y,ZFull$Z,gamm,activeset,1e-04,p,MN,k,k,s,C)
            

            betaPred <- GG$beta

        }

        if (group == "SparseLag") {

            GG <- .SparseGroupLassoVAR(beta, jj, ZFull$Y, ZFull$Z, 
                                       gamm, alpha, INIactive = activeset, 1e-04, q1a,p,MN,C)

            betaPred <- GG$beta

        }

        if (group == "OwnOther") {


            GG <- .GroupLassoOO(beta, kk, ZFull$Y, ZFull$Z, gamm, 
                                activeset, 1e-04,p,MN,C)

            betaPred <- GG$beta

        }

        if (group == "SparseOO") {

            GG <- .SparseGroupLassoVAROO(beta, kk, ZFull$Y, ZFull$Z, 
                                         gamm, alpha, INIactive = activeset, 1e-04,q1a,p,MN,FALSE,C)

            betaPred <- GG$beta
            

        }

        if (group=="HVARC")
            {

                betaPred <- .HVARCAlg(beta,ZFull$Y,ZFull$Z,gamm,1e-05,p,MN,C)

            }

        if(group=="HVAROO")
            {

                betaPred <- .HVAROOAlg(beta,ZFull$Y,ZFull$Z,gamm,1e-05,p,MN,C)


            }

        if(group=="HVARELEM")
            {

                betaPred<-.HVARElemAlg(beta,ZFull$Y,ZFull$Z,gamm,1e-05,p,MN,C) 	

            }

        if(group=="Tapered")
            {
                betaPred <- .lassoVARTL(beta,ZFull$Z,ZFull$Y,gamm,1e-4,p,MN,palpha,C)            
            }    
        

        betaPred <- as.matrix(betaPred[,,1])

        betaPred <- matrix(betaPred,nrow=k)

        

        return(list(MSFE=MSFE,betaPred=betaPred,predictions=preds))
        

    }



#' Converts a VAR coefficient matrix of order p to multiple companion form
#' 
#' @param B a \eqn{k \times kp} coefficient matrix
#' @param p Lag order
#' @param k Number of Series
#' @return Returns a \eqn{kp \times kp} coefficient matrix representing all coefficient matrices contained in Ai as a VAR(1).
#' @references See page 15 of Lutkepohl, "A New Introduction to Multiple Time Series Analysis"
#' @seealso \code{\link{MultVarSim}}
#' @examples
#' k=3;p=6
#' B=matrix(0,nrow=k,ncol=p*k)
#' A1<- matrix(c(.4,-.02,.01,-.02,.3,.02,.01,.04,.3),ncol=3,nrow=3)
#' A2 <- matrix(c(.2,0,0,0,.3,0,0,0,.13),ncol=3,nrow=3)
#' B[,1:k]=A1
#' B[,(4*k+1):(5*k)]=A2
#' A <- VarptoVar1MC(B,p,k)
#' @export
VarptoVar1MC <- function(B,p,k)
    {

        Fp=matrix(0,nrow=k*p,ncol=k*p)      

        Fp[1:k,] = B

        Fp[-(1:k),1:(k*(p-1))] = diag(k*(p-1))
                                        # We require that the coefficient matrix generates a stationary VAR
        if(max(Mod(eigen(Fp)$values))>1){warning("Coefficient Matrix is not stationary")}

        return(Fp)

    }


#' Simulate a VAR
#' 
#' @param k Number of Series
#' @param A1 Either a \eqn{k \times k} coefficient matrix or a \eqn{kp \times kp} matrix created using \code{\link{VarptoVar1MC}}. 
#' @param p Maximum Lag Order
#' @param Sigma Residual Covariance Matrix of dimension \eqn{k\times k}
#' @param T Number of simulations
#' @return Returns a \eqn{T \times k} of realizations from a VAR.
#' @references Lutkepohl, "A New Introduction to Multiple Time Series Analysis"
#' @seealso \code{\link{VarptoVar1MC}}
#' @examples
#' k=3;p=6
#' B=matrix(0,nrow=k,ncol=p*k)
#' A1<- matrix(c(.4,-.02,.01,-.02,.3,.02,.01,.04,.3),ncol=3,nrow=3)
#' A2 <- matrix(c(.2,0,0,0,.3,0,0,0,.13),ncol=3,nrow=3)
#' B[,1:k]=A1
#' B[,(4*k+1):(5*k)]=A2
#' A <- VarptoVar1MC(B,p,k)
#' Y <-MultVarSim(k,A,p,.1*diag(k),100)
#' @export
#' @importFrom MASS mvrnorm
MultVarSim <- function (k, A1, p, Sigma, T) 
{

    if(max(Mod(eigen(A1)$values))>1){stop("Error: Generator Matrix is not stationary")}
    
                                        # add 500 observations for initialization purposes

    Y <- matrix(0, nrow = T+500+p , ncol = k)

    YY <- as.vector(Y)

    for (i in seq(from = (k * p + 1), to = (nrow(Y) * k - 1), 
                  by = k)) {

        u <- as.vector(c(mvrnorm(1, rep(0, k), Sigma), rep(0, 
                                                           k * p - k)))

        YY[(i + k):(i - k * p + 1 + k)] <- A1 %*% YY[(i):(i - 
                                                          k * p + 1)] + as.matrix(u, ncol = 1)

    }

    YY <- YY[YY!=0]

    Y <- matrix(YY, ncol = k, byrow = TRUE)

    Y <- Y[,c(ncol(Y):1)]

    Y <- Y[501:nrow(Y), ]

    return(Y)
}

                                        # function to create subsets for lag group VARX-L
.groupfun <- function(p,k)
    {

        jjj <- list()
        jjj[[1]] <- 1:k

        if(p>1){

            for(i in 2:p){
                jjj[[i]] <- jjj[[i-1]]+k

            }

        }

        return(jjj)

    }

                                        #C++ groupings to account for 0 indexing
.groupfuncpp <- function(p,k)
{
    jjj <- list()

    jjj[[1]] <- 0:(k-1)

    if(p>1)
        {
            for(i in 2:p){

                jjj[[i]] <- jjj[[i-1]]+k
            }
        }
    return(jjj)
}

                                        # subsetting complement of groups in rcpp
.groupfuncomp <- function(p,k)
    {


        ownoth <- .groupfuncpp(p,k)

        kk2 <- list()

        pmax <- max(unlist(ownoth))

        to <- 0:(pmax)

        for(i in 1:length(ownoth))
            {

                kk2[[i]] <- to[is.na(pmatch(to,ownoth[[i]]))]
            }

        return(kk2)

    }    

                                        # Group indexing for own/other VARX-L
.lfunction2 <- function(p,k)
    {

        kk <- list()

        kk[[1]] <- 1:(k^2)

        if(p>1)
            {
                for(i in 2:p)
                    {
                        kk[[i]] <- 1:(k^2)+tail(kk[[i-1]],1)

                    }
            }    
        return(kk)
    }

.lfunction2cpp <- function(p,k)
    {

        kk <- list()

        kk[[1]] <- 0:(k^2-1)

        if(p>1)

            {

                for(i in 2:p)

                    {

                        kk[[i]] <- 0:(k^2-1)+tail(kk[[i-1]],1)+1


                    }

            }    

        return(kk)


    }

.lfunction3 <- function(p,k)
    {

        kk <- .lfunction2(p,k)

        oo <- list()

        pp <- list()

        for(i in 1:length(kk))
            {
                j <- 0
                oo[[i]] <- kk[[i]][(seq(1,length(kk[[1]]),k+1)+(j*k^2))]
                pp[[i]] <- kk[[i]][-(seq(1,length(kk[[1]]),k+1)+(j*k^2))]
                j <- j+1

            }

        ownoth <- c(oo,pp)
        return(ownoth)
    }    

.lfunction3cpp <- function(p,k)
    {

        kk <- .lfunction2cpp(p,k)
        oo <- list()
        pp <- list()

        for(i in 1:length(kk))
            {
                j <- 0
                oo[[i]] <- kk[[i]][(seq(1,length(kk[[1]]),k+1)+(j*k^2))]
                pp[[i]] <- kk[[i]][-(seq(1,length(kk[[1]]),k+1)+(j*k^2))]
                j <- j+1
            }

        ownoth <- c(oo,pp)

        return(ownoth)

    }    


.lfunctioncomp <- function(p,k)
    {

        ownoth <- .lfunction3cpp(p,k)
        kk2 <- list()
        pmax <- max(unlist(ownoth))
        to <- 0:(pmax)
        for(i in 1:length(ownoth))
            {
                kk2[[i]] <- to[is.na(pmatch(to,ownoth[[i]]))]
            }

        return(kk2)
    }    


                                        # This function should work for arbitrary groups
.lfunction <- function(groups,p)
    {

        H <- as.vector(do.call('cbind',groups))
        kk <- list()
        kk[[1]] <- H
        if(p>1)
            {
                for(i in 2:p){
                    kk[[i]] <-as.vector(do.call('cbind',groups))+tail(kk[[i-1]],1) }
            }
        return(kk)

    }

                                        # Indexing for HVAR
.vsubs <- function(p,k)
    {
        vi <- list()

        for(i in p:1)
            {
                g <- max(k*i-k,0)
                vi[[i]] <- g:(k*(p)-1)
            }
        return(vi)   

    }

                                        #indexing for HVAR OO
.oofun <- function(p,k)
    {

        kk <- .lfunction2(p, k)
        oo <- list()
        pp <- list()

        for (i in 1:length(kk)) {
            j <- 0
            oo[[i]] <- kk[[i]][(seq(1, length(kk[[1]]), k + 1) + 
                                (j * k^2))]
            pp[[i]] <- kk[[i]][-(seq(1, length(kk[[1]]), k + 1) + 
                                 (j * k^2))]
            j <- j + 1
        }
        ownoth <- list()
        jj <- .lfunction3(p,k)

        for(i in 1:length(jj))

            {

                if(i==1)

                    {

                        ownoth[[i]] <- jj[[i]]
                        oo[[1]] <- NULL

                    }

                if(i==length(jj))
                    {
                        ownoth[[i]] <- tail(jj,1)
                        pp[[1]] <- NULL
                    }

                if(i!=1& i%%2!=0)
                    {

                        ownoth[[i]] <- head(oo,1)
                        oo[[1]] <- NULL
                    }

                if(i!=length(jj)&i%%2==0)
                    {
                        ownoth[[i]] <- head(pp,1)
                        pp[[1]] <- NULL
                    }


            }

        return(rev(ownoth))

    }

.oocumfun <-function(p,k)
{

    kk <- rev(.oofun(p,k))
    oogroups <- list()
    oogroups[[1]] <- unlist(kk)
    for(i in 2:length(kk))
        {
            oogroups[[i]] <- unlist(kk[-(1:(i-1))])

        }
    
    return(oogroups)
}


                                        # indexing function for Own/Other HVAR
.vecoovars<-function(p,k,k1)
{

    vv <- list()

    vv[[1]] <- 1:(p*k)

    vv[[2]] <- vv[[1]][-k1]

    q1 <- 1

    if(p>1){
        for(i in 3:(2*p))
            {
                if(i%%2!=0)
                    {
                        vv[[i]] <- (q1*k+1):(k*p)
                        q1 <- q1+1
                    }else{
                        vv[[i]] <- vv[[i-1]][-k1]
                    }
            }
    }
    return(vv)

}

                                        # indexing to start at zero for use within rcpp
.vecoovarscpp<-function(p,k,k1)
{

    vv <- list()
    vv[[1]] <- 0:(p*k-1)
    vv[[2]] <- vv[[1]][-(k1)]
    q1 <- 1

    if(p>1){
        for(i in 3:(2*p))
            {
                if(i%%2!=0)
                    {
                        vv[[i]] <- (q1*k):(k*p-1)
                        q1 <- q1+1
                    }else{
                        
                        vv[[i]] <- vv[[i-1]][-(k1)]

                    }

            }

    }

    return(vv)

}


                                        #VARX Lag Group function
groupfunVARX <- function(p,k,k1,s)
    {

        jj <- list()
        m <- k-k1
        jj <- .groupfuncpp(p, k1)
        kp <- k1*p+m*s-1
        jj2 <- list()
        startjj <- max(unlist(jj))+1
        for(i in seq(startjj,kp,by=1))
            {
                jj[[i]] <- i                
            }
        jj[sapply(jj, is.null)] <- NULL
        
        return(jj)
    }

groupfunVARXcomp <- function(p,k,k1,s)
    {
        ownoth <- groupfunVARX(p,k,k1,s)
        kk2 <- list()
        pmax <- max(unlist(ownoth))
        to <- 0:(pmax)
        for(i in 1:length(ownoth))
            {
                kk2[[i]] <- to[is.na(pmatch(to,ownoth[[i]]))]
            }
        return(kk2)

    }


                                        # indexing starts at 1 for use in R        
groupfunVARXLG <- function(p,k,k1,s)
    {

        jj <- list()
        jj <- .groupfun(p, k1)
        m <- k-k1
        kp <- k1*p+m*s
        jj2 <- list()
        startjj <- max(unlist(jj))+1
        for(i in seq(startjj,kp,by=1))
            {
                jj[[i]] <- i
                
            }
        jj[sapply(jj, is.null)] <- NULL

        return(jj)
    }



diaggroupfunVARX <- function(p,k,k1,s)
    {
        m <- k-k1
        jj <- list()
        jj <- .lfunction3cpp(p, k1)
        kp <-k1*(p*k1+s*m)-1
        jj2 <- list()
        startjj <- max(unlist(jj))+1

        for(i in seq(startjj,kp,by=k1))
            {
                jj[[i]] <- i:(i+k1-1)
                
            }

        jj[sapply(jj, is.null)] <- NULL


        return(jj)

    }

diaggroupfunVARXcomp <- function(p,k,k1,s)
    {

        ownoth <- diaggroupfunVARX(p,k,k1,s)
        kk2 <- list()
        pmax <- max(unlist(ownoth))
        to <- 0:(pmax)

        for(i in 1:length(ownoth))
            {
                kk2[[i]] <- to[is.na(pmatch(to,ownoth[[i]]))]
            }
        
        return(kk2)
    }

diaggroupfunVARXLG <- function(p,k,k1,s)
    {

        m <- k-k1
        jj <- list()
        jj <- .lfunction3(p, k1)
        kp <- k1*(p*k1+s*m)
        jj2 <- list()
        startjj <- max(unlist(jj))+1
        for(i in seq(startjj,kp,by=k1))
            {
                jj[[i]] <- i:(i+(k1-1))                
            }
        
        jj[sapply(jj, is.null)] <- NULL

        return(jj)

    }

diaggroupfunVARXLGL <- function(p,k,k1)
    {

        jj <- list()
        jj <- .lfunction3(p, k1)
        kp <- k1*p*k
        jj2 <- list()
        startjj <- max(unlist(jj))+1

        for(i in seq(startjj,kp,by=1))
            {
                jj[[i]] <- i                
            }
        
        jj[sapply(jj, is.null)] <- NULL

        return(jj)
    }


diaggroupfunVARXL <- function(p,k,k1)
    {
        jj <- list()
        jj <- .lfunction3cpp(p, k1)
        kp <- k1*p*k-1
        jj2 <- list()
        startjj <- max(unlist(jj))+1

        for(i in seq(startjj,kp,by=1))
            {
                jj[[i]] <- i
                
            }

        jj[sapply(jj, is.null)] <- NULL
        return(jj)

    }

diaggroupfunVARXcompL <- function(p,k,k1)
    {

        ownoth <- diaggroupfunVARXL(p,k,k1)
        kk2 <- list()
        pmax <- max(unlist(ownoth))

        to <- 0:(pmax)

        for(i in 1:length(ownoth))

            {

                kk2[[i]] <- to[is.na(pmatch(to,ownoth[[i]]))]

            }

        return(kk2)

    }


                                        # iterative procedure to find a less coarse bound for lambda starting value via binary search
LGSearchX <- function(gstart,Y,Z,BOLD,group,k1,p,s,m,gs,k,MN,alpha,C)
    {

        tk <- 1/max(Mod(eigen(Z%*%t(Z))$values))
        lambdah <- gstart
        lambdal <- 0
        activeset <- list(rep(rep(list(0), length(gs))))

        while(max(lambdah-lambdal)>.00001)
            {

                lambda <- (lambdah+lambdal)/2
                if(group=="EFX"){
                    BOLD <- .EFVARX(BOLD,Y,Z,lambda,1e-4,MN,k1,s,m,p,C)
                    param <- BOLD[,2:(k1*p+m*s+1),1]
                }

                if(group=="Basic"){
                    param <- .lassoVARFistX(BOLD,Z,Y[,1:k1],lambda,1e-04,p,MN,k1+m,k1,s,m,C)[,2:(k1*p+m*s+1),]
                }

                if(group=="Lag"){

                    jj <- groupfunVARX(p,k,k1,s)
                    jjcomp <- groupfunVARXcomp(p,k,k1,s)
                    BB <- .GroupLassoVAR1(BOLD,jj,jjcomp,Y[,1:k1],Z,lambda,activeset,1e-4,p,MN,k,k1,s,C)
                    BOLD <- BB$beta
                    param <- BB$beta[,2:(k1*p+m*s+1),]
                    activeset <- BB$active
                }
                

                if(group=="OwnOther")
                    {

                        kk <- diaggroupfunVARX(p, k,k1,s)
                        BB <- .GroupLassoOOX(BOLD, kk, Y, Z, lambda,activeset, 1e-04,p,MN,k,k1,s,C)
                        param <- BB$beta[,2:(k1*p+m*s+1),]
                        BOLD <- BB$beta
                        activeset <- BB$active
                    }

                if(group=="SparseOO")
                    {

                        kk <- diaggroupfunVARX(p, k,k1,s)
                        BB <- .SparseGroupLassoVAROOX(BOLD, kk, Y[,1:k1], Z, lambda,alpha,activeset, 1e-04,p,MN,k1,s,k,FALSE,C)
                        param <- BB$beta[,2:(k1*p+m*s+1),]
                        BOLD <- BB$beta
                        activeset <- BB$active

                    }
                

                if(group=="SparseLag"){

                    jj <- groupfunVARX(p, k,k1,s)
                    jjcomp <- groupfunVARXcomp(p,k,k1,s)
                    q1a=list()

                    for (i in 1:(p+s)) {

                        q1a[[i]] <- matrix(runif(length(jj[[i]]), -1, 1), ncol = 1)

                    }
                    
                    BB <- .SparseGroupLassoVARX(BOLD,jj,Y[,1:k1],Z,lambda,alpha,activeset,1e-4,q1a,p,MN,k,s,k1,C)
                    param <- BB$beta[,2:(k1*p+m*s+1),]
                    BOLD <- BB$beta
                    activeset <- BB$active
                }

                if(MN){
                    ## diag(param[1:k1,1:k1]) <- 0
                    ## diag(BOLD[1:(k1),2:(k1+1),1]) <- 0

                    
                    diag(param[1:k1,1:k1]) <- ifelse(C==0,diag(param[1:k1,1:k1]),diag(param[1:k1,1:k1])-C)
                    diag(BOLD[,2:(k1*p+1+m*s),]) <- ifelse(C==0,diag(BOLD[,2:(k1*p+1+m*s),]),diag(BOLD[,2:(k1*p+1+m*s),])-C)

                }

                if(max(abs(param))<sqrt(.Machine$double.eps))
                    {
                        lambdah <- lambda
                    }else{

                        lambdal <- lambda
                        
                    }

            }

        lambdah
    }

                                        # Same as above, but for the VAR
LGSearch <- function(gstart,Y,Z,BOLD,group,k,p,gs,MN,alpha,C)
    {

        tk <- 1/max(Mod(eigen(Z%*%t(Z))$values))
        lambdah <- gstart
        lambdal <- 0
        activeset <- list(rep(rep(list(0), length(gs))))

        if(group=="SparseOO")
            {
                kk <- .lfunction3cpp(p, k)
                activeset <- rep(list(rep(rep(list(0), length(kk)))),1)
                q1a=list()
                for (i in 1:(2*p)) {
                    q1a[[i]] <- matrix(runif(length(gs[[i]]), -1, 1), ncol = 1)
                }

            }

        while(max(abs(lambdah-lambdal))>.00001)
            {

                lambda <- (lambdah+lambdal)/2
                if(group=="Basic"){
                    param <- .lassoVARFist(BOLD,Z,Y,lambda,1e-04,p,MN,C)[,2:(k*p+1),]
                }

                if(group=="Tapered"){

                    param <- .lassoVARTL(BOLD,Z,Y,lambda,1e-04,p,MN,rev(seq(0,1,length=10)),C)[,2:(k*p+1),]

                    param <- param[,,1]                    
                }
                
                if(group=="Lag"){

                    jj <- .groupfuncpp(p, k)
                    jjcomp <- .groupfuncomp(p,k)
                    BB <- .GroupLassoVAR1(BOLD,jj,jjcomp,Y,Z,lambda,activeset,1e-4,p,MN,k,k,p,C)
                    BOLD <- BB$beta
                    param <- BB$beta[,2:(k*p+1),]
                    activeset <- BB$active                  
                }

                if(group=="SparseLag"){

                    jj <- .groupfuncpp(p, k)
                    jjcomp <- .groupfuncomp(p,k)
                    q1a=list()
                    for (i in 1:p) {
                        q1a[[i]] <- matrix(runif(k, -1, 1), ncol = 1)
                    }
                    BB <- .SparseGroupLassoVAR(BOLD,jj,Y,Z,lambda,alpha,activeset,1e-4,q1a,p,MN,C)
                    param <- BB$beta[,2:(k*p+1),]
                    BOLD <- BB$beta
                    activeset <- BB$active
                }
                

                if(group=="OwnOther")
                    {
                        kk <- .lfunction3cpp(p, k)
                        BB <- .GroupLassoOO(BOLD, kk, Y, Z, lambda,activeset, 1e-04,p,MN,C)
                        param <- BB$beta[,2:(k*p+1),]
                        BOLD <- BB$beta
                        activeset <- BB$active

                    }


                if(group=="SparseOO")
                    {
                        BB <- .SparseGroupLassoVAROO(BOLD, kk, Y, Z, lambda,alpha,activeset, 1e-04,q1a,p,MN,FALSE,C)
                        param <- BB$beta[,2:(k*p+1),]
                        BOLD <- BB$beta
                        activeset <- BB$active                  
                    }

                if(group=="HVARC")
                    {
                        BOLD <- .HVARCAlg(BOLD,Y,Z,lambda,1e-4,p,MN,C)                  
                        param <- BOLD[,2:(k*p+1),]
                    }
                if(group=="HVAROO")
                    {
                        BOLD <- .HVAROOAlg(BOLD,Y,Z,lambda,1e-4,p,MN,C)
                        param <- BOLD[,2:(k*p+1),]
                    }

                if(group=="HVARELEM")
                    {
                        BOLD <- .HVARElemAlg(BOLD,Y,Z,lambda,1e-4,p,MN,C)
                        param <- BOLD[,2:(k*p+1),]
                    }


                if(MN){

                    diag(param[1:k,1:k]) <- ifelse(C==0,diag(param[1:k,1:k]),0)
                    diag(BOLD[,2:(k*p+1),]) <- ifelse(C==0,diag(BOLD[,2:(k*p+1),]),0)
                    
                }
                
                if(max(abs(param))==0)
                    {
                        lambdah <- lambda

                    }else{
                        lambdal <- lambda
                        

                    }


            }
        
        lambdah

    }


#' Evaluate forecasts from a VAR or VARX with lag orders selected by AIC/BIC
#' 
#' @param Y a \eqn{T \times k} multivariate time series 
#' @param X a \eqn{T \times m} multivariate time series of unmodeled exogenous variables
#' @param p maximum lag order for endogenous series
#' @param s maximum lag order for exogenous series
#' @param T1 start of forecast evaluation period.
#' @param T2 end of forecast evaluation period
#' @param IC specifies whether to select lag order according to "AIC" or "BIC"
#' @param h desired forecast horizon
#' @param iterated indicator as to whether to use iterated or direct multistep forecasts (if applicable, VAR context only)
#' @return Returns the one-step ahead MSFE over the evaluation period.
#' @details This function evaluates the one-step ahead forecasts of a VAR or VARX fit by least squares over an evaluation period.  At every point in time, lag orders for the endogenous and exogenous series are selected according to AIC or BIC.  This function is run automatically when \code{\link{cv.BigVAR}} is called unless \code{ic} is set to \code{FALSE} in \code{\link{constructModel}}.      
#' @references Neumaier, Arnold, and Tapio Schneider. "Estimation of parameters and eigenmodes of multivariate autoregressive models." ACM Transactions on Mathematical Software (TOMS) 27.1 (2001): 27-57.
#' @seealso \code{\link{VARXFit}},\code{\link{constructModel}}, \code{\link{cv.BigVAR}}
#' @examples
#' data(Y)
#'
#' # Evaluate the performance of a VAR with lags selected by BIC.
#' p <- 4
#' T1 <- floor(nrow(Y))/3
#' T2 <- floor(2*nrow(Y))/3
#' # Matrix of zeros for X
#' X <- matrix(0,nrow=nrow(Y),ncol=ncol(Y))
#' BICMSFE <- VARXForecastEval(Y,X,p,0,T1,T2,"BIC",1)
#' 
#' @export
VARXForecastEval <- function(Y,X,p,s,T1,T2,IC,h,iterated=FALSE)
{


    if(T1>nrow(Y) | T2>nrow(Y) |T2<T1){stop("Training dates exceed series length")}

    if(!IC%in%c("AIC","BIC") )
        {

            stop("IC must either be AIC or BIC")

        }

    MSFE <- c()

    k <- ncol(Y)
    m <- ifelse(s!=0,ncol(X),0)
    for(i in (T1-h+2):T2){

        if(h+i-1>T2){break}

        testY <- as.matrix(Y[1:(i-1),])
        testX <- as.matrix(X[1:(i-1),])

        if(!iterated){
            hd=h
            
        }else{

            hd=1
            
            }
        if(IC=="BIC"){
            popt <- ICX(testY,testX,k,p,s,m,"BIC",h=hd)
        }
        if(IC=="AIC"){

            popt <- ICX(testY,testX,k,p,s,m,"AIC",h=hd) 
        }
        B1 <- popt$B
        
        if(popt$p==0 &popt$s==0){
            
            eZ <- matrix(rep(1,1),ncol=1)

            pred <- B1%*%eZ

        }else{


            C <- max(popt$p,popt$s)

            ## print(C)
            ## if(C==1){

                ## # possibly memory leak in VARX lag matrix construction in Eigen if maxlag is 1.
                ## # to be on the safe side, we will perform it in R
                ## if(popt$s==0){
                ## eZ <- c(1,Y[i-1,])
                ## eZ <- matrix(eZ,ncol=1)
                ## }else{
                ##     eZ <- c(1,Y[i-1,],X[i-1,])               
                ##     }
                ## }else{
            eZ <- VARXCons(as.matrix(Y[(i-C):(i),]),as.matrix(X[(i-C):(i),]),k,popt$p,m,popt$s)

            ## }
            
            pred <- B1%*%eZ
            
                                        # iterated multistep forecasts (if VAR and horizon greater than 1)
            if(h>1 & s==0 & iterated ){
                
                pred <- predictMS(matrix(pred,nrow=1),Y,h-1,B1,C,FALSE)

            }

        }

        MSFEi <- norm2(Y[i+h-1,]-pred)^2
        MSFE <- c(MSFE,MSFEi)
    }

    return(MSFE)


}

#' Fit a VAR or VARX model by least squares
#' 
#' @param Y a \eqn{t \times k} multivariate time series
#' @param p maximum lag order
#' @param IC Information criterion indicator, if set to \code{NULL}, it will fit a least squares VAR(X) of orders p and s.  Otherwise, if set to "AIC" or "BIC" it return the model with lag orders that minimize the given IC. 
#' @param VARX a list of VARX specifications (as in \code{\link{constructModel}} (or NULL )
#' @return Returns a list with four entries:
#' \itemize{
#' \item{"Bhat"}{Estimated \eqn{k\times kp+ms} coefficient matrix}
#' \item{"SigmaU}{Estimated \eqn{k\times k} residual covariance matrix}
#' \item{"phat"}{Selected lag order for VAR component}
#' \item{"shat"}{Selected lag order for VARX component}
#' }
#' @details This function uses a modified form of the least squares technique proposed by Neumaier and Schneider (2001).  It fits a least squares VAR or VARX via a QR decomposition that does not require explicit matrix inversion.  This results in improved computational performance as well as numerical stability over the conventional least squares approach. 
#' @references Neumaier, Arnold, and Tapio Schneider. "Estimation of parameters and eigenmodes of multivariate autoregressive models." ACM Transactions on Mathematical Software (TOMS) 27.1 (2001): 27-57.
#' @seealso \code{\link{constructModel}}, \code{\link{cv.BigVAR}}
#' @examples
#' data(Y)
#' # fit a VAR_3(3)
#' mod <- VARXFit(Y,3,NULL,NULL)
#' # fit a VAR_3 with p= 6 and lag selected according to AIC
#' modAIC <- VARXFit(Y,6,"AIC",NULL)
#' # Fit a VARX_{2,1} with p=6, s=4 and lags selected by BIC
#' modXBIC <- VARXFit(Y,6,"BIC",list(k=1,s=4))
#' 
#' @export
VARXFit <- function(Y,p,IC,VARX=NULL)
    {

        if(!is.null(VARX)){

            if(is.list(VARX) & !(exists('k',where=VARX) & exists('s',where=VARX)))
                {

                    stop("VARX Specifications entered incorrectly")

                }

        }
        if(is.list(VARX) & (length(VARX)!=0)){

            k1 <- VARX$k
            s <- VARX$s
            Y1 <- matrix(Y[,1:k1],ncol=k1)
            m <- ncol(Y)-k1
            X <- matrix(Y[,(k1+1):ncol(Y)],ncol=m)
            Z <- VARXCons(Y1,X,k1,p,m,s)
            offset <- max(p,s)+1
            YT <- matrix(Y1[offset:nrow(Y),],ncol=k1)
            X <- matrix(X[offset:nrow(X),],ncol=m)
            
        }else{

            k <- ncol(Y)
            k1 <- k
            s=0;m=0
            offset <- p+1
            X <- matrix(0,nrow=nrow(Y))
            
            Z <- VARXCons(Y,X,k,p,m,s)
            YT <- matrix(Y[(offset):nrow(Y),],ncol=ncol(Y))
            
        }
        if(is.null(IC)){

            Res <- ARFitVARXR(cbind(t(Z),YT),k1,p,m,s)

            shat <- s
            phat <- p
            
        }else{
            if(!IC%in%c("AIC","BIC") )
                {

                    stop("IC must either be AIC,BIC, or set to NULL")

                }

            Res <- ICX(YT,X,k1,p,s,m,IC)

            shat <- Res$s
            phat <- Res$p
            
        }

        list(Bhat=Res$B,SigmaU=Res$SigmaU,phat=phat,shat=shat)
        
    }

                                        # Recursive multi-step predictions


predictMS <- function(pred,Y,n.ahead,B,p,MN=FALSE){

                                        # Augment Y with predictions, create lag matrix (no intercept if MN)
    Y <- rbind(Y,pred)

    # Can't call this function directly from R due to assert errors
    ## Z <- ZmatF(Y,p,ncol(Y),oos=TRUE,intercept=!MN)

    Z <- VARXCons(Y,matrix(0,nrow=nrow(Y),ncol=1),ncol(Y),p,0,0,oos=TRUE)
    if(MN){
        Z <- Z[2:nrow(Z),]
        }
    Z <- Z[,ncol(Z)]

    pred <- matrix(B%*%Z,ncol=ncol(Y),nrow=1)

    if(n.ahead==1){return(pred)}
    
    predictMS(pred,Y,n.ahead-1,B,p,MN)
    
}

                                        # Multi-step VARX with new data.
predictMSX <- function(pred,Y,n.ahead,B,p,newxreg,X,m,s,cumulative,MN){

    Y <- rbind(Y,pred)
    X <- rbind(X,matrix(newxreg[1:cumulative,],ncol=m))
    Z <- VARXCons(as.matrix(Y),X,ncol(Y),p,m,s,TRUE)
    Z <- Z[,ncol(Z)]
    if(MN){

        Z <- as.matrix(Z[2:nrow(Z),])
    }    
    pred <- matrix(B%*%Z,ncol=ncol(Y),nrow=1)

    if(n.ahead==1){return(pred)}

    predictMSX(pred,Y,n.ahead-1,B,p,newxreg,X,m,s,cumulative+1,MN)    

}




                                        # Find optimal values in 2-d gridsearch
findind <- function(opt,lambda1,lambda2)
{
    if(opt<length(lambda2)){
        lambda1ind <- 1
    }else{
        lambda1ind <- ceiling(opt/length(lambda2))
    }
    if(lambda1ind==1)
        {
            jind <- opt
        }else{
            jind <- opt-(length(lambda2))*(lambda1ind-1)
        }
    return(c(lambda1ind,jind))
}

