% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Boruta.R
\name{Boruta}
\alias{Boruta}
\alias{Boruta.default}
\alias{Boruta.formula}
\title{Feature selection with the Boruta algorithm}
\usage{
Boruta(x, ...)

\method{Boruta}{default}(
  x,
  y,
  pValue = 0.01,
  mcAdj = TRUE,
  maxRuns = 100,
  doTrace = 0,
  holdHistory = TRUE,
  getImp = getImpRfZ,
  ...
)

\method{Boruta}{formula}(formula, data, ...)
}
\arguments{
\item{x}{data frame of predictors.}

\item{...}{additional parameters passed to \code{getImp}.}

\item{y}{response vector; factor for classification, numeric vector for regression, \code{Surv} object for survival (supports depends on importance adapter capabilities).}

\item{pValue}{confidence level. Default value should be used.}

\item{mcAdj}{if set to \code{TRUE}, a multiple comparisons adjustment using the Bonferroni method will be applied. Default value should be used; older (1.x and 2.x) versions of Boruta were effectively using \code{FALSE}.}

\item{maxRuns}{maximal number of importance source runs.
You may increase it to resolve attributes left Tentative.}

\item{doTrace}{verbosity level. 0 means no tracing, 1 means reporting decision about each attribute as soon as it is justified, 2 means the same as 1, plus reporting each importance source run, 3 means the same as 2, plus reporting of hits assigned to yet undecided attributes.}

\item{holdHistory}{if set to \code{TRUE}, the full history of importance is stored and returned as the \code{ImpHistory} element of the result.
Can be used to decrease a memory footprint of Boruta in case this side data is not used, especially when the number of attributes is huge; yet it disables plotting of such made \code{Boruta} objects and the use of the \code{\link{TentativeRoughFix}} function.}

\item{getImp}{function used to obtain attribute importance.
The default is getImpRfZ, which runs random forest from the \code{ranger} package and gathers Z-scores of mean decrease accuracy measure.
It should return a numeric vector of a size identical to the number of columns of its first argument, containing importance measure of respective attributes.
Any order-preserving transformation of this measure will yield the same result.
It is assumed that more important attributes get higher importance. +-Inf are accepted, NaNs and NAs are treated as 0s, with a warning.}

\item{formula}{alternatively, formula describing model to be analysed.}

\item{data}{in which to interpret formula.}
}
\value{
An object of class \code{Boruta}, which is a list with the following components:
\item{finalDecision}{a factor of three value: \code{Confirmed}, \code{Rejected} or \code{Tentative}, containing final result of feature selection.}
\item{ImpHistory}{a data frame of importances of attributes gathered in each importance source run.
Beside predictors' importances, it contains maximal, mean and minimal importance of shadow attributes in each run.
Rejected attributes get \code{-Inf} importance.
Set to \code{NULL} if \code{holdHistory} was given \code{FALSE}.}
\item{timeTaken}{time taken by the computation.}
\item{impSource}{string describing the source of importance, equal to a comment attribute of the \code{getImp} argument.}
\item{call}{the original call of the \code{Boruta} function.}
}
\description{
Boruta is an all relevant feature selection wrapper algorithm, capable of working with any classification method that output variable importance measure (VIM); by default, Boruta uses Random Forest.
The method performs a top-down search for relevant features by comparing original attributes' importance with importance achievable at random, estimated using their permuted copies, and progressively eliminating irrelevant features to stabilise that test.
}
\details{
Boruta iteratively compares importances of attributes with importances of shadow attributes, created by shuffling original ones.
Attributes that have significantly worst importance than shadow ones are being consecutively dropped.
On the other hand, attributes that are significantly better than shadows are admitted to be Confirmed.
Shadows are re-created in each iteration.
Algorithm stops when only Confirmed attributes are left, or when it reaches \code{maxRuns} importance source runs.
If the second scenario occurs, some attributes may be left without a decision.
They are claimed Tentative.
You may try to extend \code{maxRuns} or lower \code{pValue} to clarify them, but in some cases their importances do fluctuate too much for Boruta to converge.
Instead, you can use \code{\link{TentativeRoughFix}} function, which will perform other, weaker test to make a final decision, or simply treat them as undecided in further analysis.
}
\examples{
set.seed(777)

#Boruta on the "small redundant XOR" problem; read ?srx for details
data(srx)
Boruta(Y~.,data=srx)->Boruta.srx

#Results summary
print(Boruta.srx)

#Result plot
plot(Boruta.srx)

#Attribute statistics
attStats(Boruta.srx)

#Using alternative importance source, rFerns
Boruta(Y~.,data=srx,getImp=getImpFerns)->Boruta.srx.ferns
print(Boruta.srx.ferns)

#Verbose
Boruta(Y~.,data=srx,doTrace=2)->Boruta.srx

\dontrun{
#Boruta on the iris problem extended with artificial irrelevant features
#Generate said features
iris.extended<-data.frame(iris,apply(iris[,-5],2,sample))
names(iris.extended)[6:9]<-paste("Nonsense",1:4,sep="")
#Run Boruta on this data
Boruta(Species~.,data=iris.extended,doTrace=2)->Boruta.iris.extended
#Nonsense attributes should be rejected
print(Boruta.iris.extended)
}

\dontrun{
#Boruta on the HouseVotes84 data from mlbench
library(mlbench); data(HouseVotes84)
na.omit(HouseVotes84)->hvo
#Takes some time, so be patient
Boruta(Class~.,data=hvo,doTrace=2)->Bor.hvo
print(Bor.hvo)
plot(Bor.hvo)
plotImpHistory(Bor.hvo)
}
\dontrun{
#Boruta on the Ozone data from mlbench
library(mlbench); data(Ozone)
library(randomForest)
na.omit(Ozone)->ozo
Boruta(V4~.,data=ozo,doTrace=2)->Bor.ozo
cat('Random forest run on all attributes:\n')
print(randomForest(V4~.,data=ozo))
cat('Random forest run only on confirmed attributes:\n')
print(randomForest(ozo[,getSelectedAttributes(Bor.ozo)],ozo$V4))
}
\dontrun{
#Boruta on the Sonar data from mlbench
library(mlbench); data(Sonar)
#Takes some time, so be patient
Boruta(Class~.,data=Sonar,doTrace=2)->Bor.son
print(Bor.son)
#Shows important bands
plot(Bor.son,sort=FALSE)
}
}
\references{
Miron B. Kursa, Witold R. Rudnicki (2010). Feature Selection with the Boruta Package.
\emph{Journal of Statistical Software, 36(11)}, p. 1-13.
URL: \doi{10.18637/jss.v036.i11}
}
