% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SARestim.R
\name{sar}
\alias{sar}
\title{Estimate SAR model}
\usage{
sar(
  formula,
  contextual,
  Glist,
  lambda0 = NULL,
  fixed.effects = FALSE,
  optimizer = "optim",
  opt.ctr = list(),
  print = TRUE,
  cov = TRUE,
  data
)
}
\arguments{
\item{formula}{an object of class \link[stats]{formula}: a symbolic description of the model. The \code{formula} should be as for example \code{y ~ x1 + x2 | x1 + x2}
where \code{y} is the endogenous vector, the listed variables before the pipe, \code{x1}, \code{x2} are the individual exogenous variables and
the listed variables after the pipe, \code{x1}, \code{x2} are the contextual observable variables. Other formulas may be
\code{y ~ x1 + x2} for the model without contextual effects, \code{y ~ -1 + x1 + x2 | x1 + x2} for the model
without intercept or \code{ y ~ x1 + x2 | x2 + x3} to allow the contextual variable to be different from the individual variables.}

\item{contextual}{(optional) logical; if true, this means that all individual variables will be set as contextual variables. Set the
\code{formula} as \code{y ~ x1 + x2} and \code{contextual} as \code{TRUE} is equivalent to set the formula as \code{y ~ x1 + x2 | x1 + x2}.}

\item{Glist}{the adjacency matrix or list sub-adjacency matrix.}

\item{lambda0}{(optional) starting value of \eqn{\lambda}. The parameter \eqn{\gamma} should be removed if the model
does not contain contextual effects (see details).}

\item{fixed.effects}{logical; if true, group heterogeneity is included as fixed effects.}

\item{optimizer}{is either \code{nlm} (referring to the function \link[stats]{nlm}) or \code{optim} (referring to the function \link[stats]{optim}).
Other arguments
of these functions such as, the control values and the method can be defined through the argument \code{opt.ctr}.}

\item{opt.ctr}{list of arguments of \link[stats]{nlm} or \link[stats]{optim} (the one set in \code{optimizer}) such as control, method, ...}

\item{print}{a Boolean indicating if the estimate should be printed at each step.}

\item{cov}{a Boolean indicating if the covariance should be computed.}

\item{data}{an optional data frame, list or environment (or object coercible by \link[base]{as.data.frame} to a data frame) containing the variables
in the model. If not found in data, the variables are taken from \code{environment(formula)}, typically the environment from which \code{mcmcARD} is called.}
}
\value{
A list consisting of:
\item{info}{list of general information on the model.}
\item{estimate}{Maximum Likelihood (ML) estimator.}
\item{cov}{covariance matrix of the estimate.}
\item{details}{outputs as returned by the optimizer.}
}
\description{
\code{sar} is used to estimate peer effects continuous variables (see details). The model is presented in Lee(2004).
}
\details{
\subsection{Model}{

The variable \eqn{\mathbf{y}}{y} is given for all i as
\deqn{y_i = \lambda \mathbf{g}_i y + \mathbf{x}_i'\beta + \mathbf{g}_i\mathbf{X}\gamma + \epsilon_i,}{y_i = \lambda g_i*y + x_i'\beta + g_i*X\gamma + \epsilon_i,}
where \eqn{\epsilon_i \sim N(0, \sigma^2)}{\epsilon_i --> N(0, \sigma^2)}.
}
}
\examples{
\donttest{
# Groups' size
M      <- 5 # Number of sub-groups
nvec   <- round(runif(M, 100, 1000))
n      <- sum(nvec)

# Parameters
lambda <- 0.4
beta   <- c(2, -1.9, 0.8)
gamma  <- c(1.5, -1.2)
sigma  <- 1.5
theta  <- c(lambda, beta, gamma, sigma)

# X
X      <- cbind(rnorm(n, 1, 1), rexp(n, 0.4))

# Network
Glist  <- list()

for (m in 1:M) {
  nm           <- nvec[m]
  Gm           <- matrix(0, nm, nm)
  max_d        <- 30
  for (i in 1:nm) {
    tmp        <- sample((1:nm)[-i], sample(0:max_d, 1))
    Gm[i, tmp] <- 1
  }
  rs           <- rowSums(Gm); rs[rs == 0] <- 1
  Gm           <- Gm/rs
  Glist[[m]]   <- Gm
}


# data
data    <- data.frame(x1 = X[,1], x2 =  X[,2])

rm(list = ls()[!(ls() \%in\% c("Glist", "data", "theta"))])

ytmp    <- simsar(formula = ~ x1 + x2 | x1 + x2, Glist = Glist,
                  theta = theta, data = data) 

y       <- ytmp$y

# plot histogram
hist(y, breaks = max(y))

data    <- data.frame(yt = y, x1 = data$x1, x2 = data$x2)
rm(list = ls()[!(ls() \%in\% c("Glist", "data"))])

out     <- sar(formula = yt ~ x1 + x2, contextual = TRUE, 
                 Glist = Glist, optimizer = "optim", data = data)
summary(out)
}
}
\references{
Lee, L. F. (2004). Asymptotic distributions of quasi-maximum likelihood estimators for spatial autoregressive models. \emph{Econometrica}, 72(6), 1899-1925, \doi{10.1111/j.1468-0262.2004.00558.x}.
}
\seealso{
\code{\link{sart}}, \code{\link{cdnet}}, \code{\link{simsar}}.
}
