% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GetAllStateChanges.R
\name{GetAllStateChanges}
\alias{GetAllStateChanges}
\title{Finds all state changes on a tree using stochastic character mapping}
\usage{
GetAllStateChanges(CladisticMatrix, Tree, TimeBins, NSimulations = 10,
  PolymorphismBehaviour = "equalp", UncertaintyBehaviour = "equalp",
  InapplicableBehaviour = "missing")
}
\arguments{
\item{CladisticMatrix}{A character-taxon matrix in the format imported by \link{ReadMorphNexus}.}

\item{Tree}{A time-scaled tree (phylo object) that represents the relationships of the taxa in \code{CladisticMatrix}.}

\item{TimeBins}{A vector of ages representing the boundaries of a series of time bins.}

\item{NSimulations}{The number of simulations to perform (passed to \code{make.simmap}.}

\item{PolymorphismBehaviour}{What to do with polymorphic (&) characters. One of "equalp", "missing", or "random". See details.}

\item{UncertaintyBehaviour}{What to do with uncertain (/) characters. One of "equalp", "missing", or "random". See details.}

\item{InapplicableBehaviour}{What to do with inapplicable characters. Only one option currently ("missing"). See details.}
}
\value{
\item{RootStates}{A matrix of the root states for each character (column) and simulation (rows).}
\item{AllStateChanges}{A matrix of rows for each change with columns corresponding to the character, the simulation number, the edge number, the time the change occurred, and the start and end states.}
\item{CharacterTimes}{A vector of the sampled tree-length (in Ma) for each character.}
\item{EdgeLengthsPerBin}{A matrix of time bins (columns) and characters (rows) indicating the sampled tree-length (in Ma).}
\item{TerminalEdgeLengthsPerBin}{As above, but for terminal edges only.}
\item{InternalEdgeLengthsPerBin}{As above, but for internal edges only.}
}
\description{
Takes a cladistic matrix and time-scaled tree and makes point estimates for every character change using stochastic character mapping.
}
\details{
Important: this function is not yet complete and should not be used.

A wrapper function for \link{make.simmap} in the \link{phytools} package.

This function is intended to enumerate all possible changes on a tree (including to and from missing or inapplicable states) under the assumptions of stochastic character mapping as an alternative means of establishing branch-lengths (for rate analyses) or recording the state occupied at a particular point in time for disparity analyses.
}
\examples{

# Set random seed:
set.seed(2)

# Use Day 2016 as source matrix:
CladisticMatrix <- Day2016

# Prune out continuous characters:
CladisticMatrix <- MatrixPruner(CladisticMatrix =
  CladisticMatrix, blocks2prune = 1)

# Prune out majority of characters so
# example runs quickly:
CladisticMatrix <- MatrixPruner(CladisticMatrix =
  CladisticMatrix, characters2prune = 1:32)

# Generete random tree for matrix taxa:
Tree <- rtree(nrow(Day2016$Matrix_1$Matrix))

# Add taxon names to tree:
Tree$tip.label <- rownames(Day2016$Matrix_1$Matrix)

# Add root age to tree:
Tree$root.time <- max(diag(vcv(Tree)))

# Get all state changes for two simulations:
StateChanges <-
  GetAllStateChanges(CladisticMatrix = CladisticMatrix,
  Tree = Tree, TimeBins = seq(Tree$root.time, 0,
  length.out = 3), NSimulations = 2)

# View matrix of all stochstic character changes:
StateChanges$AllStateChanges

# View vector of sampled time for each
# character:
StateChanges$CharacterTimes

# View matrix of edge lengths in each time bin:
StateChanges$EdgeLengthsPerBin

# View matrix of termnial edge lengths in each time bin:
StateChanges$TerminalEdgeLengthsPerBin

# View matrix of internal edge lengths in each time bin:
StateChanges$InternalEdgeLengthsPerBin

}
\author{
Graeme T. Lloyd \email{graemetlloyd@gmail.com}
}
