% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/colles.like.index.R
\name{colless.like.index}
\alias{colless.like.index}
\title{Computes the Colless-like balance index of a phylogenetic tree}
\usage{
colless.like.index(tree, f.size = "ln", diss = "MDM", norm = FALSE)
}
\arguments{
\item{tree}{a single phylogenetic tree. It can be entered as a string in Newick format, as a 'phylo' object (\code{ape} package) or as an 'igraph' object (\code{igraph} package).}

\item{f.size}{function to compute the f-size of the tree. See (Mir et al. , 2017) for details. Its default value is "ln" for f(n)=ln(n+e). Other value can be "exp" (f(n)=exp(n)). It can also be a user-defined function but in this case, the index cannot be normalized}

\item{diss}{by default, the dissimilarity used to compute the balance index. See (Mir et al. , 2017) for details. Its default value is MDM (mean deviation from the median). Other values can be set as "sd" (sample standard deviation) or "var" (sample variance) . It can also be a user-defined function but in this case the index cannot be normalized.}

\item{norm}{a logical object indicating if the indices should be normalized or not.}
}
\value{
A numeric value.
}
\description{
Given a phylogenetic tree, computes the Colless-like balance index of that phylogenetic tree.
}
\details{
The Colless-Like balance index is the generalization of the Colless balance index (see Colless,1982) for non-binary trees.

Given a function that computes the f-size of a tree and a dissimarity function that computes the difference of the f-sizes of the subtrees rooted at the children of every internal node of the tree, the Colless-Like index is defined as the sum of these dissimilarities for all internal nodes of the tree. (Mir et al. , 2017)

By default, the f-size function is \code{f(n)=exp(n)} and the dissimilarity is the mean deviation from the median (MDM). 
It is possible to change them by specifying it with the parameters \code{f.size} and \code{diss}, with "exp" the f-size would be \code{f(n)=exp(n)}, and with "var" (or "sd") the dissimilarity would be the sample variance (or the sample standard deviation).
It is also possible to set a new function for both parameters, see "References".
}
\examples{
# Computation of the Colless-Like balance index of trees 
# entered in newick format:
colless.like.index("(1,2,3,4,5);")
colless.like.index("(1,(2,(3,(4,5))));")

# Computation of the Colless-Like balance index of trees 
# entered as a phylo object:
require(ape)
random.tree = rtree(5,rooted=TRUE)
colless.like.index(random.tree)

# Computation of the Colless-Like balance index of a tree
# entered as a igraph object. The tree is randomly 
# generated from all trees with 5 leaves following
# the alpha-gamma model with alpha=0.5 and gamma=0.3.
a.g.tree = a.g.model(5,0.5,0.3)
colless.like.index(a.g.tree)

# All of them can be normalized (values between 0 and 1)
colless.like.index("(1,2,3,4,5);",norm=TRUE)
colless.like.index("(1,(2,(3,(4,5))));",norm=TRUE)
colless.like.index(random.tree,norm=TRUE)
colless.like.index(a.g.tree,norm=TRUE)

# Computation of the Colless-Like balance index of the
# previous generated tree with f-size function f(n)=exp(n):
colless.like.index(a.g.tree,f.size="exp")

# Computation of the Colless-Like balance index of the 
# previous generated tree that sets the sample variance 
# and the sample standard deviation as dissimilarity.
colless.like.index(a.g.tree,diss="var")
colless.like.index(a.g.tree,diss="sd")

# Computation of the Colless-Like balance index of the 
# previous generated tree with f-size function f(n)=exp(n)
# that sets the sample variance and the sample standard 
# deviation as dissimilarity.
colless.like.index(a.g.tree,f.size="exp",diss="var")
colless.like.index(a.g.tree,f.size="exp",diss="sd")

}
\references{
A. Mir, F. Rossello, L.Rotger, A Colless-like balance index for multifurcating phylogenetic trees.\emph{}

D. H. Colless, Review of "Phylogenetics: the theory and practice of phylogenetic systematics".
\emph{Sys. Zool}, \bold{31} (1982), 100--104.
}
\author{
Lucia Rotger
}
