% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/D3mirt.R
\name{D3mirt}
\alias{D3mirt}
\title{3D DMIRT Model Estimation}
\usage{
D3mirt(x, constructs = NULL)
}
\arguments{
\item{x}{A data frame with rows for items and columns for model parameters or an S4 object of class 'SingleGroupClass' exported from \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012).
Regarding the data frame, the number of columns must be more than or equal to 4, i.e., three columns with (\emph{a}) parameters and at least one column for difficulty (\emph{d}) parameters.}

\item{constructs}{Optional. Nested lists with integer values indicating construct. The default is \code{constructs = NULL}.}
}
\value{
A S3 object of class \code{D3mirt} with lists of \emph{a} and \emph{d} parameters from the M2PL or MGRM estimation, multidimensional difficulty (MDIFF), multidimensional discrimination (MDISC), direction cosines and degrees for vector angles, construct lists, and vector coordinates.
}
\description{
Descriptive multidimensional item response theory model estimation (DMIRT; Reckase, 2009, 1985, Reckase & McKinley, 1991) for dichotomous and polytomous items restricted to three dimensions.
}
\details{
The \code{D3mirt()} function takes in model parameters from a compensatory three-dimensional multidimensional two-parameter logistic model (M2PL) or a multidimensional graded
response model (MGRM), either in the form of a data frame or an S4 object of class 'SingleGroupClass' exported from \link[mirt:mirt]{mirt::mirt} (Chalmers, 2012) function fitted in accordance with the descriptive item response theory model specifications described below.
The function returns DMIRT estimates that can be visualized with \link{plot} that graph vector arrows representing item response characteristics in a three-dimensional space.
Regarding the former, this includes visualization of the single multidimensional discrimination (MDISC) parameter and the multidimensional difficulty (MDIFF) parameters (Reckase, 2009, 1985; Reckase & McKinley, 1991).

The user has the option of including constructs in the estimation.
Constructs, in this context, refer to the assumption that a subset of items can measure a higher-order latent variable.
To include constructs, the user must create one or more nested lists that indicate what items belong to what construct (from one item up to all items in the set; see the examples section below).
From this, the \code{D3mirt()} function calculates the average direction by adding and normalizing the direction cosines using the items in the nested lists.
Constructs are visualized when plotting as solid black arrows running across the model space.
In addition, if constructs are used the output will also contain the directional discrimination (DDISC) parameters for all items assessed in the direction indicated by the construct vectors.
This makes it possible to compare item discrimination under the assumption that they measure the same latent variable.

Note, model parameters from the multidimensional M2PL or MGRM must be assessed prior to using the \code{D3mirt()} function (see examples section below or the package vignette).
This means that the model must first be identified (see \link{modid} for more on model identification).
For more on theory and how to interpret statistical estimates, please see the package vignette.
}
\examples{
\donttest{
# Load data
data("anes0809offwaves")
x <- anes0809offwaves
x <- x[,3:22] # Remove columns for age and gender

# Fit a three-dimensional graded response model with orthogonal factors
# Example below uses Likert items from the built-in data set "anes0809offwaves"
# Item W7Q3 and item W7Q20 was selected with `modid()`
# The model specification set all items in the data set (1-20)
# to load on all three factors (F1-F3)
# The START and FIXED commands are used on the two items to identify the DMIRT model
spec <- '  F1 = 1-20
           F2 = 1-20
           F3 = 1-20

           START=(W7Q3,a2,0)
           START=(W7Q3,a3,0)

           START=(W7Q20,a3,0)

           FIXED=(W7Q3,a2)
           FIXED=(W7Q3,a3)

           FIXED=(W7Q20,a3) '


mod1 <- mirt::mirt(x,
                   spec,
                   itemtype = 'graded',
                   SE = TRUE,
                   method = 'QMCEM')

# Optional: Load the mod1 data as a data frame directly from the package file
# load(system.file("extdata/mod1.Rdata", package = "D3mirt"))

# Call D3mirt() using the mod1 as input
g <- D3mirt(mod1)

# Show summary of results
summary(g)

# Call to D3mirt(), including optional nested lists for three constructs
# Item W7Q16 is not included in any construct because of model violations
# The model violations for the item W7Q16 can be seen when plotting the model
c <- list(list(1,2,3,4,5,6,7,8,9,10),
          list(11,12,13,14),
          list(15,17,18,19,20))
g <- D3mirt(mod1, c)

# Show summary of results
summary(g)
}
}
\references{
Chalmers, R., P. (2012). mirt: A Multidimensional Item Response Theory Package for the R Environment. \emph{Journal of Statistical Software, 48}(6), 1-29.

Reckase, M. D. (2009). \emph{Multidimensional Item Response Theory}. Springer.

Reckase, M. D. (1985). The Difficulty of Test Items That Measure More Than One Ability. \emph{Applied Psychological Measurement, 9}(4), 401-412. https://doi-org.ezp.sub.su.se/10.1177/014662168500900409

Reckase, M. D., & McKinley, R. L. (1991). The Discriminating Power of Items That Measure More Than One Dimension. \emph{Applied Psychological Measurement, 15}(4), 361-373. https://doi-org.ezp.sub.su.se/10.1177/014662169101500407
}
\author{
Erik Forsberg
}
