% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/runVoom.R
\name{runVoom}
\alias{runVoom}
\title{Run functions in a typical voom/lmFit workflow}
\usage{
runVoom(
  dgeObj,
  designMatrixName,
  dupCorBlock,
  runDupCorTwice = TRUE,
  qualityWeights = TRUE,
  var.design,
  mvPlot = TRUE,
  runEBayes = TRUE,
  robust = TRUE,
  proportion = 0.01
)
}
\arguments{
\item{dgeObj}{A DGEobj containing a DGEList (e.g. from runEdgeRNorm.) (Required)}

\item{designMatrixName}{Name of a design matrix within dgeObj. (Required)}

\item{dupCorBlock}{Supply a block argument to trigger duplicateCorrelation. (Optional)
Should be a vector the same length as ncol with values to indicate common
group membership for duplicateCorrelation.}

\item{runDupCorTwice}{Default = TRUE. Gordon Smyth recommends running duplicateCorrelation
twice. Set this to false to run duplicateCorrelation just once.}

\item{qualityWeights}{Runs limma::voomWithQualityWeights if set to TRUE (Default = TRUE).
This should normally be set to TRUE.}

\item{var.design}{Provide a design matrix (from model.matrix) to identify
replicate groups (e.g. "~ ReplicateGroup") for quality weight determination.
Causes quality weights to be determined on a group basis.  If omitted
limma::voomWithQualityWeights treats each sample individually.}

\item{mvPlot}{Enables the voom mean-variance plot. (Default = TRUE)}

\item{runEBayes}{Runs eBayes after lmFit. (Default = TRUE)}

\item{robust}{Used by eBayes. (Default = TRUE)}

\item{proportion}{Proportion of genes expected to be differentially expressed
(used by eBayes) (Default = 0.01) Modify the prior accordingly if the 1st pass analysis shows
a significantly higher or lower proportion of genes regulated than the default.}
}
\value{
A DGEobj now containing designMatrix, Elist, and fit object.
}
\description{
In the default workflow, this function runs voomWithQualityWeights followed by
lmFit and optionally eBayes. If the contrasts of interest are already represented
in the model, enable eBayes. To use contrasts.fit downstream, run eBayes
after that step instead. eBayes should always be run last.
}
\details{
Input is minimally a DGEobj containing a DGEList (typically TMM-normalized),
and a formula (character representation).  Other arguments can invoke
the duplicateCorrelation method and modify use of quality weights.

Returns a DGEobj class object containing the VoomElist (voom
output), and Fit object (lmFit output).

Quality weights should be enabled unless there is a good reason to turn them
off. If all samples are equal quality, the weights will all approach 1.0 with no
consequence on the results. More typically, some samples are better than others
and using quality weights improves the overall result.

Use var.design if the quality weights are correlated with some factor in the experiment.
This will cause the quality weights to be calculated as a group instead of individually.

Use duplicate correlation (dupCorBlock) when there are subjects that have been sampled more
than once (e.g. before and after some treatment).  This calculates a within-
subject correlation and includes this in the model.
}
\examples{
   dgeObj <- readRDS(system.file("exampleObj.RDS", package = "DGEobj"))
   for (name in names(dgeObj)[11:length(dgeObj)]) {
       dgeObj <- DGEobj::rmItem(dgeObj, name)
   }

   dgeObj <- runVoom(dgeObj,
                     designMatrixName = "ReplicateGroupDesign",
                     mvPlot = TRUE)

   # Note the Elist and fit objects have been added
   DGEobj::inventory(dgeObj)

}
