% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/StatModels.R
\name{LinearRegressionDP}
\alias{LinearRegressionDP}
\title{Privacy-preserving Linear Regression}
\description{
This class implements differentially private linear regression
using the objective perturbation technique \insertCite{Kifer2012}{DPpack}.
}
\details{
To use this class for linear regression, first use the \code{new}
method to construct an object of this class with the desired function
values and hyperparameters. After constructing the object, the \code{fit}
method can be applied with a provided dataset and data bounds to fit the
model. In fitting, the model stores a vector of coefficients \code{coeff}
which satisfy differential privacy. These can be released directly, or used
in conjunction with the \code{predict} method to privately predict the
outcomes of new datapoints.

Note that in order to guarantee differential privacy for linear regression,
certain constraints must be satisfied for the values used to construct the
object, as well as for the data used to fit. The regularizer must be
convex. Additionally, it is assumed that if x represents a single row of
the dataset X, then the l2-norm of x is at most p for all x, where p is the
number of predictors (including any possible intercept term). In order to
ensure this constraint is satisfied, the dataset is preprocessed and
scaled, and the resulting coefficients are postprocessed and un-scaled so
that the stored coefficients correspond to the original data. Due to this
constraint on x, it is best to avoid using an intercept term in the model
whenever possible. If an intercept term must be used, the issue can be
partially circumvented by adding a constant column to X before fitting the
model, which will be scaled along with the rest of X. The \code{fit} method
contains functionality to add a column of constant 1s to X before scaling,
if desired.
}
\examples{
# Build example dataset
n <- 500
X <- data.frame(X=seq(-1,1,length.out = n))
true.theta <- c(-.3,.5) # First element is bias term
p <- length(true.theta)
y <- true.theta[1] + as.matrix(X)\%*\%true.theta[2:p] + stats::rnorm(n=n,sd=.1)

# Construct object for linear regression
regularizer <- 'l2' # Alternatively, function(coeff) coeff\%*\%coeff/2
eps <- 1
delta <- 0 # Indicates to use pure eps-DP
gamma <- 1
regularizer.gr <- function(coeff) coeff

lrdp <- LinearRegressionDP$new('l2', eps, delta, gamma, regularizer.gr)

# Fit with data
# We must assume y is a matrix with values between -p and p (-2 and 2
#   for this example)
upper.bounds <- c(1, 2) # Bounds for X and y
lower.bounds <- c(-1,-2) # Bounds for X and y
lrdp$fit(X, y, upper.bounds, lower.bounds, add.bias=TRUE)
lrdp$coeff # Gets private coefficients

# Predict new data points
# Build a test dataset
Xtest <- data.frame(X=c(-.5, -.25, .1, .4))
predicted.y <- lrdp$predict(Xtest, add.bias=TRUE)

}
\references{
\insertRef{Kifer2012}{DPpack}
}
\section{Super class}{
\code{\link[DPpack:EmpiricalRiskMinimizationDP.KST]{DPpack::EmpiricalRiskMinimizationDP.KST}} -> \code{LinearRegressionDP}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-LinearRegressionDP-new}{\code{LinearRegressionDP$new()}}
\item \href{#method-LinearRegressionDP-fit}{\code{LinearRegressionDP$fit()}}
\item \href{#method-LinearRegressionDP-clone}{\code{LinearRegressionDP$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="DPpack" data-topic="EmpiricalRiskMinimizationDP.KST" data-id="predict"><a href='../../DPpack/html/EmpiricalRiskMinimizationDP.KST.html#method-EmpiricalRiskMinimizationDP.KST-predict'><code>DPpack::EmpiricalRiskMinimizationDP.KST$predict()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-LinearRegressionDP-new"></a>}}
\if{latex}{\out{\hypertarget{method-LinearRegressionDP-new}{}}}
\subsection{Method \code{new()}}{
Create a new LinearRegressionDP object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{LinearRegressionDP$new(regularizer, eps, delta, gamma, regularizer.gr = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{regularizer}}{String or regularization function. If a string, must be
'l2', indicating to use l2 regularization. If a function, must have form
\code{regularizer(coeff)}, where \code{coeff} is a vector or matrix, and
return the value of the regularizer at \code{coeff}. See
\code{\link{regularizer.l2}} for an example. Additionally, in order to
ensure differential privacy, the function must be convex.}

\item{\code{eps}}{Positive real number defining the epsilon privacy budget. If set
to Inf, runs algorithm without differential privacy.}

\item{\code{delta}}{Nonnegative real number defining the delta privacy parameter.
If 0, reduces to pure eps-DP.}

\item{\code{gamma}}{Nonnegative real number representing the regularization
constant.}

\item{\code{regularizer.gr}}{Optional function representing the gradient of the
regularization function with respect to \code{coeff} and of the form
\code{regularizer.gr(coeff)}. Should return a vector. See
\code{\link{regularizer.gr.l2}} for an example. If \code{regularizer} is
given as a string, this value is ignored. If not given and
\code{regularizer} is a function, non-gradient based optimization methods
are used to compute the coefficient values in fitting the model.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A new LinearRegressionDP object.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-LinearRegressionDP-fit"></a>}}
\if{latex}{\out{\hypertarget{method-LinearRegressionDP-fit}{}}}
\subsection{Method \code{fit()}}{
Fit the differentially private linear regression model. The
function runs the objective perturbation algorithm
\insertCite{Kifer2012}{DPpack} to generate an objective function. A
numerical optimization method is then run to find optimal coefficients
for fitting the model given the training data and hyperparameters. The
\code{\link[nloptr]{nloptr}} function is used. If \code{regularizer} is given as
'l2' or if \code{regularizer.gr} is given in the construction of the
object, the gradient of the objective function and the Jacobian of the
constraint function are utilized for the algorithm, and the NLOPT_LD_MMA
method is used. If this is not the case, the NLOPT_LN_COBYLA method is
used. The resulting privacy-preserving coefficients are stored in coeff.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{LinearRegressionDP$fit(X, y, upper.bounds, lower.bounds, add.bias = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{X}}{Dataframe of data to be fit.}

\item{\code{y}}{Vector or matrix of true values for each row of \code{X}.}

\item{\code{upper.bounds}}{Numeric vector of length \code{ncol(X)+1} giving upper
bounds on the values in each column of \code{X} and the values of
\code{y}. The last value in the vector is assumed to be the upper bound
on \code{y}, while the first \code{ncol(X)} values are assumed to be in
the same order as the corresponding columns of \code{X}. Any value in the
columns of \code{X} and in \code{y} larger than the corresponding upper
bound is clipped at the bound.}

\item{\code{lower.bounds}}{Numeric vector of length \code{ncol(X)+1} giving lower
bounds on the values in each column of \code{X} and the values of
\code{y}. The last value in the vector is assumed to be the lower bound
on \code{y}, while the first \code{ncol(X)} values are assumed to be in
the same order as the corresponding columns of \code{X}. Any value in the
columns of \code{X} and in \code{y} larger than the corresponding lower
bound is clipped at the bound.}

\item{\code{add.bias}}{Boolean indicating whether to add a bias term to \code{X}.
Defaults to FALSE.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-LinearRegressionDP-clone"></a>}}
\if{latex}{\out{\hypertarget{method-LinearRegressionDP-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{LinearRegressionDP$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
