% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/network.estimation.R
\name{network.estimation}
\alias{network.estimation}
\title{Apply a Network Estimation Method}
\usage{
network.estimation(
  data,
  n = NULL,
  corr = c("auto", "cor_auto", "cosine", "pearson", "spearman"),
  na.data = c("pairwise", "listwise"),
  model = c("BGGM", "glasso", "TMFG"),
  network.only = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{data}{Matrix or data frame.
Should consist only of variables to be used in the analysis}

\item{n}{Numeric (length = 1).
Sample size if \code{data} provided is a correlation matrix}

\item{corr}{Character (length = 1).
Method to compute correlations.
Defaults to \code{"auto"}.
Available options:

\itemize{

\item \code{"auto"} --- Automatically computes appropriate correlations for
the data using Pearson's for continuous, polychoric for ordinal,
tetrachoric for binary, and polyserial/biserial for ordinal/binary with
continuous. To change the number of categories that are considered
ordinal, use \code{ordinal.categories}
(see \code{\link[EGAnet]{polychoric.matrix}} for more details)

\item \code{"cor_auto"} --- Uses \code{\link[qgraph]{cor_auto}} to compute correlations.
Arguments can be passed along to the function

\item \code{"cosine"} --- Uses \code{\link[EGAnet]{cosine}} to compute cosine similarity

\item \code{"pearson"} --- Pearson's correlation is computed for all
variables regardless of categories

\item \code{"spearman"} --- Spearman's rank-order correlation is computed
for all variables regardless of categories

}

For other similarity measures, compute them first and input them
into \code{data} with the sample size (\code{n})}

\item{na.data}{Character (length = 1).
How should missing data be handled?
Defaults to \code{"pairwise"}.
Available options:

\itemize{

\item \code{"pairwise"} --- Computes correlation for all available cases between
two variables

\item \code{"listwise"} --- Computes correlation for all complete cases in the dataset

}}

\item{model}{Character (length = 1).
Defaults to \code{"glasso"}.
Available options:

\itemize{

\item \code{"BGGM"} --- Computes the Bayesian Gaussian Graphical Model.
Set argument \code{ordinal.categories} to determine
levels allowed for a variable to be considered ordinal.
See \code{?BGGM::estimate} for more details

\item \code{"glasso"} --- Computes the GLASSO with EBIC model selection.
See \code{\link[EGAnet]{EBICglasso.qgraph}} for more details

\item \code{"TMFG"} --- Computes the TMFG method.
See \code{\link[EGAnet]{TMFG}} for more details

}}

\item{network.only}{Boolean (length = 1).
Whether the network only should be output.
Defaults to \code{TRUE}.
Set to \code{FALSE} to obtain all output for the
network estimation method}

\item{verbose}{Boolean (length = 1).
Whether messages and (insignificant) warnings should be output.
Defaults to \code{FALSE} (silent calls).
Set to \code{TRUE} to see all messages and warnings for every function call}

\item{...}{Additional arguments to be passed on to
\code{\link[EGAnet]{auto.correlate}} and the different
network estimation methods (see \code{model} for
model specific details)}
}
\value{
Returns a matrix populated with a network from the input data
}
\description{
General function to apply network estimation methods in \code{\link{EGAnet}}
}
\examples{
# Load data
wmt <- wmt2[,7:24]

# EBICglasso (default for EGA functions)
glasso_network <- network.estimation(
  data = wmt, model = "glasso"
)

# TMFG
tmfg_network <- network.estimation(
  data = wmt, model = "TMFG"
)

}
\references{
\strong{Graphical Least Absolute Shrinkage and Selection Operator (GLASSO)} \cr
Friedman, J., Hastie, T., & Tibshirani, R. (2008).
Sparse inverse covariance estimation with the graphical lasso.
\emph{Biostatistics}, \emph{9}(3), 432–441.

\strong{GLASSO with Extended Bayesian Information Criterion (EBICglasso)} \cr
Epskamp, S., & Fried, E. I. (2018).
A tutorial on regularized partial correlation networks.
\emph{Psychological Methods}, \emph{23}(4), 617–634.

\strong{Bayesian Gaussian Graphical Model (BGGM)} \cr
Williams, D. R. (2021).
Bayesian estimation for Gaussian graphical models: Structure learning, predictability, and network comparisons.
\emph{Multivariate Behavioral Research}, \emph{56}(2), 336–352.

\strong{Triangulated Maximally Filtered Graph (TMFG)} \cr
Massara, G. P., Di Matteo, T., & Aste, T. (2016).
Network filtering for big data: Triangulated maximally filtered graph.
\emph{Journal of Complex Networks}, \emph{5}, 161-178.
}
\author{
Hudson Golino <hfg9s at virginia.edu> and Alexander P. Christensen <alexpaulchristensen@gmail.com>
}
