% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/factor_analysis.R
\name{align_loadings}
\alias{align_loadings}
\title{Reorder MCMC Samples of Factor Loadings}
\usage{
align_loadings(
  emc = NULL,
  lambda = NULL,
  n_cores = 1,
  verbose = TRUE,
  rotate_fun = NULL
)
}
\arguments{
\item{emc}{an 'emc' object of type \code{infnt_factor}.}

\item{lambda}{Needs to be supplied if emc is not supplied.
Array of factor loadings with dimensions p (variables) x q (factors) x n (MCMC iterations)}

\item{n_cores}{Number of cores for parallel processing}

\item{verbose}{Logical; whether to print progress information}

\item{rotate_fun}{A function that returns an orthogonally rotated factor loadings matrix. If NULL uses \code{varimax}}
}
\value{
A list containing:
\item{lambda_reordered}{Array of reordered loadings}
\item{lambda_reordered_mcmc}{Array of reordered loadings as MCMC object}
\item{lambda_hat}{Matrix of mean loadings after reordering}
\item{v_vectors}{Matrix of permutation vectors}
\item{c_vectors}{Matrix of sign-switching vectors}
}
\description{
This function reorders MCMC samples of factor loadings to address the label switching problem
in Bayesian factor analysis. It implements a parallelized version of the code and
algorithm proposed by Papastamoulis and Ntzoufras (2022)
}
\examples{
# This function works natively with emc objects, but also factor arrays:
# Simulate a small example with 5 variables, 2 factors, and 10 MCMC iterations
set.seed(123)
p <- 5  # Number of variables
q <- 2  # Number of factors
n <- 10 # Number of MCMC iterations

# Create random factor loadings with label switching
lambda <- array(0, dim = c(p, q, n))
for (i in 1:n) {
  # Generate base loadings
  base_loadings <- matrix(rnorm(p*q, 0, 0.5), p, q)
  base_loadings[1:3, 1] <- abs(base_loadings[1:3, 1]) + 0.5  # Strong loadings on factor 1
  base_loadings[4:5, 2] <- abs(base_loadings[4:5, 2]) + 0.5  # Strong loadings on factor 2

  # Randomly switch labels and signs
  if (runif(1) > 0.5) {
    # Switch factor order
    base_loadings <- base_loadings[, c(2, 1)]
  }
  if (runif(1) > 0.5) {
    # Switch sign of factor 1
    base_loadings[, 1] <- -base_loadings[, 1]
  }
  if (runif(1) > 0.5) {
    # Switch sign of factor 2
    base_loadings[, 2] <- -base_loadings[, 2]
  }

  lambda[,,i] <- base_loadings
}

# Align the loadings
result <- align_loadings(lambda = lambda, verbose = TRUE, n_cores = 1)

# Examine the aligned loadings
print(result)

}
\references{
Papastamoulis, P., & Ntzoufras, I. (2022). On the identifiability of Bayesian factor
analytic models. \emph{Statistical Computing}, 32(2), 1-29. doi: 10.1007/s11222-022-10084-4
}
