% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EllDistrEst.R
\name{EllDistrEst}
\alias{EllDistrEst}
\title{Nonparametric estimation of the density generator of an elliptical distribution}
\usage{
EllDistrEst(
  X,
  mu = 0,
  Sigma_m1 = diag(d),
  grid,
  h,
  Kernel = "epanechnikov",
  a = 1,
  mpfr = FALSE,
  precBits = 100,
  dopb = TRUE
)
}
\arguments{
\item{X}{matrix of observations.}

\item{mu}{(estimated) mean of X.}

\item{Sigma_m1}{(estimated) inverse of the covariance matrix of X.}

\item{grid}{grid of values on which to estimate the density generator}

\item{h}{bandwidth of the kernel}

\item{Kernel}{kernel used for the smoothing}

\item{a}{tuning parameter to improve the performance at 0.
See Liebscher (2005), Example p.210.}

\item{mpfr}{if \code{mpfr = TRUE}, multiple precision floating point is set.
This allows for a higher accuracy, at the expense of computing times.
It is recommended to use this option for higher dimensions.}

\item{precBits}{number of precBits used for floating point precision
(only used if \code{mpfr = TRUE}).}

\item{dopb}{if \code{dopb = TRUE}, a progressbar is displayed.}
}
\value{
the values of the density generator of the elliptical copula,
estimated at each point of the \code{grid}.
}
\description{
This function uses Liebscher's algorithm to estimate the density generator
of an elliptical distribution by kernel smoothing.
}
\examples{
# Comparison between the estimated and true generator of the Gaussian distribution
X = matrix(rnorm(500*3), ncol = 3)
grid = seq(0,5,by=0.1)
g_3 = EllDistrEst(X = X, grid = grid, a = 0.7, h=0.05)
g_3mpfr = EllDistrEst(X = X, grid = grid, a = 0.7, h=0.05,
                      mpfr = TRUE, precBits = 20)
plot(grid, g_3, type = "l")
lines(grid, exp(-grid/2)/(2*pi)^{3/2}, col = "red")

# In higher dimensions
\donttest{
d = 250
X = matrix(rnorm(500*d), ncol = d)
grid = seq(0, 400, by = 25)
true_g = exp(-grid/2) / (2*pi)^{d/2}

g_d = EllDistrEst(X = X, grid = grid, a = 100, h=40)

g_dmpfr = EllDistrEst(X = X, grid = grid, a = 100, h=40,
                      mpfr = TRUE, precBits = 10000)
ylim = c(min(c(true_g, as.numeric(g_dmpfr[which(g_dmpfr>0)]))),
         max(c(true_g, as.numeric(g_dmpfr)), na.rm=TRUE) )
plot(grid, g_dmpfr, type = "l", col = "red", ylim = ylim, log = "y")
lines(grid, g_d, type = "l")
lines(grid, true_g, col = "blue")
}

}
\references{
Liebscher, E. (2005).
A semiparametric density estimator based on elliptical distributions.
Journal of Multivariate Analysis, 92(1), 205.
\doi{10.1016/j.jmva.2003.09.007}
}
\seealso{
\code{\link{EllDistrSim}} for the simulation of elliptical distribution samples,
\code{\link{EllCopEst}} for the estimation of elliptical copulas.
}
\author{
Alexis Derumigny, Rutger van der Spek
}
