% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plot.R
\name{plot.dcm}
\alias{plot.dcm}
\title{Plot Data from a Deterministic Compartmental Epidemic Model}
\usage{
\method{plot}{dcm}(
  x,
  y,
  popfrac = FALSE,
  run,
  col,
  lwd,
  lty,
  alpha = 0.9,
  legend,
  leg.name,
  leg.cex = 0.8,
  axs = "r",
  grid = FALSE,
  add = FALSE,
  ...
)
}
\arguments{
\item{x}{An \code{EpiModel} object of class \code{dcm}.}

\item{y}{Output compartments or flows from \code{dcm} object to plot.}

\item{popfrac}{If \code{TRUE}, plot prevalence of values rather than numbers
(see details).}

\item{run}{Run number to plot, for models with multiple runs
(default is run 1).}

\item{col}{Color for lines, either specified as a single color in a standard
R color format, or alternatively as a color palette from
\code{\link{RColorBrewer}} (see details).}

\item{lwd}{Line width for output lines.}

\item{lty}{Line type for output lines.}

\item{alpha}{Transparency level for lines, where 0 = transparent and
1 = opaque (see \code{adjustcolor} function).}

\item{legend}{Type of legend to plot. Values are "n" for no legend, "full"
for full legend, and "lim" for limited legend (see details).}

\item{leg.name}{Character string to use for legend, with the default
determined automatically based on the \code{y} input.}

\item{leg.cex}{Legend scale size.}

\item{axs}{Plot axis type (see \code{\link{par}} for details), with default
of "r".}

\item{grid}{If \code{TRUE}, a grid is added to the background of plot
(see \code{\link{grid}} for details), with default of nx by ny.}

\item{add}{If \code{TRUE}, new plot window is not called and lines are added
to existing plot window.}

\item{...}{Additional arguments to pass to main plot window (see
\code{\link{plot.default}}).}
}
\description{
Plots epidemiological data from a deterministic compartment
             epidemic model solved with \code{dcm}.
}
\details{
This function plots epidemiological outcomes from a deterministic
compartmental model solved with \code{\link{dcm}}. Depending on the number of
model runs (sensitivity analyses) and number of groups, the default plot is
the fractional proportion of each compartment in the model over time. The
specific compartments or flows to plot may be set using the \code{y}
parameter, and in multiple run models the specific run may also be specified.
}
\section{The \code{popfrac} Argument}{

Compartment prevalence are the size of a compartment over some denominator.
To plot the raw numbers from any compartment, use \code{popfrac=FALSE}; this
is the default. The \code{popfrac} parameter calculates
and plots the denominators of all specified compartments using these rules:
1) for one-group models, the prevalence of any compartment is the compartment
size divided by the total population size; 2) for two-group models, the
prevalence of any compartment is the compartment size divided by the group
size.
}

\section{Color Palettes}{

Since \code{\link{dcm}} supports multiple run sensitivity models, plotting
the results of such models uses a complex color scheme for distinguishing
runs. This is accomplished using the \code{\link{RColorBrewer}} color
palettes, which include a range of linked colors using named palettes. For
\code{plot.dcm}, one may either specify a brewer color palette listed in
\code{\link{brewer.pal.info}}, or, alternatively, a vector of standard R
colors (named, hexidecimal, or positive integers; see \code{\link{col2rgb}}).
}

\section{Plot Legends}{

There are three automatic legend types available, and the legend is
added by default for plots. To turn off the legend, use \code{legend="n"}. To
plot a legend with values for every line in a sensitivity analysis, use
\code{legend="full"}. With models with many runs, this may be visually
overwhelming. In those cases, use \code{legend="lim"} to plot a legend
limited to the highest and lowest values of the varying parameter in the
model. In cases where the default legend names are not helpful, one may
override those names with the \code{leg.name} argument.
}

\examples{
# Deterministic SIR model with varying act rate
param <- param.dcm(inf.prob = 0.2, act.rate = 1:10,
                   rec.rate = 1/3, a.rate = 0.011, ds.rate = 0.01,
                   di.rate = 0.03, dr.rate = 0.01)
init <- init.dcm(s.num = 1000, i.num = 1, r.num = 0)
control <- control.dcm(type = "SIR", nsteps = 100, dt = 0.25)
mod <- dcm(param, init, control)

# Plot disease prevalence by default
plot(mod)

# Plot prevalence of susceptibles
plot(mod, y = "s.num", popfrac = TRUE, col = "Greys")

# Plot number of susceptibles
plot(mod, y = "s.num", popfrac = FALSE, col = "Greys", grid = TRUE)

# Plot multiple runs of multiple compartments together
plot(mod, y = c("s.num", "i.num"),
     run = 5, xlim = c(0, 50), grid = TRUE)
plot(mod, y = c("s.num", "i.num"),
     run = 10, lty = 2, legend = "n", add = TRUE)

}
\seealso{
\code{\link{dcm}}, \code{\link{brewer.pal.info}}
}
\keyword{plot}
