\name{fExtDep.np}
\alias{fExtDep.np}

\title{Non-parametric extremal dependence estimation}
\description{This function estimates the bivariate extremal dependence structure using a non-parametric approach based on Bernstein Polynomials.}

\usage{
fExtDep.np(method, data, cov1=NULL, cov2=NULL, u, mar.fit=TRUE, 
           mar.prelim=TRUE, par10, par20, sig10, sig20, param0=NULL, 
           k0=NULL, pm0=NULL, prior.k="nbinom", prior.pm="unif", 
           nk=70, lik=TRUE, 
           hyperparam = list(mu.nbinom=3.2, var.nbinom=4.48), 
           nsim, warn=FALSE, type="rawdata")
}

\arguments{
	\item{method}{A character string indicating the estimation method inlcuding \code{"Bayesian"}, \code{"Frequentist"} and \code{"Empirical"}.}
	\item{data}{A matrix containing the data.}
	\item{cov1, cov2}{A covariate vector/matrix for linear model on the location parameter of the marginal distributions. \code{length(cov1)}/\code{nrow(cov1)} needs to match \code{nrow(data)}. If \code{NULL} it is assumed to be constant. Required when \code{method="Bayesian"}.}
	\item{u}{When \code{method="Bayesian"}: a bivariate indicating the threshold for the exceedance approach. If logical \code{TRUE} the threshold are calculated by default as the 90\% quantiles. When missing, a block maxima approach is considered. When \code{method="Frequentist"}: the associated quantile is used to select observations with the largest radial components. If missing, the 90\% quantile is used. }
	\item{mar.fit}{A logical value indicated whether the marginal distributions should be fitted. When \code{method="Frequentist"}, data are empirically transformed to unit Frechet scale if \code{mar.fit=TRUE}. Not required when \code{method="Empirical"}.}
	\item{rawdata}{A character string specifying if the data is \code{"rawdata"} or \code{"maxima"}. Required when \code{method="Frequentist"}.}

	\item{mar.prelim}{A logical value indicated whether a preliminary fit of marginal distributions should be done prior to estimating the margins and dependence. Required when \code{method="Bayesian"}.}
  \item{par10, par20}{Vectors of starting values for the marginal parameter estimation. Required when \code{method="Bayesian"} and \code{mar.fit=TRUE}}
  \item{sig10, sig20}{Positive reals representing the initial value for the scaling parameter of the multivariate normal proposal distribution for both margins. Required when \code{method="Bayesian"} and \code{mar.fit=TRUE}}
  \item{param0}{A vector of initial value for the Bernstein polynomial coefficients. It should be a list with elements \code{$eta} and \code{$beta} which can be generated by the internal function \code{rcoef} if missing. Required when \code{method="Bayesian"}. }
  \item{k0}{An integer indicating the initial value of the polynomial order. Required when \code{method="Bayesian"}.}
  \item{pm0}{A list of initial values for the probability masses at the boundaries of the simplex. It should be a list with two elements \code{$p0} and \code{$p1}. Randomly generated if missing. Required when \code{method="Bayesian"}.}
  \item{prior.k}{A character string indicating the prior distribution on the polynomial order. By default \code{prior.k="nbinom"} (negative binomial) but can also be \code{"pois"} (Poisson). Required when \code{method="Bayesian"}.}
  \item{prior.pm}{A character string indicating the prior on the probability masses at the endpoints of the simplex. By default \code{prior.pm="unif"} (uniform) but can also be \code{"beta"} (beta). Required when \code{method="Bayesian"}.}
  \item{nk}{An integer indicating the maximum polynomial order. Required when \code{method="Bayesian"}.}
  \item{lik}{A logical value; if \code{FALSE}, an approximation of the likelihood, by means of the angular measure in Bernstein form is used (\code{TRUE} by default). Required when \code{method="Bayesian"}.}
  \item{hyperparam}{A list of the hyper-parameters depending on the choice of \code{prior.k} and \code{prior.pm}. See \bold{details}. Required when \code{method="Bayesian"}.}
  \item{nsim}{An integer indicating the number of iterations in the Metropolis-Hastings algorithm. Required when \code{method="Bayesian"}.}	
  \item{warn}{A logical value. If \code{TRUE} warnings are printed when some specifics (e.g., \code{param0}, \code{k0}, \code{pm0} and \code{hyperparam}) are not provided and set by default. Required when \code{method="Bayesian"}.}
  \item{type}{A character string indicating whther the data are \code{"rawdata"} or \code{"maxima"}. Required when \code{method="Bayesian"}.}

}

\details{
When \code{method="Bayesian"}, the vector of hyper-parameters is provided in the argument \code{hyperparam}. It should include:
\itemize{
\item{If \code{prior.pm="unif"}}{ requires \code{hyperparam$a.unif} and \code{hyperparam$b.unif}.}
\item{If \code{prior.pm="beta"}}{ requires \code{hyperparam$a.beta} and \code{hyperparam$b.beta}.}
\item{If \code{prior.k="pois"}}{ requires \code{hyperparam$mu.pois}.}
\item{If \code{prior.k="nbinom"}}{ requires \code{hyperparam$mu.nbinom} and \code{hyperparam$var.nbinom} or \code{hyperparam$pnb} and \code{hyperparam$rnb}. The relationship is \code{pnb = mu.nbinom/var.nbinom} and \code{rnb = mu.nbinom^2 / (var.nbinom-mu.nbinom)}.}
}
When \code{u} is specified Algorithm 1 of Beranger et al. (2021) is applied whereas when it is not specified Algorithm 3.5 of Marcon et al. (2016) is considered.

When \code{method="Frequentist"}, if \code{type="rawdata"} then pseudo-polar coordinates are extracted and only observations with a radial component above some high threshold (the quantile equivalent of \code{u} for the raw data) are retained. The inferential approach proposed in Marcon et al. (2017) based on the approximate likelihood is applied.

When \code{method="Empirical"}, the empirical estimation procedure presented in Einmahl et al. (2013) is applied.
}

\value{
Outputs take the form of list including:
  \itemize{
  \item{\code{method}: }{The argument.}
  \item{\code{type}: } whether it is \code{"maxima"} or \code{"rawdata"} (in the broader sense that a threshold exceedance model was taken).
  }
If \code{method="Bayesian"} the list also includes:
  \itemize{
  \item{\code{mar.fit}: }{The argument.}
  \item{\code{pm}: }{The posterior sample of probability masses.}
  \item{\code{eta}: }{The posterior sample for the coeficients of the Bernstein polynomial.}
  \item{\code{k}: }{The posterior sample for the Bernstein polynoial order.}
  \item{\code{accepted}: }{A binary vector indicating if the proposal was accepted.}
  \item{\code{acc.vec}: }{A vector containing the acceptance probabilities for the dependence parameters at each iteration.}
  \item{\code{prior}: }{A list containing \code{hyperparam}, \code{prior.pm} and \code{prior.k}.}
  \item{\code{nsim}: }{The argument.}
  \item{\code{data}: }{The argument.}
  }
In addition if the marginal parameters are estimated (\code{mar.fit=TRUE}):
  \itemize{
  \item{\code{cov1}, \code{cov2}: }{The arguments.}
  \item{\code{accepted.mar}, \code{accepted.mar2}: }{Binary vectors indicating if the marginal proposals were accepted.}
  \item{\code{straight.reject1}, \code{straight.reject2}: }{Binary vectors indicating if the marginal proposals were rejected straight away as not respecting existence conditions (proposal is multivariate normal).}
  \item{\code{acc.vec1}, \code{acc.vec2}: }{Vectors containing the acceptance probabilities for the marginal parameters at each iteration.}
  \item{\code{sig1.vec}, \code{sig2.vec}: }{Vectors containing the values of the scaling parameter in the marginal proposal distributions.}
  }
Finally, if the argument \code{u} is provided, the list also contains:
  \itemize{
  \item{\code{threshold}: }{A bivariate vector indicating the threshold level for both margins.}
  \item{\code{kn}: }{The empirical estimate of the probability of being greater than the threshold.}
  }
  
When \code{method="Frequentist"}, the list includes:  
  
When \code{method="Empirical"}, the list includes:
  \itemize{
  \item{\code{hhat}: }{An estimate of the angular density.}
  \item{\code{Hhat}: }{An estimate of the angular measure.}
  \item{\code{p0}, \code{p1}: }{The estimates of the probability mass at 0 and 1.}
  \item{\code{Ahat}: }{An estimate of the PIckands dependence function.}
  \item{\code{w}: }{A sequence of value on the bivariate unit simplex.}
  \item{\code{q}: }{A real in \eqn{[0,1]}{[0,1]} indicating the quantile associated with the threshold \code{u}. Takes value \code{NULL} if \code{type="maxima"}.}
  \item{\code{data}: }{The data on the unit Frechet scale (empirical transformation) if \code{type="rawdata"} and \code{mar.fit=TRUE}. Data on the original scale otherwise.}
  }

}

\author{
	Simone Padoan, \email{simone.padoan@unibocconi.it},
	\url{https://faculty.unibocconi.it/simonepadoan/};
	Boris Beranger, \email{borisberanger@gmail.com}
	\url{https://www.borisberanger.com};
}

\references{
	Beranger, B., Padoan, S. A. and Sisson, S. A. (2021).
	Estimation and uncertainty quantification for extreme quantile regions.
  \emph{Extremes}, \bold{24}, 349-375.

  Einmahl, J. H. J., de Haan, L. and Krajina, A. (2013).
  Estimating extreme bivariate quantile regions.
  \emph{Extremes}, \bold{16}, 121-145.
  
  Marcon, G., Padoan, S. A. and Antoniano-Villalobos, I. (2016).
  Bayesian inference for the extremal dependence.
  \emph{Electronic Journal of Statistics}, \bold{10}, 3310-3337.

  Marcon, G., Padoan, S.A., Naveau, P., Muliere, P. and Segers, J. (2017)
  Multivariate Nonparametric Estimation of the Pickands Dependence
  Function using Bernstein Polynomials.
  \emph{Journal of Statistical Planning and Inference}, \bold{183}, 1-17.
}

\seealso{\code{\link{dExtDep}}, \code{\link{pExtDep}}, \code{\link{rExtDep}}, \code{\link{fExtDep}} }

\examples{
# Example Bayesian estimation, 
# Threshold exceedances approach, threshold set by default
# Joint estimation margins + dependence
# Default uniform prior on pm
# Default negative binomial prior on polynomial order
# Quadratic relationship between location and max temperature

\dontrun{
  data(MilanPollution)
  data <- Milan.winter[,c("NO2", "SO2", "MaxTemp")]
  data <- data[complete.cases(data),]
  
  covar <- cbind(rep(1,nrow(data)), data[,3], data[,3]^2)
  hyperparam <- list(mu.binom=6, var.binom=8, a.unif=0, b.unif=0.2)
  pollut <- fExtDep.np(method="Bayesian", data = data[,-3], u=TRUE,
                       cov1 = covar, cov2 = covar, mar.prelim=FALSE,
                       par10 = c(100,0,0,35,1), par20 = c(20,0,0,20,1),
                       sig10 = 0.1, sig20 = 0.1, k0 = 5,
                       hyperparam = hyperparam, nsim = 5e+4)
  # Warning: This is slow!                     
}

# Example Frequentist estimation
# Data are maxima

data(WindSpeedGust)
years <- format(ParcayMeslay$time, format="\%Y")
attach(ParcayMeslay[which(years \%in\% c(2004:2013)),])
  
WS_th <- quantile(WS,.9)
DP_th <- quantile(DP,.9)
  
pars.WS <- evd::fpot(WS, WS_th, model="pp")$estimate
pars.DP <- evd::fpot(DP, DP_th, model="pp")$estimate
  
data_uf <- trans2UFrechet(cbind(WS,DP), type="Empirical")
  
rdata <- rowSums(data_uf)
r0 <- quantile(rdata, probs=.90)
extdata <- data_uf[rdata>=r0,]
  
SP_mle <- fExtDep.np(method="Frequentist", data=extdata, k0=10, 
                     type="maxima")
}

\keyword{models}
