% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fkf.SP.R
\name{fkf.SP}
\alias{fkf.SP}
\title{Fast Kalman Filtering using Sequential Processing.}
\usage{
fkf.SP(a0, P0, dt, ct, Tt, Zt, HHt, GGt, yt)
}
\arguments{
\item{a0}{A \code{vector} giving the initial value/estimation of the state variable}

\item{P0}{A \code{matrix} giving the variance of a0}

\item{dt}{A \code{matrix} giving the intercept of the transition equation}

\item{ct}{A \code{matrix} giving the intercept of the measurement equation}

\item{Tt}{An \code{array} giving factor of the transition equation}

\item{Zt}{An \code{array} giving the factor of the measurement equation}

\item{HHt}{An \code{array} giving the variance of the innovations of the transition equation}

\item{GGt}{A \code{vector} giving the diagonal elements of the \code{matrix} for the variance of disturbances of the measurement equation. Covariance between disturbances
is not supported under the sequential processing method.}

\item{yt}{A \code{matrix} containing the observations. "NA"- values are allowed}
}
\value{
A \code{numeric} value corresponding to the log-likelihood calculated by the Kalman filter. Ideal for maximum likelihood estimation through optimization routines such as \code{optim}.

\bold{fkf and fkf.SP values}:

Outputs of the \code{fkf} and \code{fkf.SP} functions differ slightly in that \code{fkf} returns a list object of filtered values returned by the
algorithm, whereas \code{fkf.SP} returns only a \code{numeric} value corresponding to the log-likelihood returned by the filter. \code{fkf}
is thus appropriate when filtered values are required and \code{fkf.SP} is appropriate for efficient parameter estimation through maximum likelihood
estimation.

\bold{Log-Likelihood Values:}

When there are no missing observations (i.e. "NA" values) in argument \code{yt}, the return of function \code{fkf.SP} and the \code{logLik}
object returned within the list of function \code{fkf} are identical. When NA's are present, however, log-likelihood values returned
by \code{fkf.SP} are always higher. The log-likelihood value of the C code of \code{FKF} is instantiated through the calculation of the first term of the log-likelihood function,
\mjeqn{- 0.5 \times n \times d \times log(2\pi)}{- 0.5 * n * d * log(2 PI)}, where \mjeqn{n}{n} is the number of columns of argument
\code{yt} and \mjeqn{d}{d} is the number of rows of argument \code{yt}. Under the assumption that there are
missing observations, \mjeqn{d}{d} would instead become \mjeqn{d_t}{d_t}, where \mjeqn{d_t \leq d \forall t}{d_t <= d forall t}.
Whilst this doesn't influence parameter estimation, because observation matrix \code{yt} and thus the offset resulting from this is kept constant during maximum likelihood estimation,
this does result in low bias of the log-likelihood values output by \code{fkf}.
}
\description{
\loadmathjax
The \code{fkf.SP} function performs fast and flexible Kalman filtering using sequential processing. It is designed for efficient parameter
estimation through maximum likelihood estimation. \code{fkf.SP} wraps the C-function \code{fkf_SP} which relies upon the linear algebra subroutines of BLAS (Basic Linear Algebra Subprograms).
Sequential processing (SP) is a univariate treatment of a multivariate series of observations that increases computational efficiency over traditional Kalman filtering in the general case. SP takes
the additional assumption that the variance of disturbances in the measurement equation are independent. \code{fkf.SP} is based from the \code{fkf} function of the \code{FKF} package but is, in general, a faster Kalman filtering method.
\code{fkf} and \code{fkf.SP} share identical arguments (except for the \code{GGt} argument, see \bold{arguments}). \code{fkf.SP} is compatible with missing observations (i.e. NA's in argument \code{yt}).
}
\details{
\bold{Parameters}:

The \code{fkf.SP} function builds upon the \code{fkf} function of the \code{FKF} package by adjusting the Kalman filtering algorithm to
utilize SP. The \code{fkf.SP} and \code{fkf} functions feature highly similar
arguments for compatibility purposes; only argument \code{GGt} has changed from an \code{array} type object to a \code{vector} or \code{matrix} type object.
The \code{fkf.SP} function takes the additional assumption over the \code{fkf} function that the variance of the disturbances of the measurement
equation are independent; a requirement of SP (see below).

Parameters can either be constant or deterministic
time-varying. Assume the number of discrete time observations is \mjeqn{n}{n}
i.e. \mjeqn{y = y_t}{y = y_t} where \mjeqn{t = 1, \cdots, n}{t = 1, ..., n}. Let \mjeqn{m}{m} be the
dimension of the state variable and \mjeqn{d}{d} the dimension of the observations. Then, the parameters admit the following
classes and dimensions:

\tabular{ll}{
\code{dt} \tab either a \mjeqn{m \times n}{m * n} (time-varying) or a \mjeqn{m \times 1}{m * 1} (constant) matrix. \cr
\code{Tt} \tab either a \mjeqn{m \times m \times n}{m * m * n} or a \mjeqn{m \times m \times 1}{m * m * 1} array. \cr
\code{HHt} \tab either a \mjeqn{m \times m \times n}{m * m * n} or a \mjeqn{m \times m \times 1}{m * m * 1} array. \cr
\code{ct} \tab either a \mjeqn{d \times n}{d * n} or a \mjeqn{d \times 1}{d * 1} matrix. \cr
\code{Zt} \tab either a \mjeqn{d \times m \times n}{d * m * n} or a \mjeqn{d \times m \times 1}{d * m * 1} array. \cr
\code{GGt} \tab either a \mjeqn{d \times n}{d * n} (time-varying) or a \mjeqn{d \times 1}{d * 1} matrix. \cr
\code{yt} \tab a \mjeqn{d \times n}{d * n} matrix.
}

\bold{State Space Form}

The following notation follows that of Koopman \emph{et al.} (1999) and the documentation of the \code{fkf} function. The Kalman filter is characterized by the transition and measurement equations:

\mjdeqn{\alpha_{t + 1} = d_t + T_t \cdot \alpha_t + H_t \cdot \eta_t}{alpha(t + 1) = d(t) + T(t) alpha(t) + H(t) * eta(t)}
\mjdeqn{y_t = c_t + Z_t \cdot \alpha_t + G_t \cdot \epsilon_t}{y(t) = c(t) + Z(t) alpha(t) + G(t) * epsilon(t)}

where \mjeqn{\eta_t}{eta(t)} and \mjeqn{\epsilon_t}{epsilon(t)} are i.i.d.
\mjeqn{N(0, I_m)}{N(0, I(m))} and i.i.d. \mjeqn{N(0, I_d)}{N(0, I(d))},
respectively, and \mjeqn{\alpha_t}{alpha(t)} denotes the state
vector. The parameters admit the following dimensions:

\tabular{lll}{
\mjeqn{a_t \in R^m}{a\link{t} \in R^m} \tab \mjeqn{d_t \in R^m}{d\link{t} \in R^m} \tab \mjeqn{\eta_t \in R^m}{eta\link{t} \in R^m} \cr
\mjeqn{T_t \in R^{m \times m}}{T\link{t} \in R^(m * m )} \tab \mjeqn{H_t \in R^{m \times m}}{H\link{t} \in R^(m * m)} \tab \cr
\mjeqn{y_t \in R^d}{y\link{t} \in R^d} \tab \mjeqn{c_t \in R^d}{c\link{t} \in R^d} \tab \mjeqn{\epsilon_t \in R^d}{epsilon\link{t} \in R^d} \cr
\mjeqn{Z_t \in R^{d \times m}}{Z\link{t} \in R^(d * m)} \tab \mjeqn{G_t \in R^{d \times d}}{G\link{t}
\in R^(d * d)} \tab
}

Note that \code{fkf.SP} takes as input \code{HHt} and \code{GGt} which corresponds to \mjeqn{H_t H_t'}{H\link{t} * t(H\link{t})} and
\mjeqn{diag(G_t)^2}{diag(G\link{t})^2} respectively.

\bold{Sequential Processing Iteration}:

Traditional Kalman filtering takes the entire observational vector \mjeqn{y_t}{y\link{t}} as the items for analysis. SP
is an alternate approach that filters the elements of \mjeqn{y_t}{y\link{t}} one at a time. Sequential processing is described in the textbook of Durbin and Koopman (2001) and is described below.

Let \mjeqn{p}{p} equal the number of observations at time \mjeqn{t}{t} (i.e. when considering possible missing observations \mjeqn{p \leq {d}}{p <= d}).
The SP iteration involves treating the vector series: \mjeqn{y_1,\cdots,y_n}{y_1,...,y_n} instead as the scalar series
\mjeqn{y_{1,1},\cdots,y_{(1,p)},y_{2,1},\cdots,y_{(n,p_n)}}{y_{1,1},...,y_{(1,p)},y_{2,1},...,y_{(n,p_n )}}. This univariate treatment of the multivariate series
has the advantage that the function of the covariance matrix, \mjeqn{F_t}{F\link{t}}, becomes \mjeqn{1 \times 1}{1 * 1}, avoiding the calculation of both the inverse and determinant of
a \mjeqn{p \times p}{p * p} matrix. This can increase computational efficiency (especially under the case of many observations, i.e. \mjeqn{p}{p} is large)

For any time point, the observation vector is given by:

\mjdeqn{y_t'=(y_{(t,1)},\cdots,y_{(t,p)} )}{y_t'=(y_(t,1),...,y_(t,p) )}

The filtering equations are written as:

\mjdeqn{a_{t,i+1} = a_{t,i} + K_{t,i} v_{t,i}}{a\link{t,i+1} = a\link{t,i} + K\link{t,i} v\link{t,i}}
\mjdeqn{P_{t,i+1} = P_{t,i} - K_{t,i} F_{t,i} K_{t,i}'}{P\link{t,i+1} = P\link{t,i} - K\link{t,i} F\link{t,i} K\link{t,i}'}
Where:
\mjdeqn{\hat y_{t,i} = c_t + Z_t \cdot a_{t,i}}{^y\link{t} = c\link{t} + Z\link{t} * a\link{t,i}}
\mjdeqn{v_{t,i}=y_{t,i}-\hat y_{t,i}}{v\link{t,i}=y\link{t,i}-\hat y\link{t,i}}
\mjdeqn{F_{t,i} = Z_{t,i} P_{t,i} Z_{t,i}'+ GGt_{t,i}}{GGt\link{t,i}}
\mjdeqn{K_{t,i} = P_{t,i} Z_{t,i}' F_{t,i}^{-1}}{A}
\mjdeqn{i = 1, \cdots, p}{i = 1, ..., p}

Transition from time \mjeqn{t}{t} to \mjeqn{t+1}{t+1} occurs through the standard transition equations.

\mjdeqn{\alpha_{t + 1,1} = d_t + T_t \cdot \alpha_{t,p}}{alpha(t + 1,1) = d(t) + T(t) alpha(t,p)}

\mjdeqn{P_{t + 1,1} = T_t \cdot P_{t,p} \cdot T_t' + HHt}{P(t + 1,1) = T(t) P(t,p) T(t)' + H(t)}

The log-likelihood at time \mjeqn{t}{t} is given by:

\mjdeqn{log L_t = -\frac{p}{2}log(2\pi) - \frac{1}{2}\sum_{i=1}^p(log F_i + \frac{v_i^2}{F_i})}{log L_t = -p/2 * log(2 * pi) - 1/2 * sum_(i=1)^p (log F_i + (v_i^2)/F_i)}

Where the log-likelihood of observations is:

\mjdeqn{log L = \sum_t^n{log L_t}}{log L = \sum_t^n(log L\link{t})}
}
\examples{

# 'fkf.SP' is suitable for maximum likelihood estimation, the following examples
# showcase how to estimate parameters of different models.

## <-------------------------------------------------------------------------------
##Example 1 - ARMA(2,1) model estimation.
## <-------------------------------------------------------------------------------

#Length of series
n <- 1000

#AR parameters
AR <- c(ar1 = 0.6, ar2 = 0.2, ma1 = -0.2, sigma = sqrt(0.2))

## Sample from an ARMA(2, 1) process
a <- stats::arima.sim(model = list(ar = AR[c("ar1", "ar2")], ma = AR["ma1"]), n = n,
innov = rnorm(n) * AR["sigma"])

##State space representation of the four ARMA parameters
arma21ss <- function(ar1, ar2, ma1, sigma) {
Tt <- matrix(c(ar1, ar2, 1, 0), ncol = 2)
Zt <- matrix(c(1, 0), ncol = 2)
ct <- matrix(0)
dt <- matrix(0, nrow = 2)
GGt <- matrix(0)
H <- matrix(c(1, ma1), nrow = 2) * sigma
HHt <- H \%*\% t(H)
a0 <- c(0, 0)
P0 <- matrix(1e6, nrow = 2, ncol = 2)
return(list(a0 = a0, P0 = P0, ct = ct, dt = dt, Zt = Zt, Tt = Tt, GGt = GGt,
            HHt = HHt))
            }

## The objective function passed to 'optim'
objective <- function(theta, yt) {
sp <- arma21ss(theta["ar1"], theta["ar2"], theta["ma1"], theta["sigma"])
 ans <- fkf.SP(a0 = sp$a0, P0 = sp$P0, dt = sp$dt, ct = sp$ct, Tt = sp$Tt,
               Zt = sp$Zt, HHt = sp$HHt, GGt = sp$GGt, yt = yt)
 return(-ans)
}

## Parameter estimation - maximum likelihood estimation:
theta <- c(ar = c(0, 0), ma1 = 0, sigma = 1)
ARMA_MLE <- optim(theta, objective, yt = rbind(a), hessian = TRUE)


## <-------------------------------------------------------------------------------
#Example 2 - Nile Example:
## <-------------------------------------------------------------------------------

#Nile's annual flow:
yt <- Nile

##Incomplete Nile Data - two NA's are present:
yt[c(3, 10)] <- NA

## Set constant parameters:
dt <- ct <- matrix(0)
Zt <- Tt <- matrix(1)
a0 <- yt[1]   # Estimation of the first year flow
P0 <- matrix(100)       # Variance of 'a0'

## Parameter estimation - maximum likelihood estimation:
##Unknown parameters initial estimates:
GGt <- HHt <- var(yt, na.rm = TRUE) * .5
#Perform maximum likelihood estimation
Nile_MLE <- optim(c(HHt = HHt, GGt = GGt),
                fn = function(par, ...)
                -fkf.SP(HHt = matrix(par[1]), GGt = matrix(par[2]), ...),
                yt = rbind(yt), a0 = a0, P0 = P0, dt = dt, ct = ct,
                Zt = Zt, Tt = Tt)
## <-------------------------------------------------------------------------------
#Example 3 - Dimensionless Treering Example:
## <-------------------------------------------------------------------------------


## tree-ring widths in dimensionless units
y <- treering

## Set constant parameters:
dt <- ct <- matrix(0)
Zt <- Tt <- matrix(1)
a0 <- y[1]            # Estimation of the first width
P0 <- matrix(100)     # Variance of 'a0'

## Parameter estimation - maximum likelihood estimation:
Treering_MLE <- optim(c(HHt = var(y, na.rm = TRUE) * .5,
                 GGt = var(y, na.rm = TRUE) * .5),
                 fn = function(par, ...)
                -fkf.SP(HHt = array(par[1],c(1,1,1)), GGt = matrix(par[2]), ...),
                 yt = rbind(y), a0 = a0, P0 = P0, dt = dt, ct = ct,
                 Zt = Zt, Tt = Tt)

}
\references{
Anderson, B. D. O. and Moore. J. B. (1979). \emph{Optimal Filtering} Englewood Cliffs: Prentice-Hall.

Fahrmeir, L. and tutz, G. (1994) \emph{Multivariate Statistical Modelling Based on Generalized Linear Models.} Berlin: Springer.

Koopman, S. J., Shephard, N., Doornik, J. A. (1999). Statistical algorithms for models in state space using SsfPack 2.2. \emph{Econometrics Journal}, Royal Economic Society, vol. 2(1), pages 107-160.

Durbin, James, and Siem Jan Koopman (2001). \emph{Time series analysis by state space methods.} Oxford university press.

David Luethi, Philipp Erb and Simon Otziger (2018). FKF: Fast Kalman Filter. R package version 0.1.5.
https://CRAN.R-project.org/package=FKF
}
