% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wrAdd.R
\name{wrAdd}
\alias{wrAdd}
\alias{wrAdd.default}
\alias{wrAdd.formula}
\title{Computes a vector of relative weights specific to a species in an entire data frame.}
\usage{
wrAdd(wt, ...)

\method{wrAdd}{default}(wt, len, spec, units = c("metric", "English"),
  ...)

\method{wrAdd}{formula}(wt, data, units = c("metric", "English"), ...)
}
\arguments{
\item{wt}{A numeric vector that contains weight measurements or a formula of the form \code{wt~len+spec} where \dQuote{wt} generically represents the weight variable, \dQuote{len} generically represents the length variable, and \dQuote{spec} generically represents the species variable. Note that this formula can only contain three variables and they must be in the order of weight first, length second, species third.}

\item{\dots}{Not used.}

\item{len}{A numeric vector that contains length measurements. Not used if \code{wt} is a formula.}

\item{spec}{A character or factor vector that contains the species names. Not used if \code{wt} is a formula.}

\item{units}{A string that indicates whether the weight and length data in \code{formula} are in (\code{"metric"} (DEFAULT; mm and g) or \code{"English"} (in and lbs) units.}

\item{data}{A data.frame that minimally contains variables of the the observed lengths, observed weights, and the species names given in the \code{formula=}.}
}
\value{
Returns A numeric vector that contains the computed relative weights, in the same order as in \code{data=}.
}
\description{
This computes a vector that contains the relative weight specific to each species for all individuals in an entire data frame.
}
\details{
This computes a vector that contains the relative weight specific to each species for all individuals in an entire data frame. The vector can be appended to an existing data.frame to create a variable that contains the relative weights for each individual. The relative weight value will be \code{NA} for each individual for which a standard weight equation does not exist in \code{\link{WSlit}}, a standard weight equation for the units given in \code{units=} does not exist in \code{\link{WSlit}}, a standard weight equation for the 75th percentile does not exist in \code{\link{WSlit}}, or if the individual is shorter or longer than the lengths for which the standard weight equation should be applied. Either the linear or quadratic equation has been listed as preferred for each species, so only that equation will be used. The use of the 75th percentile is by far the most common and, because this function is designed for use on entire data frames, it will be the only percentile allowed. Thus, to use equations for other percentiles, one will have to use \dQuote{manual} methods. See \code{\link{WSlit}} and \code{\link{wsVal}} for more details about types of equations, percentiles, finding which species have published standard weight equations, etc. See the examples for one method for changing species names to something that this function will recognize.
}
\section{IFAR Chapter}{
 8-Condition.
}

\examples{
## Create random data for three species
# just to control the randomization
set.seed(345234534)
dbt <- data.frame(species=factor(rep(c("Bluefin Tuna"),30)),tl=round(rnorm(30,1900,300),0))
dbt$wt <- round(4.5e-05*dbt$tl^2.8+rnorm(30,0,6000),1)
dbg <- data.frame(species=factor(rep(c("Bluegill"),30)),tl=round(rnorm(30,130,50),0))
dbg$wt <- round(4.23e-06*dbg$tl^3.316+rnorm(30,0,10),1)
dlb <- data.frame(species=factor(rep(c("Largemouth Bass"),30)),tl=round(rnorm(30,350,60),0))
dlb$wt <- round(2.96e-06*dlb$tl^3.273+rnorm(30,0,60),1)
df <- rbind(dbt,dbg,dlb)
str(df)

df$Wr1 <- wrAdd(wt~tl+species,data=df)
## same but with non-formula interface
df$Wr2 <- wrAdd(df$wt,df$tl,df$species)

## Same as above but using dplyr
if (require(dplyr)) {
  df <- mutate(df,Wr3a=wrAdd(wt,tl,species))
}

df
 
## Example with only one species in the data.frame
bg <- Subset(df,species=="Bluegill")
bg$Wr4 <- wrAdd(wt~tl+species,data=bg)

}
\references{
Ogle, D.H. 2016. \href{http://derekogle.com/IFAR}{Introductory Fisheries Analyses with R}. Chapman & Hall/CRC, Boca Raton, FL.
}
\seealso{
See \code{\link{wsVal}}, \code{\link{WSlit}}, and \code{\link{psdAdd}} for related functionality. See \code{\link{mapvalues}} for help in changing species names to match those in \code{\link{WSlit}}.
}
\author{
Derek H. Ogle, \email{derek@derekogle.com}
}
\keyword{manip}
