% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/thin_points.R
\name{thin_points}
\alias{thin_points}
\title{Spatial Thinning of Points}
\usage{
thin_points(
  data,
  long_col = NULL,
  lat_col = NULL,
  group_col = NULL,
  method = c("brute_force", "kd_tree", "round_hash", "grid", "precision"),
  trials = 10,
  all_trials = FALSE,
  target_points = NULL,
  seed = NULL,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{data}{A data frame or tibble containing the points to thin. Must contain longitude and latitude columns.}

\item{long_col}{Name of the column with longitude coordinates (default: "decimalLongitude").}

\item{lat_col}{Name of the column with latitude coordinates (default: "decimalLatitude").}

\item{group_col}{Name of the column for grouping points (e.g., species name, year). If NULL, no grouping is applied.}

\item{method}{Thinning method to use `c("brute_force", "kd_tree", "round_hash", "grid", "precision")`.}

\item{trials}{Number of thinning iterations to perform (default: 10).}

\item{all_trials}{If TRUE, returns results of all attempts; if FALSE, returns the best attempt with the most points retained (default: FALSE).}

\item{target_points}{Optional; a numeric value specifying the exact number of points to keep. If NULL (default), maximizes the number of kept points.}

\item{seed}{Optional; an integer seed for reproducibility of results.}

\item{verbose}{If TRUE, prints progress messages (default: FALSE).}

\item{...}{Additional parameters passed to specific thinning methods (e.g., thin_dist, precision, resolution, origin, R).}
}
\value{
A tibble of thinned points, or a combined result of all attempts if `all_trials` is TRUE.
}
\description{
This function performs spatial thinning of geographic points to reduce
point density while maintaining spatial representation. Points are thinned
based on a specified distance, grid, or precision, and multiple trials can be
performed to identify the best thinned dataset.
}
\details{
The thinning methods available are:
- `brute_force`: Uses a brute force approach to thin points.
- `kd_tree`: Uses K-D trees for thinning.
- `round_hash`: Uses rounding and hashing for efficient thinning.
- `grid`: Applies a grid-based thinning method.
- `precision`: Utilizes precision-based thinning.

For more information on specific thinning methods and inputs, refer to their respective documentation:
- `brute_force_thinning()`
- `grid_thinning()`
- `kd_tree_thinning()`
- `rounding_hashing_thinning()`
- `precision_thinning()`
}
\examples{
# Generate sample data
set.seed(123)
sample_data <- data.frame(
  decimalLongitude = runif(100, -180, 180),
  decimalLatitude = runif(100, -90, 90)
)

# Perform thinning using K-D tree method
thinned_data <- thin_points(sample_data,
                             long_col = "decimalLongitude",
                             lat_col = "decimalLatitude",
                             method = "kd_tree",
                             trials = 5,
                             verbose = TRUE)

# Perform thinning with grouping
sample_data$species <- sample(c("species_A", "species_B"), 100, replace = TRUE)
thinned_grouped_data <- thin_points(sample_data,
                                     long_col = "decimalLongitude",
                                     lat_col = "decimalLatitude",
                                     group_col = "species",
                                     method = "kd_tree",
                                     trials = 10)

}
