################################################################################
# This file is released under the GNU General Public License, Version 3, GPL-3 #
# Copyright (C) 2020 Yohann Demont                                             #
#                                                                              #
# It is part of IFC package, please cite:                                      #
# -IFC: An R Package for Imaging Flow Cytometry                                #
# -YEAR: 2020                                                                  #
# -COPYRIGHT HOLDERS: Yohann Demont, Gautier Stoll, Guido Kroemer,             #
#                     Jean-Pierre Marolleau, Loïc Garçon,                      #
#                     INSERM, UPD, CHU Amiens                                  #
#                                                                              #
# DISCLAIMER:                                                                  #
# -You are using this package on your own risk!                                #
# -We do not guarantee privacy nor confidentiality.                            #
# -This program is distributed in the hope that it will be useful, but WITHOUT #
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        #
# FITNESS FOR A PARTICULAR PURPOSE. In no event shall the copyright holders or #
# contributors be liable for any direct, indirect, incidental, special,        #
# exemplary, or consequential damages (including, but not limited to,          #
# procurement of substitute goods or services; loss of use, data, or profits;  #
# or business interruption) however caused and on any theory of liability,     #
# whether in contract, strict liability, or tort (including negligence or      #
# otherwise) arising in any way out of the use of this software, even if       #
# advised of the possibility of such damage.                                   #
#                                                                              #
# You should have received a copy of the GNU General Public License            #
# along with IFC. If not, see <http://www.gnu.org/licenses/>.                  #
################################################################################

#' @title Copy Populations from One File to Another File
#' @description
#' Copies populations from a DAF file into a copy of another DAF file.
#' Only creates new file with copied population.
#' @param from path to file to copy populations from.
#' @param into path to file that will be used as a template to copy population into.
#' Caution, it is mandatory that 'into' contains 'from' starting at 'offset'.
#' @param write_to pattern used to export file.
#' Placeholders, like "\%d/\%s_fromR.\%e", will be substituted:\cr
#' -\%d: with full path directory of 'into'\cr
#' -\%p: with first parent directory of 'into'\cr
#' -\%e: with extension of 'into' (without leading .)\cr
#' -\%s: with shortname from 'into' (i.e. basename without extension).\cr
#' Exported file extension will be deduced from this pattern. Note that it has to be a .daf.
#' @param pops regular expression or vector of desired populations present in 'from'.\cr
#' If missing, the default, all populations found will be copied.\cr
#' If given but not found, a warning will be sent.
#' @param use_regex whether to use regex to pick up population into 'from'. Default is FALSE.
#' @param overwrite whether to overwrite existing file or not. Default is FALSE.
#' Note that if TRUE, it will overwrite exported file if path of 'into' and deduced from 'write_to' arguments are different.
#' Otherwise, you will get an error saying that overwritting source file is not allowed.\cr
#' Note also that an original file, i.e. generated by IDEAS(R) or INSPIRE(R), will never be overwritten.
#' Otherwise, you will get an error saying that overwritting original file is not allowed.\cr
#' @param append_name whether to append_name basename(from) to exported populations. Default is TRUE.
#' @param offset Object number of 1st object of 'from' in 'into'. Default is 0.
#' @param endianness The endian-ness ("big" or "little") of the target system for the file. Default is .Platform$endian.\cr
#' Endianness describes the bytes order of data stored within the files. This parameter may not be modified.
#' @param verbose whether to display information (use for debugging purpose). Default is FALSE.
#' @param ... Other arguments to be passed.
#' @details Populations are exported as tagged populations.
#' @return a new file is created containing exported populations.\cr
#' It invisibly returns full path of exported file.
#' @export
popsCopy = function(from, into, write_to, pops, use_regex = FALSE, overwrite = FALSE, append_name = TRUE, offset = 0, endianness = .Platform$endian, verbose = FALSE, ...) {
  dots = list(...)
  if(missing(from)) stop("'from' can't be missing")
  if(missing(into)) stop("'into' can't be missing")
  f_Ext = getFileExt(from); assert(f_Ext, len=1, alw="daf")
  i_Ext = getFileExt(into); assert(i_Ext, len=1, alw="daf")
  if(!file.exists(from)) stop(paste("can't find",from,sep=" "))
  if(!file.exists(into)) stop(paste("can't find",into,sep=" "))
  if(missing(write_to)) stop("'write_to' can't be missing")
  assert(write_to, len = 1, typ = "character")
  if(file.exists(write_to)) if(!overwrite) stop(paste0("file ",write_to," already exists"))
  use_regex = as.logical(use_regex); assert(use_regex, len = 1, alw = c(TRUE, FALSE))
  overwrite = as.logical(overwrite); assert(overwrite, len = 1, alw = c(TRUE, FALSE))
  append_name = as.logical(append_name); assert(append_name, len = 1, alw = c(TRUE, FALSE))
  offset = na.omit(as.integer(offset)); offset = offset[offset>=0]
  assert(offset, len = 1, typ = "integer")
  assert(endianness, len = 1, alw= c("big", "little"))
  verbose = as.logical(verbose); assert(verbose, len = 1, alw = c(TRUE, FALSE))
  
  daf = ExtractFromDAF(fileName = from, extract_features = TRUE, extract_images = FALSE, extract_offsets = FALSE, extract_stats = FALSE, endianness = endianness)
  obj_number = as.integer(getInfo(fileName = into)$objcount)
  if(verbose) print(names(daf$pops))
  if(missing(pops)) {
    pops = names(daf$pops)
  } else {
    if(use_regex) pops = names(daf$pops)[grepl(pops, names(daf$pops))]
    tmp = pops%in%names(daf$pops)
    if(!all(tmp)) warning(paste0("Some pops are not present in 'from' file:\n", paste0(pops[!tmp], collapse="\n")))
  }
  pops = daf$pops[pops]
  pops = lapply(pops, FUN=function(x) {
    if(append_name) x$name <- paste(x$name, basename(from), sep="_")
    x$type <- "T"
    K = class(x$obj)
    x$obj <- which(x$obj) + offset - 1
    x$obj = x$obj[x$obj %in% (0:(obj_number-1))]
    return(x)
  })
  ExportToDAF(fileName = into, write_to = write_to, pops = pops, endianness = endianness, verbose = verbose, overwrite = overwrite)
}
