% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulation.R
\name{irtree_sim}
\alias{irtree_sim}
\title{Run a simulation by generating from and fitting an ItemResponseTrees model}
\usage{
irtree_sim(
  R = 1,
  gen_model = NULL,
  fit_model = gen_model,
  N = NULL,
  sigma = NULL,
  itempar = NULL,
  link = c("logit", "probit"),
  na_okay = TRUE,
  engine = c("mirt", "mplus", "tam"),
  verbose = FALSE,
  control = NULL,
  improper_okay = FALSE,
  par_type = "difficulty",
  plan = NULL,
  plan_args = list(),
  file = NULL,
  dir = tempdir(),
  save_rdata = TRUE,
  in_memory = c("reduced", "everything", "nothing")
)
}
\arguments{
\item{R}{Number of replications. Can be either a single number indicating the
number of replications (e.g., \code{R = 100}), or can be a range (e.g., \code{R = 1:100}).}

\item{gen_model}{Object of class \code{irtree_model} describing the
data-generating model. See \link{irtree_model} for more information.}

\item{fit_model}{Object of class \code{irtree_model} describing the model that
should be fit to the data. May be a list of multiple objects of class
\code{irtree_model} if different models should be fit to the same data set. See
\link{irtree_model} for more information.}

\item{N}{Integer, the number of persons.}

\item{sigma}{Either a matrix or a function that returns a matrix. This matrix
is the variance-covariance matrix of the person parameters that is passed
to \code{\link[MASS:mvrnorm]{MASS::mvrnorm()}}. Note that the order of the person
parameters is taken from the section Processes in the model \code{object} (see
\link{irtree_model}).}

\item{itempar}{Either a list or a function that returns a list. The list has
an element \code{beta} and an element \code{alpha}. Each of these is a
matrix of item parameters. Note that the order of items (rows) is taken from the
section Items and the order of processes (columns) is taken from the
section Processes in the \code{model} (see \link{irtree_model}).}

\item{link}{Character. Link function to use.}

\item{na_okay}{Logical indicating whether variables with unobserved response
categories are permitted. If \code{FALSE}, rejection sampling
is used to ensure that all categories are observed.}

\item{engine}{String specifying whether to use mirt, Mplus, or TAM for
estimation.}

\item{verbose}{Logical indicating whether output should be printed to the
console.}

\item{control}{List. The allowed elements of this list depend on the
\code{engine}. Use \code{\link[=control_mirt]{control_mirt()}}, \code{\link[=control_mplus]{control_mplus()}}, or \code{\link[=control_tam]{control_tam()}} for
convenience. Note that the \code{fit()} function does not use \code{...}, but that
you can use the \verb{control_*()} functions to pass additional arguments.}

\item{improper_okay}{Logical indicating whether the model should also be fit
if it is not a proper IR-tree model. Set this only to \code{TRUE} if you really
know what you are doing.}

\item{par_type}{Only used if the fit engine was mirt. Item parameters (or
thresholds) can be either of type \code{easiness} (the mirt default) or
\code{difficulty} (as in Mplus and TAM).}

\item{plan}{Parameter passed as argument \code{strategy} to \code{\link[future:plan]{future::plan()}}. May
be set to, for example, \code{multiprocess} in order to run the simulations in
parallel.}

\item{plan_args}{Named list. Parameters passed \code{\link[future:plan]{future::plan()}}.}

\item{file}{String giving the file path used to save the output if
\code{save_rdata = TRUE}. Note that the file ending is automatically set to
\code{.rda}. This argument is also passed to \code{\link[=irtree_fit_mplus]{irtree_fit_mplus()}} if applicable.}

\item{dir}{Path name that is used to save the results of every run if
\code{save_rdata = TRUE}.}

\item{save_rdata}{Logical indicating whether to save the results to an RData
file.}

\item{in_memory}{Character string indicating what output should be kept in
memory (note the argument \code{save_rdata}, which is not affected by
\code{in_memory}). If \code{"reduced"}, the output of \code{\link[=fit.irtree_model]{fit()}} is
discarded and only summary information is retained. The alternative is to
keep \code{"everything"} in memory, or to keep \code{"nothing"} in memory (which
makes only sense in combination with \code{save_rdata = TRUE}).}
}
\value{
Returns a list of length \code{R}. For each replication, a list is
returned with two elements. The element \code{spec} contains various
specifications (such as the data). The element \code{fits} is a list with one
element for each \code{fit_model} that contains the output of
\code{\link[=fit.irtree_model]{fit()}} as well as the elements \code{glanced}, \code{tidied},
and \code{augmented} (see \code{\link[=glance]{glance()}}, \code{\link[=tidy]{tidy()}}, and \code{\link[=augment]{augment()}}). Thus,
\code{res$sim3$fits$m2$glanced} gives model-fit information such as AIC for the
second model in the third replication, and \code{res$sim3$spec$data} contains
the corresponding data set.

If \code{in_memory = "nothing"}, returns \code{NULL}.
}
\description{
The function \code{irtree_sim()} generates data from an \link{irtree_model} and fits
one or more models to these data. This process is repeated \code{R} times, and the
argument \code{plan} allows to run the simulation in parallel.
}
\examples{
\donttest{
# Running these examples may take a while

m1 <- "
Equations:
1 = 1-a
2 = a*(1-b)
3 = a*b

IRT:
a BY x1@1, x2@1, x3@1, x4@1, X5@1, X6@1, X7@1;
b BY x1@1, x2@1, x3@1, x4@1, X5@1, X6@1, X7@1;

Class:
Tree
"

m2 <- "
IRT:
a BY x1@1, x2@1, x3@1, x4@1, X5@1, X6@1, X7@1;

Class:
GRM
"

model1 <- irtree_model(m1)
model2 <- irtree_model(m2)

res <- irtree_sim(
    ### Data generation ###
    gen_model = model1,
    link = "logit",
    N = 500,
    sigma = function(x) diag(2),
    itempar = function(x) list(
        beta = matrix(sort(runif(model1$J*model1$P, -2, 2)),
                      model1$J, model1$P),
        alpha = matrix(1, model1$J, model1$P)),
    na_okay = FALSE,

    ### Estimation ###
    fit_model = list(model1, model2),
    engine = "mirt",
    control = control_mirt(SE = FALSE),
    par_type = "difficulty",

    ### Replications ###
    R = 2,
    save_rdata = FALSE,

    ### Optional parallelization ###
    plan = "multiprocess",
    plan_args = list(workers = future::availableCores() - 1)
)

tab1 <- matrix(NA, 0, 4, dimnames = list(NULL, c("Rep", "Model", "AIC", "BIC")))

for (ii in seq_along(res)) {
    for (jj in seq_along(res[[ii]]$fits)) {
        IC <- res[[ii]]$fits[[jj]]$glanced
        tab1 <- rbind(tab1, c(ii, jj, round(IC$AIC, -1), round(IC$BIC, -1)))
    }
}
tab1
#>      Rep Model  AIC  BIC
#> [1,]   1     1 6900 6970
#> [2,]   1     2 7000 7060
#> [3,]   2     1 6810 6880
#> [4,]   2     2 6880 6940
}
}
\seealso{
The data are generated via \code{\link[=irtree_gen_data]{irtree_gen_data()}}, and the models are
fit via \code{\link[=fit.irtree_model]{fit()}}.
}
