% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/run.functions.R
\name{mbnma.linear}
\alias{mbnma.linear}
\title{Run MBNMA model with a linear dose-response function}
\usage{
mbnma.linear(
  network,
  slope = "rel",
  method = "common",
  class.effect = list(),
  UME = FALSE,
  cor = TRUE,
  var.scale = NULL,
  parameters.to.save = NULL,
  pd = "pv",
  parallel = FALSE,
  likelihood = NULL,
  link = NULL,
  priors = NULL,
  arg.params = NULL,
  ...
)
}
\arguments{
\item{network}{An object of class \code{mbnma.network}.}

\item{slope}{Refers to the slope parameter of the linear dose-response function.
Can take either \code{"rel"}, \code{"common"}, \code{"random"}, or be assigned a numeric value (see details).}

\item{method}{Can take either \code{"common"} or \code{"random"} to indicate whether relative effects
should be modelled with between-study heterogeneity or not (see details).}

\item{class.effect}{A list of named strings that determines which dose-response
parameters to model with a class effect and what that effect should be
(\code{"common"} or \code{"random"}). Element names should match dose-response parameter names (which will therefore
depend on whether or not a wrapper function has been used for \code{mbnma.run()}).
For example: \code{list("beta.2"="fixed", "beta.3"="random")} when using
\code{mbnma.run()} or \code{list("ed50"="fixed", "hill"="random")} when using \code{mbnma.emax.hill()}.}

\item{UME}{A boolean object to indicate whether to fit an Unrelated Mean Effects model
that does not assume consistency and so can be used to test if the consistency
assumption is valid.}

\item{cor}{A boolean object that indicates whether correlation should be modelled
between relative effect dose-response parameters (\code{TRUE}) or not (\code{FALSE}). This is
automatically set to \code{FALSE} if class effects are modelled or if multiple dose-response
functions are fitted.}

\item{var.scale}{A numeric vector indicating the relative scale of variances between
correlated dose-response parameters when relative effects are modelled on more than
one dose-response parameter and \code{cor=TRUE} (see details). Each element of
the vector refers to the relative scale of each of the dose-response parameters that is
modelled using relative effects.}

\item{parameters.to.save}{A character vector containing names of parameters
to monitor in JAGS}

\item{pd}{Can take either:
\itemize{
\item \code{pv} only pV will be reported (as automatically outputted by \code{R2jags}).
\item \code{plugin} calculates pD by the plug-in
method \insertCite{spiegelhalter2002}{MBNMAdose}. It is faster, but may output negative
non-sensical values, due to skewed deviances that can arise with non-linear models.
\item \code{pd.kl} calculates pD by the Kullback-Leibler divergence \insertCite{plummer2008}{MBNMAdose}. This
will require running the model for additional iterations but
will always produce a positive result.
\item \code{popt} calculates pD using an optimism adjustment which allows for calculation
of the penalized expected deviance \insertCite{plummer2008}{MBNMAdose}
}}

\item{parallel}{A boolean value that indicates whether JAGS should be run in
parallel (\code{TRUE}) or not (\code{FALSE}). If \code{TRUE} then the number of cores to
use is automatically calculated.}

\item{likelihood}{A string indicating the likelihood to use in the model. Can take either \code{"binomial"},
\code{"normal"} or \code{"poisson"}. If left as \code{NULL} the likelihood will be inferred from the data.}

\item{link}{A string indicating the link function to use in the model. Can take any link function
defined within JAGS (e.g. \code{"logit"}, \code{"log"}, \code{"probit"}, \code{"cloglog"}) or be assigned the value \code{"identity"} for
and identity link function. If left as \code{NULL} the link function will be automatically assigned based
on the likelihood.}

\item{priors}{A named list of parameter values (without indices) and
replacement prior distribution values given as strings
\strong{using distributions as specified in JAGS syntax} (see examples).}

\item{arg.params}{Contains a list of arguments sent to \code{mbnma.run()} by dose-response
specific wrapper functions}

\item{...}{Arguments to be sent to R2jags.}
}
\value{
An object of S3 \code{class(c("mbnma", "rjags"))} containing parameter
results from the model. Can be summarized by \code{print()} and can check
traceplots using \code{R2jags::traceplot()} or various functions from the package \code{mcmcplots}.

Nodes that are automatically monitored (if present in the model) have the
following interpretation. These will have an additional suffix that relates
to the name/number of the dose-response parameter to which they correspond
(e.g. \code{d.ed50} or \code{d.1}):
\itemize{
\item \code{d} The pooled effect for each agent for a given dose-response
parameter. Will be estimated by the model if dose-response parameters (\code{beta.1},
\code{beta.2}, \code{beta.3}, \code{beta.4}) are set to \code{"rel"}.
\item \code{sd} (without a suffix) - the between-study SD (heterogeneity) for relative effects, reported if
\code{method="random"}.
\item \code{D} The class effect for each class for a given dose-response
parameter. Will be estimated by the model if specified in \code{class.effect}.
\item \code{sd.D} The within-class SD for different agents within the same class. Will
be estimated by the model if any dose-response parameter in \code{class.effect} is
set to \code{"random"}.
\item \code{beta} The absolute value of a given dose-response parameter across the whole
network (does not vary by agent/class). Will be estimated by the model if
dose-response parameters (\code{beta.1}, \code{beta.2}, \code{beta.3}, \code{beta.4}) are set to \code{"common"}
or \code{"random"}.
\item \code{sd} (with a suffix) - the between-study SD (heterogeneity) for absolute dose-response
parameters, reported if \code{beta.1}, \code{beta.2}, \code{beta.3} or \code{beta.4} are set to \code{"random"}
\item \code{totresdev} The residual deviance of the model
\item \code{deviance} The deviance of the model
}

If there are errors in the JAGS model code then the object will be a list
consisting of two elements - an error message from JAGS that can help with
debugging and \code{model.arg}, a list of arguments provided to \code{mbnma.run()}
which includes \code{jagscode}, the JAGS code for the model that can help
users identify the source of the error.
}
\description{
Fits a Bayesian model-based network meta-analysis (MBNMA) with a defined
dose-response function. Follows the methods
of \insertCite{mawdsley2016;textual}{MBNMAdose}. This function acts as a wrapper for \code{mbnma.run()} that
uses more clearly defined parameter names.
}
\section{Dose-response parameters}{

\itemize{
\item \code{"rel"} implies that relative effects should be pooled for this dose-response
parameter separately for each agent in the network.
\item \code{"common"} implies that all studies estimate the same true absolute effect
(akin to a "fixed effects" meta-analysis) across the whole network
\item \code{"random"} implies that all studies estimate a separate true absolute effect, but
that each of these true effects vary randomly around a true mean effect. This
approach allows for modelling of between-study heterogeneity.
\item \code{numeric()} Assigned a numeric value. It indicates that
this dose-response parameter should not be estimated from the data but should be
assigned the numeric value determined by the user. This can be useful for fixing
specific dose-response parameters (e.g. Hill parameters in Emax functions) to a value.
}
}

\examples{
\donttest{
# Using the triptans data
network <- mbnma.network(HF2PPITT)

# Fit a linear dose-response MBNMA with random treatment effects
linear <- mbnma.linear(network, slope="rel", method="random")

# Fit a linear dose-response MBNMA using a cloglog link function
linear <- mbnma.linear(network, slope="rel", link="cloglog")


####### Priors #######

# Obtain priors from linear dose-response MBNMA
linear <- mbnma.linear(network, slope="rel", method="random")
print(linear$model.arg$priors)

# Set new more informative prior distributions
newpriors <- list(sd = "dnorm(0,0.5) T(0,)")

linear <- mbnma.linear(network, slope="rel", method="random",
              priors=newpriors)


########## Sampler options ##########

# Change the number of MCMC iterations, the number of chains, and the thin
linear <- mbnma.linear(network, slope="rel", method="random",
              n.iter=5000, n.thin=5, n.chains=4)

# Calculate effective number of parameters via plugin method
linear <- mbnma.linear(network, slope="rel", method="random",
              pd="plugin")

# Calculate effective number of parameters via Kullback-Leibler method
linear <- mbnma.linear(network, slope="rel", method="random",
              pd="pd.kl")


####### Examine MCMC diagnostics (using mcmcplots package) #######

# Density plots
mcmcplots::denplot(linear)

# Traceplots
mcmcplots::traplot(linear)

# Caterpillar plots
mcmcplots::caterplot(linear, "d.slope")


########## Output ###########

# Print R2jags output and summary
print(linear)
summary(linear)

# Plot forest plot of results
plot(linear)
}

}
\references{
\insertAllCited
}
