#' NULL constraint handling method for MOEA/D
#'
#' Construct the preference index matrix based only on performance values.
#'
#' This function ignores the violation values when constructing the preference
#' index matrix, using only the scalarized performance values.
#'
#' @param B Matrix of neighborhoods (generated by \code{define_neighborhood(...))})
#' @param bigZ Matrix of scalarized objective values for each neighborhood and the
#' incumbent solution (generated by \code{scalarize_values})
#' @param bigV Matrix of violation values for each neighborhood and the
#' incumbent solution
#' @param ... other parameters (unused, included for compatibility with
#' generic call)
#'
#' @return `[ N x (T+1) ]` matrix of preference indices. Each row `i` contains
#' a permutation of `{1, 2, ..., (T+1)}`, where `1,...,T` correspond
#' to the solutions contained in the neighborhood of the i-th subproblem,
#' `B[i, ]`, and `T+1` corresponds to the incumbent solution for that
#' subproblem. The order of the permutation is defined by the increasing values
#' of `f(xk)`, where `f(xk)` is the aggregation function value of
#' the k-th solution being compared.
#'
#' @export
#'
#' @section References:
#' F. Campelo, L.S. Batista, C. Aranha (2020): The {MOEADr} Package: A
#' Component-Based Framework for Multiobjective Evolutionary Algorithms Based on
#' Decomposition. Journal of Statistical Software \doi{10.18637/jss.v092.i06}\cr
#'

constraint_none <- function(B, bigZ, bigV, ...)
{
  # ========== Error catching and default value definitions
  assertthat::assert_that(
    identical(dim(bigZ),dim(bigV))
    )
  # ==========

  # Get the selection matrix for all neighborhoods
  sel.indx <- t(apply(bigZ,
                      MARGIN = 2,
                      FUN = function (X) {
                        unlist(as.matrix(sort.int(X,
                                                  index.return = TRUE))[2]) }))
  # Code snipped for getting vector of sorting indexes from
  # https://joelgranados.com/2011/03/01/r-finding-the-ordering-index-vector/

  return(sel.indx)
}
