#' Function to generate a unit space for the Mahalanobis-Taguchi Adjoint (MTA)
#'   method
#'
#' \code{MTA} generates a unit space for the Mahalanobis-Taguchi Adjoint (MTA)
#'   method. In \code{\link{general_MT}}, cofactor matrix is used for A and
#'   the data are normalized based on \code{unit_space_data}.
#'
#' @param unit_space_data Matrix with n rows (samples) and p columns (variables).
#'                          Data to generate the unit space. All data should be
#'                          continuous values and should not have missing values.
#' @param includes_transformed_data If \code{TRUE}, then the transformed data
#'                                    are included in a return object.
#'
#' @return \code{MTA} returns an object of S3 \link[base]{class} "MTA". An
#'           object of class "MTA" is a list containing the following components:
#'
#'  \item{A}{p x p (q x q) matrix. Cofactor matrix of \code{unit_space_data}
#'            (the transformed data).}
#'  \item{calc_A}{\code{calc_cofactor}.}
#'  \item{transforms_data}{Function to be generated from the
#'                          \code{\link{generates_normalization_function}} based
#'                          on the \code{unit_space_data}.}
#'  \item{distance}{Vector with length n. Distances from the unit space to each
#'                   sample.}
#'  \item{n}{The number of samples.}
#'  \item{q}{The number of variables after the data transformation. q equals p.}
#'  \item{x}{If \code{includes_transformed_data} is \code{TRUE}, then the
#'            transformed data are included.}
#'
#' @references
#'   Taguchi, G. & Kanetaka, T. (2002). \emph{Engineering Technical Development
#'     in MT System - Lecture on Applied Quality.} Japanese Standards
#'     Association. (In Japanese)
#'
#'   Taguchi, G., & Jugulum, R. (2002). \emph{The Mahalanobis-Taguchi strategy:
#'     A pattern technology system.} John Wiley & Sons.
#'
#' @seealso \code{\link{calc_cofactor}}, \code{\link{general_MT}},
#'            \code{\link{generates_normalization_function}}, and
#'            \code{\link{diagnosis.MT}}
#'
#' @examples
#' # 40 data for versicolor in the iris dataset
#' iris_versicolor <- iris[61:100, -5]
#'
#' unit_space_MTA <- MTA(unit_space_data = iris_versicolor,
#'                       includes_transformed_data = TRUE)
#'
#' (unit_space_MTA$distance)
#'
#' @importFrom stats cov
#' @export
MTA <- function(unit_space_data, includes_transformed_data = FALSE) {

  object_MTA <- general_MT(unit_space_data = unit_space_data,
                           calc_A = function(x) calc_cofactor(cov(x)),
                           generates_transform_function =
                                               generates_normalization_function,
                           includes_transformed_data = includes_transformed_data)

  class(object_MTA) <- "MTA"

  return(object_MTA)

}

#' Diagnosis method for the Mahalanobis-Taguchi Adjoint (MTA) method
#'
#' \code{diagnosis.MTA} (via \code{\link{diagnosis}}) calculates the distance
#'   based on the unit space generated by \code{\link{MTA}} or
#'   \code{\link{generates_unit_space}}(..., method = "MTA") and classifies each
#'   sample into positive (\code{TRUE}) or negative (\code{FALSE}) by comparing
#'   the values with the set threshold value.
#'
#' @param unit_space Object of class "MTA" generated by \code{\link{MTA}} or
#'                     \code{\link{generates_unit_space}}(..., method = "MTA").
#' @param newdata Matrix with n rows (samples) and p columns (variables). The
#'                  data are used to calculate the desired distances from the
#'                  unit space. All data should be continuous values and should
#'                  not have missing values.
#' @param threshold Numeric specifying the threshold value to classify each
#'                    sample into positive (\code{TRUE}) or negative
#'                    (\code{FALSE}).
#' @param includes_transformed_newdata If \code{TRUE}, then the transformed data
#'                                       for \code{newdata} are included in a
#'                                       return object.
#'
#' @return \code{diagnosis.MTA} (via \code{\link{diagnosis}}) returns a list
#'           containing the following components:
#'
#'  \item{distance}{Vector with length n. Distances from the unit space to each
#'                   sample.}
#'  \item{le_threshold}{Vector with length n. Logical values indicating the
#'                       distance of each sample is less than or equal to the
#'                       threhold value (\code{TRUE}) or not (\code{FALSE}).}
#'  \item{threshold}{Numeric value to classify the sample into positive or
#'                    negative.}
#'  \item{unit_space}{Object of class "MTA" passed by \code{unit_space}.}
#'  \item{n}{The number of samples for \code{newdata}.}
#'  \item{q}{The number of variables after the data transformation. q equals p.}
#'  \item{x}{If \code{includes_transformed_newdata} is \code{TRUE}, then the
#'            transformed data for \code{newdata} are included.}
#'
#' @references
#'   Taguchi, G. & Kanetaka, T. (2002). \emph{Engineering Technical Development
#'     in MT System - Lecture on Applied Quality.} Japanese Standards
#'     Association. (In Japanese)
#'
#'   Taguchi, G., & Jugulum, R. (2002). \emph{The Mahalanobis-Taguchi strategy:
#'     A pattern technology system.} John Wiley & Sons.
#'
#' @seealso \code{\link{general_diagnosis.MT}} and \code{\link{MTA}}
#'
#' @examples
#' # 40 data for versicolor in the iris dataset
#' iris_versicolor <- iris[61:100, -5]
#'
#' unit_space_MTA <- MTA(unit_space_data = iris_versicolor,
#'                       includes_transformed_data = TRUE)
#'
#' # 10 data for each kind (setosa, versicolor, virginica) in the iris dataset
#' iris_test <- iris[c(1:10, 51:60, 101:111), -5]
#'
#' diagnosis_MTA <- diagnosis(unit_space = unit_space_MTA,
#'                            newdata = iris_test,
#'                            threshold = 0.5,
#'                            includes_transformed_newdata = TRUE)
#'
#' (diagnosis_MTA$distance)
#' (diagnosis_MTA$le_threshold)
#'
#' @export
diagnosis.MTA <- function(unit_space,
                          newdata,
                          threshold,
                          includes_transformed_newdata = FALSE) {

  if (!inherits(unit_space, "MTA")) {
    warning("calling diagnosis.MTA(<fake-MTA-object>) ...")
  }

  general_diagnosis.MT(unit_space = unit_space,
                       newdata = newdata,
                       threshold = threshold,
                       includes_transformed_newdata =
                                                   includes_transformed_newdata)

}
