#' Wrapper function to generate a unit space for a family of  
#'   Mahalanobis-Taguchi (MT) methods
#' 
#' \code{generates_unit_space} generates a unit space for a family of 
#'   Mahalanobis-Taguchi (MT) methods. The unit space of \code{\link{MT}} method, 
#'   \code{\link{MTA}} method or \code{\link{RT}} method can be generated by 
#'   passing a method name (character) into a parameter \code{method}. 
#'
#' @param unit_space_data Matrix with n rows (samples) and p columns (variables). 
#'                          Data to generate the unit space. All data should be 
#'                          continuous values and should not have missing values. 
#' @param method Character to designate a method. Currently, "MT", "MTA", and 
#'                 "RT" are available.
#' @param includes_transformed_data If \code{TRUE}, then the transformed data 
#'                                    are included in a return object.         
#' @param ... Passed to \code{\link[base]{solve}} for computing the inverse of 
#'              the correlation matrix in \code{\link{MT}} and \code{\link{RT}} 
#'              method.
#'          
#' @return A returned object depends on the selected method. See \code{\link{MT}}, 
#'           \code{\link{MTA}} or \code{\link{RT}}.
#'  
#' @seealso \code{\link{MT}}, \code{\link{MTA}}, \code{\link{RT}}, and 
#'            \code{\link[base]{solve}}
#' 
#' @examples 
#' # 40 data for versicolor in the iris dataset                            
#' iris_versicolor <- iris[61:100, -5]
#'                     
#' # 10 data for each kind (setosa, versicolor, virginica) in the iris dataset                         
#' iris_test <- iris[c(1:10, 51:60, 101:111), -5]
#' 
#' # MT method                                    
#' unit_space_MT <- generates_unit_space(unit_space_data = iris_versicolor,
#'                                       method = "MT")
#'                                       
#' diagnosis_MT <- diagnosis(unit_space = unit_space_MT, 
#'                           newdata = iris_test, 
#'                           threshold = 4)
#'                           
#' (diagnosis_MT$distance)
#' (diagnosis_MT$le_threshold)
#' 
#' # MTA method
#' unit_space_MTA <- generates_unit_space(unit_space_data = iris_versicolor,
#'                                        method = "MTA")
#'                                       
#' diagnosis_MTA <- diagnosis(unit_space = unit_space_MTA, 
#'                            newdata = iris_test, 
#'                            threshold = 0.5)
#'                           
#' (diagnosis_MTA$distance)
#' (diagnosis_MTA$le_threshold)
#' 
#' # RT method
#' unit_space_RT <- generates_unit_space(unit_space_data = iris_versicolor,
#'                                       method = "RT")
#'                                       
#' diagnosis_RT <- diagnosis(unit_space = unit_space_RT, 
#'                           newdata = iris_test, 
#'                           threshold = 0.2)
#'                            
#' (diagnosis_RT$distance)
#' (diagnosis_RT$le_threshold)
#' 
#' @export
generates_unit_space <- function(unit_space_data, 
                                method = c("MT", "MTA", "RT"), 
                                includes_transformed_data = FALSE,
                                ...) {
  
  method <- match.arg(method)
  
  if (method == "MT") {
    MT(unit_space_data, includes_transformed_data, ...) 
  } else if (method == "MTA") {
    MTA(unit_space_data, includes_transformed_data) 
  } else if (method == "RT") {
    RT(unit_space_data, includes_transformed_data, ...)
  } else {   
  }
  
}

#' Wrapper function to generate a model for a family of Taguchi (T) methods
#' 
#' \code{generates_model} generates a model for a family of Taguchi (MT) methods. 
#'   The model of \code{\link{T1}} method, \code{\link{Ta}} method or the 
#'   \code{\link{Tb}} method can be generated by passing a method name 
#'   (character) into a parameter \code{method}. 
#'
#' @param unit_space_data Used only for the T1 method. Matrix with n rows 
#'                          (samples) and (p + 1) columns (variables). The 1 ~ p 
#'                          th columns are independent variables and the (p + 1) 
#'                          th column is a dependent variable. Underlying data 
#'                          to obtain a representative point for the 
#'                          normalization of \code{signal_space_data}. All data 
#'                          should be continuous values and should not have 
#'                          missing values. 
#' @param signal_space_data Used only for the T1 method. Matrix with m rows 
#'                            (samples) and (p + 1) columns (variables). The 
#'                            1 ~ p th columns are independent variables and the 
#'                            (p + 1) th column is a dependent variable. 
#'                            Underlying data to generate a prediction 
#'                            expression. All data should be continuous values 
#'                            and should not have missing values. 
#' @param sample_data Used for the Ta and the Tb methods. Matrix with n rows 
#'                      (samples) and (p + 1) columns (variables). The 1 ~ p th 
#'                      columns are independent variables and the (p + 1) th 
#'                      column is a dependent variable. All data should be 
#'                      continuous values and should not have missing values. 
#' @param method Character to designate a method. Currently, "MT", "MTA", and 
#'                 "RT" are available.
#' @param includes_transformed_data If \code{TRUE}, then the transformed data 
#'                                    are included in a return object.         
#'          
#' @return A returned object depends on the selected method. See \code{\link{T1}}, 
#'           \code{\link{Ta}} or \code{\link{Tb}}.
#'  
#' @seealso \code{\link{T1}}, \code{\link{Ta}}, \code{\link{Tb}}
#' 
#' @examples 
#' # The value of the dependent variable of the following samples mediates  
#' # in the stackloss dataset.
#' stackloss_center <- stackloss[c(9, 10, 11, 20, 21), ] 
#' 
#' # The following samples are data other than the unit space data and the test 
#' # data.   
#' stackloss_signal <- stackloss[-c(2, 9, 10, 11, 12, 19, 20, 21), ] 
#' 
#' # The following test samples are chosen casually. 
#' stackloss_test <- stackloss[c(2, 12, 19), -4] 
#' 
#' # T1 method                                    
#' model_T1 <- generates_model(unit_space_data = stackloss_center, 
#'                             signal_space_data = stackloss_signal,
#'                             method = "T1")
#'                          
#' forecasting_T1 <- forecasting(model = model_T1, 
#'                               newdata = stackloss_test)
#'                           
#' (forecasting_T1$y_hat)
#' 
#' # Ta method
#' model_Ta <- generates_model(sample_data = 
#'                                    rbind(stackloss_center, stackloss_signal), 
#'                             method = "Ta")
#'                          
#' forecasting_Ta <- forecasting(model = model_Ta, 
#'                               newdata = stackloss_test)
#'                           
#' (forecasting_Ta$y_hat)
#' 
#' # Tb method
#' model_Tb <- generates_model(sample_data = 
#'                                    rbind(stackloss_center, stackloss_signal),
#'                             method = "Tb")
#'                          
#' forecasting_Tb <- forecasting(model = model_Tb, 
#'                               newdata = stackloss_test)
#'                           
#' (forecasting_Tb$y_hat)
#' 
#' @export
generates_model <- function(unit_space_data, 
                            signal_space_data, 
                            sample_data,
                            method = c("T1", "Ta", "Tb"), 
                            includes_transformed_data = FALSE) {
  
  method <- match.arg(method)
  
  if (method == "T1") {
    
    if (!missing(sample_data)) {
      warning("Parameter \"sample_data\" is not used in T1 method.")
    }
    
    T1(unit_space_data = unit_space_data, 
       signal_space_data = signal_space_data, 
       includes_transformed_data = includes_transformed_data) 
    
  } else if (method == "Ta") {
    
    if (!missing(unit_space_data)) {
      warning("Parameter \"unit_space_data\" is not used in Ta method.")
    }
    
    if (!missing(signal_space_data)) {
      warning("Parameter \"signal_space_data\" is not used in Ta method.")
    }
    
    Ta(sample_data = sample_data, 
       includes_transformed_data = includes_transformed_data) 
    
  } else if (method == "Tb") {
    
    if (!missing(unit_space_data)) {
      warning("Parameter \"unit_space_data\" is not used in Tb method.")
    }
    
    if (!missing(signal_space_data)) {
      warning("Parameter \"signal_space_data\" is not used in Tb method.")
    }
    
    Tb(sample_data = sample_data, 
       includes_transformed_data = includes_transformed_data)
    
  } else {   
  }
  
}
