% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/manifold_optim.R
\name{manifold.optim}
\alias{manifold.optim}
\alias{moptim}
\title{Manifold optimization}
\usage{
manifold.optim(prob, mani.defn, method = "LRBFGS",
  mani.params = get.manifold.params(), solver.params = get.solver.params(),
  x0 = NULL, H0 = NULL, has.hhr = FALSE)

moptim(prob, mani.defn, method = "LRBFGS",
  mani.params = get.manifold.params(), solver.params = get.solver.params(),
  x0 = NULL, H0 = NULL, has.hhr = FALSE)
}
\arguments{
\item{prob}{\link{Problem definition}}

\item{mani.defn}{Either a \link{Product manifold definition} or one of the
\link{Manifold definitions}}

\item{method}{Name of optimization method. Currently supported methods are:
\itemize{
\item{\code{"LRBFGS"}}: Limited-memory RBFGS
\item{\code{"LRTRSR1"}}: Limited-memory RTRSR1
\item{\code{"RBFGS"}}: Riemannian BFGS
\item{\code{"RBroydenFamily"}}: Riemannian Broyden family
\item{\code{"RCG"}}: Riemannian conjugate gradients
\item{\code{"RNewton"}}: Riemannian line-search Newton
\item{\code{"RSD"}}: Riemannian steepest descent
\item{\code{"RTRNewton"}}: Riemannian trust-region Newton
\item{\code{"RTRSD"}}: Riemannian trust-region steepest descent
\item{\code{"RTRSR1"}}: Riemannian trust-region symmetric rank-one update
\item{\code{"RWRBFGS"}}: Riemannian BFGS
}
See Huang et al (2016a, 2016b) for details.}

\item{mani.params}{Arguments to configure the manifold. Construct with
\link{get.manifold.params}}

\item{solver.params}{Arguments to configure the solver. Construct with
\link{get.solver.params}}

\item{x0}{Starting point for optimization. A numeric vector whose dimension
matches the total dimension of the overall problem}

\item{H0}{Initial value of Hessian. A \eqn{d \times d} matrix, where \eqn{d}
is the dimension of \code{x0}}

\item{has.hhr}{Indicates whether to apply the idea in Huang et al
(2015) section 4.1 (TRUE or FALSE)}
}
\value{
\item{xopt}{Point returned by the solver}
\item{fval}{Value of the function evaluated at \code{xopt}}
\item{normgf}{Norm of the final gradient}
\item{normgfgf0}{Norm of the gradient at final iterate divided by norm of
  the gradient at initiate iterate}
\item{iter}{Number of iterations taken by the solver}
\item{num.obj.eval}{Number of function evaluations}
\item{num.grad.eval}{Number of gradient evaluations}
\item{nR}{Number of retraction evaluations}
\item{nV}{Number of actions of vector transport}
\item{nVp}{Number of actions of vector transport}
\item{nH}{Number of actions of Hessian}
\item{elapsed}{Elapsed time for the solver (in seconds)}
}
\description{
Optimize a function on a manifold.
}
\details{
\code{moptim} is an alias for  \code{manifold.optim}.
}
\examples{
\dontrun{
# ----- Example with objective and gradient written in R -----
set.seed(1234)

p <- 5; n <- 150
B <- matrix(rnorm(n*n), nrow=n)
B <- B + t(B)
D <- diag(p:1, p)

tx <- function(x) { matrix(x, n, p) }
f <- function(x) { X <- tx(x); Trace( t(X) \%*\% B \%*\% X \%*\% D ) }
g <- function(x) { X <- tx(x); 2 * B \%*\% X \%*\% D }

mod <- Module("ManifoldOptim_module", PACKAGE = "ManifoldOptim")
prob <- new(mod$RProblem, f, g)

x0 <- as.numeric(orthonorm(matrix(rnorm(n*p), nrow=n, ncol=p)))
mani.params <- get.manifold.params(IsCheckParams = TRUE)
solver.params <- get.solver.params(isconvex = TRUE, IsCheckParams = TRUE)
mani.defn <- get.stiefel.defn(n, p)

res <- manifold.optim(prob, mani.defn, method = "RTRSR1",
	mani.params = mani.params, solver.params = solver.params, x0 = x0)
print(res)
head(tx(res$xopt))
}

\dontrun{
library(ManifoldOptim)
library(RcppArmadillo)

# ----- Example with objective and gradient written in C++ -----
set.seed(1234)

p <- 5; n <- 150
B <- matrix(rnorm(n*n), nrow=n)
B <- B + t(B) # force symmetric
D <- diag(p:1, p)

# The Problem class is written in C++. Get a handle to it and set it up from R
Rcpp::sourceCpp(code = '
//[[Rcpp::depends(RcppArmadillo,ManifoldOptim)]]
#include <RcppArmadillo.h>
#include <ManifoldOptim.h>

using namespace Rcpp;
using namespace arma;

class BrockettProblem : public MatrixManifoldOptimProblem
{
public:
	BrockettProblem(const arma::mat& B, const arma::mat& D)
	: MatrixManifoldOptimProblem(false, true), _B(B), _D(D) { }

	virtual ~BrockettProblem() { }

	double objFun(const arma::mat& X) const {
		return arma::trace(X.t() * _B * X * _D);
	}

	arma::mat gradFun(const arma::mat& X) const {
		return 2 * _B * X * _D;
	}

	const arma::mat& GetB() const { return _B; }
	const arma::mat& GetD() const { return _D; }

private:
	arma::mat _B;
	arma::mat _D;
};

RCPP_MODULE(Brockett_module) {
	class_<BrockettProblem>("BrockettProblem")
	.constructor<mat,mat>()
	.method("objFun", &BrockettProblem::objFun)
	.method("gradFun", &BrockettProblem::gradFun)
	.method("GetB", &BrockettProblem::GetB)
	.method("GetD", &BrockettProblem::GetD)
	;
}
')

prob <- new(BrockettProblem, B, D)
X0 <- orthonorm(matrix(rnorm(n*p), nrow=n, ncol=p))
x0 <- as.numeric(X0)
tx <- function(x) { matrix(x, n, p) }
mani.params <- get.manifold.params(IsCheckParams = TRUE)
solver.params <- get.solver.params(isconvex = TRUE, DEBUG = 0, Tolerance = 1e-4,
	Max_Iteration = 1000, IsCheckParams = TRUE, IsCheckGradHess = FALSE)
mani.defn <- get.stiefel.defn(n, p)

res <- manifold.optim(prob, mani.defn, method = "RTRSR1",
	mani.params = mani.params, solver.params = solver.params, x0 = x0)
print(res)
head(tx(res$xopt))
}

}
\references{
Wen Huang, P.A. Absil, K.A. Gallivan, Paul Hand (2016a). "ROPTLIB: an
object-oriented C++ library for optimization on Riemannian manifolds."
Technical Report FSU16-14, Florida State University.

Wen Huang, Kyle A. Gallivan, and P.A. Absil (2016b).
Riemannian Manifold Optimization Library.
URL \url{http://www.math.fsu.edu/~whuang2/pdf/USER_MANUAL_for_2016-04-29.pdf}

Wen Huang, K.A. Gallivan, and P.A. Absil (2015). A Broyden Class of
Quasi-Newton Methods for Riemannian Optimization. SIAM  Journal on
Optimization, 25(3):1660-1685.
}

