\name{model.interaction.plot}
\alias{model.interaction.plot}

\title{
plot of two-way model interactions
}
\description{
Image or Persective plot of two-way model interactions. Ranges of two specified predictor variables are plotted on X and Y axis, and fitted model values are plotted on the Z axis. The remaining predictor variable are fixed at their mean (for continuous predictors) or their most common value (for catagorical predictors).
}
\usage{
model.interaction.plot(model.obj = NULL, x = NULL, y = NULL, qdata.trainfn = NULL, folder = NULL, MODELfn = NULL,  pred.means = NULL, xlab = NULL, ylab = NULL, x.range = NULL, y.range = NULL, z.range = NULL, ticktype = "detailed", theta = 55, phi = 40, smooth = "none", plot.type = NULL, device.type = NULL, jpeg.res = 72, device.width = 7,  device.height = 7, cex=par()$cex, col = NULL, ...)
}

\arguments{
  \item{model.obj}{ \code{R} model object.  A RF or SGB model object produced by \code{model.build}.}

  \item{x}{String or Integer. Name of predictor variable to be plotted on the x axis. Alternativly, can be a number indicating a variable name from \code{predList}.}

  \item{y}{String or Integer. Name of predictor variable to be plotted on the y axis. Alternativly, can be a number indicating a variable name from \code{predList}.}

  \item{qdata.trainfn}{String.  The name (full path or base name with path specified by \code{folder}) of the training data file used for building the model (file should include columns for both response and predictor variables).  The file must be a comma-delimited file \code{*.csv} with column headings. \code{qdata.trainfn} can also be an \code{R} dataframe. If predictions will be made (\code{predict = TRUE} or \code{map=TRUE}) the predictor column headers must match the names of the raster layer files, or a \code{rastLUT} must be provided to match predictor columns to the appropriate raster and band.  If \code{qdata.trainfn = NULL} (the default), a GUI interface prompts user to browse to the training data file.  }

  \item{folder}{ String.  The folder used for all output.  Do not add ending slash to path string.  If \code{folder = NULL} (default), a GUI interface prompts user to browse to a folder.  To use the working directory, specify \code{folder = getwd()}.}

  \item{MODELfn}{ String.  The file name to use to save the generated model object.  If \code{MODELfn = NULL} (the default), a default name is generated by pasting \code{model.type_response.type_response.name}. If the other output filenames are left unspecified, \code{MODELfn} will be used as the basic name to generate other output filenames. The filename can be the full path, or it can be the simple basename, in which case the output will be to the folder specified by \code{folder}.}

  \item{pred.means}{Vector. Allows specification of values for other predictor variables. If Null, other predictors are set to their mean value (for continuous predictors) or their most common value (for factored predictors).}

  \item{xlab}{String. Allows manual specification of the x label.}

  \item{ylab}{String. Allows manual specification of the y label.}

  \item{x.range}{Vector. Manual range specification for the x axis.}

  \item{y.range}{Vector. Manual range specification for the y axis.}

  \item{z.range}{Vector. Manual range specification for the z axis.}

  \item{ticktype}{Character: "simple" draws just an arrow parallel to the axis to indicate direction of increase; "detailed" (default) draws normal ticks as per 2D plots. If \code{X} or \code{y} is factored, ticks will be drawn on both axes.}

  \item{theta}{Numeric. Angles defining the viewing direction. \code{theta} gives the azimuthal direction.}

  \item{phi}{Numeric. Angles defining the viewing direction. \code{phi} gives the colatitude.}

  \item{smooth}{String. controls smoothing of the predicted surface. Options are \code{"none"} (default), \code{"model"} which uses a glm model to smoth the surface, and \code{"average"} which applies a 3x3 smoothing average. Note: smoothing is not appropriate if \code{X} or \code{y} is factored.}

  \item{plot.type}{Character. \code{"persp"} gives a 3-D perspective plot. \code{"image"} gives an image plot.}
 
 \item{device.type}{ String or vector of strings.  Model validation.  One or more device types for graphical output from model validation diagnostics. 

Current choices:

\tabular{lllll}{
	  \tab \tab \tab \code{"default"} \tab default graphics device\cr
	  \tab \tab \tab \code{"jpeg"} \tab \code{*.jpg} files\cr
	  \tab \tab \tab \code{"none"} \tab no graphics produced\cr	
	  \tab \tab \tab \code{"pdf"} \tab \code{*.pdf} files\cr
	  \tab \tab \tab \code{"postscript"} \tab \code{*.ps} files\cr
	  \tab \tab \tab \code{"win.metafile"} \tab \code{*.emf} files }
 }

  \item{jpeg.res}{ Integer.  Pixels per inch for jpeg plots.  The default is 72dpi, good for on screen viewing. For printing, suggested setting is 300dpi. }

  \item{device.width}{ Integer.  The device width for diagnostic plots in inches. }

  \item{device.height}{ Integer.  The device height for diagnostic plots in inches. }

  \item{cex}{ Integer.  The cex for diagnostic plots. }

  \item{col}{ Vector.  Color table to use for image plots ( see help file on image for details). }
 
  \item{\dots}{additional graphical parameters (see \code{\link{par}}).}
}
\details{
This function provides a diagnostic plot useful in visualizing two-way interactions between predictor variables. Two of the predictor variables from the model are used to produce a grid of possible combinations of predictor values over the range of both variables. The remaining predictor variables from the model are fixed at either their means (for continuous predictors) or their most common value (for catagorical predictors). Model predictions are generated over this grid and plotted as the z axis.

This function works with both continuous and categorical predictors, though the perspective plot should be interpreted with care for categorical predictors. In particualar, the \code{smooth} option is not appropriate if either of the two selected predictor variables is categorical. 
}
\value{
}
\references{
This function is adapted from \code{gbm.perspec} version 2.9 April 2007, J Leathwick/J Elith. See appendix S3 from:


Elith, J., Leathwick, J. R. and Hastie, T. (2008). A working guide to boosted regression trees. Journal of Animal Ecology. 77:802-813.
}
\author{ Elizabeth Freeman }
\note{

}


\seealso{

}
\examples{
###########################################################################
############################# Run this set up code: #######################
###########################################################################

# set seed:
seed=38

# Define training and test files:

qdata.trainfn = system.file("external", "helpexamples","DATATRAIN.csv", package = "ModelMap")
qdata.testfn = system.file("external", "helpexamples","DATATEST.csv", package = "ModelMap")

# Define folder for all output:
folder=getwd()	

########## Continuous Response, Categorical Predictors ############

# In this example, NLCD is a categorical predictor.
#
# You must decide what you want to happen if there are categories
# present in the data to be predicted (either the validation/test set
# or in the image file) that were not present in the original training data.
# Choices:
#       na.action = "na.omit"
#                    Any validation datapoint or image pixel with a value for any
#                    categorical predictor not found in the training data will be
#                    returned as NA.
#       na.action = "na.roughfix"
#                    Any validation datapoint or image pixel with a value for any
#                    categorical predictor not found in the training data will have
#                    the most common category for that predictor substituted,
#                    and the a prediction will be made.

# You must also let R know which of the predictors are categorical, in other
# words, which ones R needs to treat as factors.
# This vector must be a subset of the predictors given in predList

#file name to store model:
MODELfn="RF_BIO_TCandNLCD"			

#predictors:
predList=c("TCB","TCG","TCW","NLCD")

#define which predictors are categorical:
predFactor=c("NLCD")

# Response name and type:
response.name="BIO"
response.type="continuous"


###########################################################################
########################### build model: ##################################
###########################################################################


### create model ###

model.obj = model.build( model.type="RF",
                       qdata.trainfn=qdata.trainfn,
                       folder=folder,		
                       MODELfn=MODELfn,
                       predList=predList,
                       predFactor=predFactor,
                       response.name=response.name,
                       response.type=response.type,
                       seed=seed
)

###########################################################################
###################### make interaction plots: ############################
###########################################################################

#########################
### Perspective Plots ###
#########################


### specify first and third  predictors in 'predList (both continuous) ###

model.interaction.plot(	model.obj,
			x=1,y=3, 
			main=response.name, 
			plot.type="persp", 
			device.type="default") 

### specify first and forth predictors in 'predList (one continuous one factored) ###

model.interaction.plot(	model.obj,
			x=1, y=4, 
			main=response.name, 
			plot.type="persp", 
			device.type="default") 

### same as previous example, but specifying predictors by name ##

model.interaction.plot(	model.obj,
			x="TCB", y="NLCD",  
			main=response.name, 
			plot.type="persp", 
			device.type="default") 


###################
### Image Plots ###
###################

### same as previous example, but image plot ###


l <- seq(100,0,length.out=101)
c <- seq(0,100,length.out=101)
col.ramp <- hcl(h = 120, c = c, l = l)
 

model.interaction.plot(		model.obj,
				x="TCB", y="NLCD",
				main=response.name,
				plot.type="image", 
				device.type="default",
				col = col.ramp) 



#########################
### 3-way Interaction ###
#########################

### use 'pred.means' argument to fix values of additional predictors ###

### factored 3rd predictor ###

nlcd<-levels(model.obj$predictor.data$NLCD)

for(i in nlcd){
	pred.means=list(NLCD=i)

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("NLCD =",i),
				pred.means=pred.means, 
				z.range=c(0,110),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}



### continuos 3rd predictor ###


tcb<-seq(	min(model.obj$predictor.data$TCB),
		max(model.obj$predictor.data$TCB),
		length=5)

tcb<-signif(tcb,2)

for(i in tcb){
	pred.means=list(TCB=i)

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("TCB =",i),
				pred.means=pred.means, 
				z.range=c(0,120),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}



### 4-way Interesting combos ###


tcb=c(1300,2900,3400)
nlcd=c(11,90,95)

for(i in 1:3){
	pred.means=list(TCB=tcb[i],NLCD=nlcd[i])

	model.interaction.plot(	model.obj,
				x="TCG", y="TCW",  
				main=paste("TCB =",tcb[i],"        NLCD =",nlcd[i]),
				pred.means=pred.means, 
				z.range=c(0,120),
				theta=290,
				plot.type="persp", 
				device.type="default") 
}




}


\keyword{ models }

