#' @include mortalityTable.R
NULL

#' Class mortalityTable.mixed - Life table as a mix of two life tables
#'
#' A cohort life table obtained by mixing two life tables with the given weights.
#' Typically, when only gender-specific tables are available, unisex tables
#' are generated by mixing the two gender-specific tables for males and for
#' females with a pre-defined, constant proportion (e.g. 60:30 or 40:60,
#' depending on the portfolio and on the security margins).
#'
#' @slot table1 The first \code{mortalityTable}
#' @slot table2 The second \code{mortalityTable}
#' @slot weight1 The weight of the first mortality table
#' @slot weight2 The weight of the second mortality table
#' @slot loading Additional security loading
#'
#' @examples
#' mortalityTables.load("Austria_Annuities_AVOe2005R")
#' # Generate a unisex table with mixing relation 60:40 from male + female tables
#' AVOe2005R.myUnisex = mortalityTable.mixed(
#'     table1 = AVOe2005R.male, table2 = AVOe2005R.female,
#'     weight1 = 0.6, weight2 = 0.4,
#'     name = "My custom AVOe 2005R unisex (60:40)")
#' plot(AVOe2005R.myUnisex, AVOe2005R.male, AVOe2005R.female, Period = 2017)
#' @export mortalityTable.mixed
#' @exportClass mortalityTable.mixed
mortalityTable.mixed = setClass(
    "mortalityTable.mixed",
    slots = c(
        table1 = "mortalityTable",
        table2 = "mortalityTable",
        weight1 = "numeric",
        weight2 = "numeric",
        loading = "numeric"
    ),
    prototype = list(
        weight1 = 1/2,
        weight2 = 1/2,
        loading = 0
    ),
    contains = "mortalityTable"
)
