% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tau.test.R
\name{tau.test}
\alias{tau.test}
\title{Random shift test of independence between a point process and a covariate}
\usage{
tau.test(
  X,
  covariate.interest,
  covariates.nuisance,
  N.shifts = 999,
  radius,
  correction,
  bws,
  n.test.points = 1000,
  nonparametric = TRUE,
  verbose = FALSE,
  bw.factor.rhonhat = 1
)
}
\arguments{
\item{X}{point pattern dataset (object of class \code{ppp})}

\item{covariate.interest}{random field (object of class \code{im})}

\item{covariates.nuisance}{list of covariates (objects of class \code{im}) determining the nuisance covariates}

\item{N.shifts}{integer, how many random shifts should be performed in the random shift test}

\item{radius}{positive real number determining the radius of the disk on which the shift vectors are uniformly distributed}

\item{correction}{which correction should be applied in the random shift test (possible choices are "torus" and "variance")}

\item{bws}{vector of positive real values from which the bandwidth is adaptively chosen if at least one nuisance covariate is present; if no nuisance covariates are present, the first value is used}

\item{n.test.points}{the number of independent test points used in computing the test statistic value}

\item{nonparametric}{logical value indicating whether nonparametric residuals should be used when computing the test statistic}

\item{verbose}{logical value indicating whether auxiliary information should be printed and auxiliary figures plotted during the computation}

\item{bw.factor.rhonhat}{multiplicative factor used when determining the bandwidth in the nonparametric estimation of the intensity function depending on the nuisance covariates (defaults to 1)}
}
\value{
The p-value of the random shift test of independence between a point process and a covariate, taking into account possible effects of nuisance covariates.
}
\description{
Nonparametric test of independence between a point process and a random field
(covariate of interest), taking into account the possible effect of nuisance covariates,
see Dvořák and Mrkvička (2022).
The test is based on random shifts. Either the torus correction or the variance
correction can be used, see Mrkvička et al. (2021).
This test has lower power than the test based on the covariate-weighted residuals
(see the function \code{CWR.test}), but it is still included for the sake of completeness.
Also, the test statistic of \code{tau.test} can be used to quantify
the partial correlation between the point process and the covariate of interest, taking into
account the possible effect of nuisance covariates, see also the function \code{tau.est}.
}
\details{
The test statistic is the Kendall's correlation coefficient between the covariate of interest
and the smoothed residual field, sampled at a given number of test points
scattered independently in the observation window, see the paper Dvořák and Mrkvička (2022).
If no nuisance covariates are given, the null model
assumes a constant intensity function of the point process. If one or more nuisance covariates are
provided, the null model assumes an intensity function depending on the nuisance
covariates (but not on the covariate of interest) and the residuals are constructed
using this intensity function.

For constructing the smoothed residual field an adaptive bandwidth selection
method can be used, see Dvořák and Mrkvička (2022). A vector of candidate
bandwidth values can be provided using the argument \code{bws}.

The residuals can be constructed in a nonparametric way (see Baddeley et al. (2012))
or in a parametric way (using the \code{ppm} function from the \code{spatstat} package,
see Baddeley et al. (2015)). This choice is given by the argument \code{nonparametric}.
The raw residuals are considered here.

The torus correction can be applied for rectangular windows. On the other hand,
the variance correction is applicable both for rectangular and for irregular windows.
The choice of the correction is given by the argument \code{correction}.
Based on the simulation studies in Dvořák and Mrkvička (2022),
the variance correction is recommended since it does not exhibit the liberality of the torus correction.

The observed point pattern should be supplied using the argument \code{X}.
The realization of the covariate of interest should be supplied using
the argument \code{covariate.interest}. The set of nuisance covariates should
be supplied as a list using the argument \code{covariates.nuisance}. This list
can be empty if no nuisance covariates are considered.

The shift vectors are generated from the
uniform distribution on the disk with radius given by the argument \code{radius}
and centered in the origin. The argument \code{verbose} determines if
auxiliary information and plots should be provided.

In case the observation window accompanying the point pattern is irregular,
it must be specified in the form of a binary mask due to the specific implementation of the test.
For details on binary masks see the help for the \code{spatstat} function \code{owin}.
}
\examples{

library(spatstat)
library(ks)

# the point pattern
X <- bei
plot(X)

# two covariates are available
elevation <- bei.extra$elev
slope <- bei.extra$grad
plot(elevation)
plot(slope)

# candidate values for adaptive bandwidth selection
bws <- seq(from=12.5, to=100, by=12.5)

# test with no nuisance covariates, with only 99 shifts to speed up the computation
out1 <- tau.test(X, covariate.interest=elevation, covariates.nuisance=NULL,
                 bws=bws, N.shifts = 99, verbose=TRUE, correction="torus", radius=250)
out1

# test with one nuisance covariate, with only 99 shifts to speed up the computation
out2 <- tau.test(X, covariate.interest=elevation, covariates.nuisance=list(slope=slope),
                 bws=bws, N.shifts = 99, verbose=TRUE, correction="torus", radius=250)
out2

}
\references{
J. Dvořák, T. Mrkvička (2022): Nonparametric testing of the covariate significance for spatial point patterns under the presence of nuisance covariates. https://arxiv.org/abs/2210.05424

T. Mrkvička, J. Dvořák, J.A. González, J. Mateu (2021): Revisiting the random shift approach for testing in spatial statistics. Spatial Statistics 42, 100430.

A. Baddeley, E. Rubak, R. Turner (2015) Spatial Point Patterns: Methodology and Applications with R. Chapman & Hall Interdisciplinary Statistics Series. CRC Press, Boca Raton, Florida.

A. Baddeley, Y.-M. Chang, Y. Song, R. Turner (2012) Nonparametric estimation of the dependence of a point process on spatial covariates. Statistics and Its Interface 5(2), 221?236.
}
